# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import Tac, os, socket, Tracing
from eunuchs.if_ether_h import ETH_P_IP, ETH_P_8021Q
from ArnetTestLib import devRecvExpPkt
from Arnet.Device import Tap
from Arnet.IpTestLib import convertIpAddr, IPV4
from RecvMsgUtils import newSeparateVlanTagByField, vlanTagOffset
from struct import pack

timeout = 10 if "ABUILD" in os.environ else 3

t0 = Tracing.trace0
timerWheel = Tac.newInstance( "Ark::TimerWheel",
                              Tac.activityManager.clock, 100,
                              10 * 60 * 5, True, 1024 )

def setupTapInterfaces( devNum=3 ):
   devList = []
   for i in range( devNum ):
      devList.append( Tap( 'Test%d' % i, blocking=False ) )
   return devList

def configureRawTxPam():
   pam = Tac.newInstance( "Arnet::RawTxPam", "MyRawTxPam" )
   pam.ethProtocol = ETH_P_IP
   pam.enabled = True
   return pam

def configureReqResponseRawTxPam():
   pam = Tac.newInstance( "Arnet::ReqResponseRawTxPam", "MyRawTxPam" )
   pam.ethProtocol = ETH_P_IP
   pam.enabled = True
   return pam


def disablePam( pam ):
   pam.enabled = False

def verifyRcvBufferSetup( pam ):
   # verify that receive buffer is set correctly
   defaultSock = socket.socket( socket.AF_PACKET, socket.SOCK_RAW )
   defaultRxBufSize = defaultSock.getsockopt( socket.SOL_SOCKET, socket.SO_RCVBUF )
   sock = socket.fromfd( pam.fileDesc.descriptor, 
                         socket.AF_PACKET, socket.SOCK_RAW )
   rxBufSize = sock.getsockopt( socket.SOL_SOCKET, socket.SO_RCVBUF )
   assert rxBufSize < defaultRxBufSize

def newRawPacket( data=None, vlanId=None ):
   offset = 0
   pktSize = 64
   pkt = Tac.newInstance( "Arnet::Pkt" )
   pkt.newSharedHeadData = pktSize
   pkt.stringValue = data if data else "0" * 64
   ethHdr = Tac.newInstance( "Arnet::EthHdrWrapper", pkt, offset )
   ethHdr.src = '00:00:00:00:00:00'
   ethHdr.dst = 'ff:ff:ff:ff:ff:ff'
   ethHdr.ethType = 'ethTypeIp'
   if vlanId is not None:
      newSeparateVlanTagByField( pkt, 0, 0, vlanId, ETH_P_8021Q )
   return pkt


def sendPkt( pam, dev, vlanId=None ):
   if vlanId is not None:
      pkt = newRawPacket( vlanId=vlanId )
      expectedPkt = pkt.stringValue[ : vlanTagOffset ] + \
                    pack( '>HH', pkt.vlanTpid, vlanId ) + \
                    pkt.stringValue[ vlanTagOffset : ]
   else:
      pkt = newRawPacket()
      expectedPkt = pkt.stringValue
   pam.sendPkt( pkt, dev.ifIndex() )
   Tac.waitFor( lambda: devRecvExpPkt( dev, expectedPkt ),
                timeout=timeout )

def getTcpConnection( ipVersion=IPV4, edgeTriggered=False ):
   '''Setup a TCP Connection and get a pair of connected sockets'''
   listenPort = 12345
   addr = convertIpAddr( '127.0.0.1', ipVersion=ipVersion )
   serverIpAddr = Tac.Value( 'Arnet::IpGenAddr', addr )
   clientIpAddr = Tac.Value( 'Arnet::IpGenAddr', addr )
   fdFactory = Tac.newInstance(
      'Arnet::TcpFileDescriptorEventRegistrationFactoryTacc' )
   # Start Server
   for _i in range( 0, 10 ):
      t0( '%s Listening on %d' % ( serverIpAddr, listenPort ) )
      listenPort += 1
      serverConfig = Tac.newInstance( 'Arnet::TcpServerConfig',
                                      serverIpAddr, listenPort )
      serverConfig.edgeTriggeredFileDescriptor = edgeTriggered
      serverStatus = Tac.newInstance( 'Arnet::TcpServerStatus' )
      server = Tac.newInstance( 'Arnet::TcpServerSm', fdFactory, timerWheel,
                                serverConfig, serverStatus )
      ret = server.startListener()
      if ret:
         break
   # Start Client
   clientConfig = Tac.newInstance( 'Arnet::TcpClientConfig',
                                   serverIpAddr, listenPort )
   clientConfig.clientIpAddr = clientIpAddr
   clientConfig.edgeTriggeredFileDescriptor = edgeTriggered
   clientStatus = Tac.newInstance( 'Arnet::TcpClientStatus' )
   client = Tac.newInstance( 'Arnet::TcpClientSm', fdFactory, timerWheel,
                             1, clientConfig, clientStatus )
   ret = client.startConnection()
   Tac.runActivities( 0.1 )  # callers are counting on this despite not needed here
   Tac.waitFor( lambda c: c.tcpClientStatus.connectionStatus == 'connected',
                args=( client, ) )
   serverConnSocket = server.tcpServerStatus.getConnSocket()
   clientConnSocket = client.tcpClientStatus.connSocket
   client.tcpClientStatus.resetConnSocket()
   assert serverConnSocket.fileDescriptor
   assert clientConnSocket.fileDescriptor
   server.stopListener()
   client.stopConnection()
   t0( 'Server %s:%s connected to Client %s:%s' %
       ( serverIpAddr, listenPort, clientIpAddr, 
          clientConnSocket.srcPort ) )
   t0( 'Server sd: %d, Client sd: %d' %
       ( serverConnSocket.fileDescriptor.descriptor,
         clientConnSocket.fileDescriptor.descriptor ) )
   del server
   del client
   return ( serverConnSocket.fileDescriptor, clientConnSocket.fileDescriptor )


