#!/usr/bin/env python
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CmdExtension
import ManagedSubprocess
import NsLib
import os
import subprocess

class CliNsCmdExtension( CmdExtension.CmdExtension ):
   def __init__( self ):
      CmdExtension.CmdExtension.__init__( self )

   def _genNsCmdLine( self, netNs, cliSession, cmdLine,
                      useSudo=True, results=None, vrfMount=False, vrfName=None ):
      assert netNs != None and netNs != ''
      assert isinstance( cmdLine, list )
      if netNs == NsLib.DEFAULT_NS:
         sudoAdvice = False
         if cmdLine[ 0 ] == 'sudo' and not useSudo:
            cmdLine.pop( 0 )
            sudoAdvice = True
         if results is not None:
            assert isinstance( results, dict )
            results[ 'Sudo' ] = sudoAdvice
         return True

      ncmd = []
      if useSudo:
         ncmd = [ 'sudo', '-E' ]
      ncmd.extend( [ 'ip', 'netns', 'exec', netNs ] )

      if vrfMount:
         assert vrfName
         ncmd.extend( [ 'vrfmount', vrfName ] )

      rootRequired = [ ( 'traceroute', '-i' ) ]
      if any( all( token in cmdLine for token in req ) for req in rootRequired ):
         # Fedora traceroute requires root privilages
         # if specifying a source interface
         pass
      elif cmdLine[ 0 ] == 'sudo':
         cmdLine.pop( 0 )
      else:
         ncmd.extend( [ 'sudo', '-E', '-u', '#%d' % os.getuid() ] )

      for x in reversed( ncmd ):
         cmdLine.insert( 0, x )
      if results is not None:
         assert isinstance( results, dict )
         results[ 'ns' ] = netNs
         results[ 'Sudo' ] = True
      return True

   @CmdExtension.setDefaultArgs
   def extendCmd( self, cmdLine, session, **kwargs ):
      netNs = kwargs[ 'netNs' ] if 'netNs' in kwargs else None
      useSudo = kwargs[ 'useSudo' ] if 'useSudo' in kwargs else True
      results = kwargs[ 'results' ] if 'results' in kwargs else None
      return self._genNsCmdLine( netNs, session, cmdLine,
                                 useSudo=useSudo, results=results )

   def _filterKwargs( self, filterArgList, kwargs ):
      for arg in filterArgList:
         if arg in kwargs:
            del kwargs[ arg ]
   
   @CmdExtension.setDefaultArgs
   def subprocessPopen( self, execCmd, session, **kwargs ):
      self.extendCmd( execCmd, session, **kwargs )
      self._filterKwargs( [ 'netNs', 'asRoot', 'useSudo' ], kwargs )
      return subprocess.Popen( args=execCmd, **kwargs)

   @CmdExtension.setDefaultArgs
   def managedSubprocessPopen( self, execCmd, session, **kwargs ):
      self.extendCmd( execCmd, session, **kwargs )
      self._filterKwargs( [ 'netNs', 'asRoot', 'useSudo' ], kwargs )
      return ManagedSubprocess.Popen( execCmd, **kwargs )

nsCmdExt = None

def addNsCmdExtender( extender ):
   global nsCmdExt
   if extender:
      nsCmdExt = extender
   else:
      nsCmdExt = CliNsCmdExtension()

def getNsCmdExtender():
   global nsCmdExt
   if not nsCmdExt:
      nsCmdExt = CliNsCmdExtension()
   return nsCmdExt
