# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements common utility functions for MPLS
#-------------------------------------------------------------------------------
'''Common utility functions for MPLS'''
import random

import CliCommand
import CliMatcher
import CliModel
import Tac

labelMin = Tac.Type( 'Arnet::MplsLabel').unassignedMin
labelMax = Tac.Type( 'Arnet::MplsLabel').max
MplsStackEntryIndex = Tac.Type( 'Arnet::MplsStackEntryIndex' )
MAX_LABEL_STACK_SIZE = MplsStackEntryIndex.max + 1
UnboundedMplsLabelOperation = Tac.Type( 'Arnet::UnboundedMplsLabelOperation' )

labelValMatcher = CliMatcher.IntegerMatcher( labelMin, labelMax,
                                             helpdesc='Value of the MPLS label' )

class LabelValWithExpNullExprFactory( CliCommand.CliExpressionFactory ):
   def generate( self, name ):
      class LabelValWithExpNullExpr( CliCommand.CliExpression ):
         expression = '{ ZERO | TWO | INTERNAL_LABEL }'
         data = {
            'ZERO' : CliMatcher.KeywordMatcher( '0',
               helpdesc='0: Explicit IPV4 Null' ),
            'TWO' : CliMatcher.KeywordMatcher( '2',
               helpdesc='2: Explicit IPV6 Null' ),
            'INTERNAL_LABEL' : labelValMatcher
         }

         @staticmethod
         def adapter( mode, args, argsList ):
            if name in args:
               return

            result = []
            for i in argsList:
               if i[ 0 ] not in ( 'ZERO', 'TWO', 'INTERNAL_LABEL' ):
                  continue
               result.append( i[ 1 ] )
            args[ name ] = result
      return LabelValWithExpNullExpr

labelsUsed = set()
def randomLabel( uniqueLabel=True ):
   label = random.randint( labelMin, labelMax )
   if uniqueLabel:
      # When unique label is requested, make sure unused labels are available to
      # avoid infinite loop below.
      assert len( labelsUsed ) <  labelMax - labelMin + 1
   while uniqueLabel and label in labelsUsed:
      label = random.randint( labelMin, labelMax )
   labelsUsed.add( label )
   return label

def genLabelStack( index, maxStackSize, genV4ExpNull=False, genV6ExpNull=False,
                   uniqueLabel=True ):
   stackSize = ( index % maxStackSize ) + 1
   stack = [ randomLabel( uniqueLabel=uniqueLabel ) for _ in range( stackSize ) ]
   if genV4ExpNull:
      # Change BOS label to 0
      stack[ stackSize - 1 ] = 0
   if genV6ExpNull:
      # Change BOS label to 2
      stack[ stackSize - 1 ] = 2
   return stack

def genLabelStacks( size, maxStackSize ):
   """Generate a list containing 'size' number of MPLS label stacks"""

   if not maxStackSize:
      # Platforms without MPLS push support have maxStackSize=0
      return None

   # Ensure 'size' input parameter is an integer
   assert isinstance( size, int )
   labelStacks = []
   for i in range( size ):
      labelStacks.append( genLabelStack( i, maxStackSize ) )
   return labelStacks

def labelListToBoundedMplsLabelStack( labelList, const=True ):
   labelStack = Tac.Value( "Arnet::BoundedMplsLabelStack" )
   for idx, label in enumerate( reversed( labelList ) ):
      labelStack.labelIs( idx, label )
   labelStack.stackSize = len( labelList )
   return Tac.const( labelStack ) if const else labelStack

def boundedMplsLabelStackToLabelList( labelStack ):
   labelList = []
   for i in range( labelStack.stackSize ):
      labelList.append( labelStack.label( i ) )
   labelList.reverse()
   return labelList

def labelStackToMplsLabelOperation( labelList, operation='push', controlWord=False,
                                    entropyLabelPositionBitMap=None,
                                    addFlowLabel=False ):
   mplsLabelOperation = Tac.newInstance( 'Arnet::MplsLabelOperation' )
   for idx, value in enumerate( labelList ):
      mplsLabelOperation.labelStackIs( idx, value )
      setattr( mplsLabelOperation, 'label%d' % idx, value )
   elPositions = entropyLabelPositionBitMap or []
   for elPos in elPositions:
      mplsLabelOperation.entropyLabelAndIndicatorIs( elPos )
   mplsLabelOperation.operation = operation
   mplsLabelOperation.controlWord = controlWord
   mplsLabelOperation.addFlowLabel = addFlowLabel
   mplsLabelOperation.stackSize = len( labelList )
   return mplsLabelOperation

def genMplsLabelOperationList( size, maxStackSize ):
   return map( labelStackToMplsLabelOperation, genLabelStacks( size, maxStackSize ) )

def getLabelStack( mplsLabelOperation ):
   """Generate a list of MPLS labels from a MplsLabelOperation value type.
      In the value type, the stack will be stored as the following:
      
      BOTTOM OF STACK (BOS)
      label0: ...
      label1: ...
      label2: ...
      TOP OF STACK (TOS)
      
      The list that is returned will look like the following:
      TOS ---> [ label2, label1, label0 ] <--- BOS"""

   # Ensure that input parameter is of type 'MplsLabelOperation'
   assert isinstance( mplsLabelOperation, Tac.Type( 'Arnet::MplsLabelOperation' ) )
   labelStack = []
   for i in reversed( range( mplsLabelOperation.stackSize ) ):
      labelStack.append( mplsLabelOperation.labelStack( i ) )
   return labelStack

def labelStackToString( labelStack ):
   """Takes an MPLS label stack, represented as either a list or tuple, and
      converts it into a string representation. For example, [ 10, 20, 30 ]
      will be converted to "10, 20, 30"."""

   assert ( isinstance( labelStack, list ) or isinstance( labelStack, tuple ) or \
      isinstance( labelStack, CliModel._TypedList ) ) # pylint: disable-msg=W0212
   return ' '.join( map( str, labelStack ) )
