#!/usr/bin/env python
# Copyright (c) 2008, 2009, 2010 Arastra, Inc.  All rights reserved.
# Arastra, Inc. Confidential and Proprietary.

import sys, re

def prefixToType( prefix ):
   '''Match an interface type prefix to the full interface type string and
   return that string.'''
   
   types = [ 'ethernet', 'port-channel', 'vlan', 'management', 'test', 'loopback',
             'vxlan', 'recirc-channel', 'unconnectedethernet', 'tunnel' ]
   for t in types:
      if t.startswith( prefix.lower() ):
         return t
   return None

def matchIntCmds( patterns, cmds, subIntf=False ):
   '''This function filters config file contents to include only those lines
   forming the interface config submodes for interfaces matching patterns
   provided.  Patterns can be any interface type prefix followed by the
   interface number, a simple interface range, or a regular expression using
   the python re module syntax.  If a pattern includes only an interface
   prefix, we assume that the rest of the pattern is in the next entry of
   the pattern list.

   This function forms the core of the "int" bash command as well as the
   "show {running,startup}-config interface" commands.

   The function returns a 2 entry tuple.  The first entry is the list of
   matching lines from the config.  The second is an error message if any
   occurred.
   
   For exact match patterns like 'et1', we avoid matching against each
   pattern one at a time and instead just do an exact match lookup using
   a set.  This is much, much faster when there are a lot of patterns.'''
   exactFilters = set()
   if len( patterns ) > 0:
      filters = []
      idx = 0
      while idx < len( patterns ):
         pattern = patterns[ idx ]
         idx += 1
         m = re.search( '^([^\d$^*.+?\[\]\|\{\}\(\)]+)', pattern )
         # Try to pick off the leading prefix indicating the interface type
         if m is not None:
            prefix = m.group( 1 )
            intfType = prefixToType( prefix )
            if intfType is None:
               return (None, "Error recognizing interface name: " + pattern)
            if prefix == pattern:
               # There are no numbers or regex chars in the pattern.  This must be
               # a two part pattern such as 'vlan 1'.
               if( idx == len( patterns ) ):
                  return (None, "Invalid interface pattern: " + pattern )
               suffix = patterns[ idx ]
               idx += 1
               m = re.match( '^([\d$^*.+?\[\]\|\{\}\(\)]+)', suffix )
               if m is None:
                  return (None, "Invalid interface pattern: " + pattern +
                          " " + suffix )
            else:
               suffix = pattern[ len( prefix ) : ]
            m = re.match( '^(\d+/)?(\d+)-(\d+)$', suffix )
            if m is not None:
               # We matched on a number range.  Make that work.
               start = int( m.group( 2 ) )
               end = int( m.group( 3 ) )
               if start >= end:
                  return (None, "Invalid interface range: " + pattern )
               if m.group( 1 ) is not None:
                  slot = m.group( 1 )
               else:
                  slot = ''
               for i in range( start, end + 1 ):
                  intfFilter = 'interface ' + intfType + slot + str( i )
                  exactFilters.add( intfFilter )                  
               continue
            if subIntf: # Optimization for matching sub interface names.
               m = re.search( '([$^*+?\[\]\|\{\}\(\)]+)', suffix )
            else:
               m = re.search( '([$^*.+?\[\]\|\{\}\(\)]+)', suffix )
            if m is not None:
               # The suffix looks like a regular expression.
               intfFilter = '^interface\s*' + intfType + suffix + '\s*' + '$'
               filters.append( intfFilter )
               continue
            # It looks like a basic interface name.  Just do an exact match.
            intfFilter = 'interface ' + intfType + suffix
            exactFilters.add( intfFilter )
            continue
         m = re.search( '[*?!\[\]]', pattern )
         # If we couldn't find a prefix, it should be a regular expression.
         # Otherwise it is probably just a typo.
         if m is not None:
            # It looks like a regular expression.  Tack on the leading '^interface'
            # and swallow it.
            filters.append( re.compile( '^interface ' + pattern ) )
            continue
         return (None, "Error recoginizing interface name pattern: " + pattern )
   else:
      filters = [ re.compile( '.*' ) ]

   matchedLines = []
   matchedCmd = False
   for line in cmds:
      if matchedCmd:
         if line.startswith( '   ' ):
            matchedLines.append( line )
            continue
      matchedCmd = False
      if line.rstrip().lower() in exactFilters:
         matchedCmd = True
      else:
         for filt in filters:
            m = re.match( filt, line.lower() )
            matchedCmd = (m is not None)
            if matchedCmd:
               break
      if matchedCmd:
         matchedLines.append( line )

   return (matchedLines, None)


if __name__ == '__main__':
   (matchedLines, error) = matchIntCmds( sys.argv[ 1 : ], sys.stdin.xreadlines() )
   if error:
      print error
   else:
      for line in matchedLines:
         print line,
         
