#!/usr/bin/env python
# Copyright (c) 2008, 2009, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac, Tracing, Arnet.Test, WaitForWarmup
from Arnet.NsLib import DEFAULT_NS, runMaybeInNetNs
import re, random

t0 = Tracing.trace0
t1 = Tracing.trace1
t2 = Tracing.trace2

receiveBufferSizeDefault = 124928
neighborCacheFile = "file:/persist/sys/neighborCacheComplete"

def getRandomIp( v6 ):
   if not v6:
      ip = '.'.join( '%s' % random.randint( 0, 255 ) for i in range( 4 ) )
   else:
      ip = "1001:abcd:" + \
           ":".join( ( "%x" % random.randint( 0, 16**4 - 1 ) for i in range( 6 ) ) )
   return ip

def getRandomMac():
   b0 = hex( random.randint( 0, 15 ) ).split( 'x' )[ 1 ]
   b1 = hex( random.randint( 0, 15 ) ).split( 'x' )[ 1 ]
   b2 = hex( random.randint( 0, 15 ) ).split( 'x' )[ 1 ]
   b3 = hex( random.randint( 0, 15 ) ).split( 'x' )[ 1 ]
   b4 = hex( random.randint( 0, 15 ) ).split( 'x' )[ 1 ]
   b5 = hex( random.randint( 0, 15 ) ).split( 'x' )[ 1 ]
   b6 = hex( random.randint( 0, 15 ) ).split( 'x' )[ 1 ]
   b7 = hex( random.randint( 0, 15 ) ).split( 'x' )[ 1 ]
   b8 = hex( random.randint( 0, 15 ) ).split( 'x' )[ 1 ]
   b9 = hex( random.randint( 0, 15 ) ).split( 'x' )[ 1 ]
   return '00:%s%s:%s%s:%s%s:%s%s:%s%s' % (b0, b1, b2, b3, b4, b5, b6, b7, b8, b9 )

def arpSmashToNeighborDir( arpSmash ):
   '''
   Convert arp smash into Arp::NeighborDir.

   Return None if arpSmash is None.
   '''
   if not arpSmash:
      return None

   nd = Tac.newInstance( "Arp::NeighborDir", "temp" )

   for entry in arpSmash.arpEntryTable().itervalues():
      intfName = entry.intfId
      if not intfName in nd.arpIntfStatus:
         nd.arpIntfStatus.newMember( intfName )
      ais = nd.arpIntfStatus[ intfName ]
      aes = Tac.newInstance( "Arp::ArpEntryStatus", entry.addr.v4Addr )
      aes.ethAddr = entry.ethAddr
      aes.isStatic = entry.isStatic
      aes.source = entry.source
      ais.arpEntry.addMember( aes )

   for entry in arpSmash.neighborEntryTable().itervalues():
      intfName = entry.intfId
      if not intfName in nd.arpIntfStatus:
         nd.arpIntfStatus.newMember( intfName )
      ais = nd.arpIntfStatus[ intfName ]
      neighborEntry = Tac.newInstance( "Arp::NeighborEntryStatus",
            entry.addr.v6Addr, entry.ethAddr, entry.isStatic )
      neighborEntry.source = entry.source
      ais.ip6NeighborEntry.addMember( neighborEntry )
   return nd

# make sure that the neighbor entries in the two neighbor tables are in sync
def compareNeighborDirs( neighborDirByDeviceName, neighborDirByIntfName,
                         intfStatusKernel ):
   intfNamesSeen = set()
   for ( deviceName, ais ) in neighborDirByDeviceName.arpIntfStatus.iteritems():
      if deviceName in intfStatusKernel.intfStatus:
         intfStatus = intfStatusKernel.intfStatus[ deviceName ]
         ais2 = neighborDirByIntfName.arpIntfStatus[ intfStatus.name ]
         assert ais == ais2

         intfNamesSeen.add( intfStatus.name )
      else:
         t2( "deviceName '%s' has no mapping to an intfName" % ( deviceName, ) )

   # make sure that there are no stray entries in neighborDirByIntfName
   assert intfNamesSeen == set( neighborDirByIntfName.arpIntfStatus.keys() )

def readKernelIp6NeighborTable( netNs=DEFAULT_NS ):
   # Build a dict containing the kernel's ipv6 Neighbor table.
   karp = {}

   arpRe = re.compile( 
         r"^([A-Fa-f0-9:.]{0,46})\s+\S+\s+(\S+)\s+\S+\s+(\S+)\s+(\S+)$" )

   # store just the ip addresses that we have seen
   ips = {}
   output = runMaybeInNetNs( netNs, [ "ip", "-6", "neigh", "show" ],
                             stdout=Tac.CAPTURE, asRoot=True )

   lines = output.strip().split( '\n' )

   for line in lines:
      m = arpRe.match( line )
      if m:
         ip = m.group( 1 )
         deviceName = m.group( 2 )
         eth = m.group( 3 )
         static = bool( m.group( 4 ) == "PERMANENT" )
         
         karp.setdefault( deviceName, {} )[ ip ] = ( eth, static )
         ips[ ip ] = ips.get( ip, [] ) + [ deviceName ]
      else:
         t0( "could not process line in kernel arp table: '%s'" % ( line, ) )
   return karp, ips

def compareIp6NeighborsAgainstSysfs( entityManager,
                                     neighborDirByDeviceName,
                                     arpSmash,
                                     intfStatusKernel,
                                     agentsToWaitFor=None,
                                     netNs=DEFAULT_NS ):
   if agentsToWaitFor:
      assert entityManager, "You must pass entityManager for agentsToWaitFor to work"
      WaitForWarmup.wait( entityManager, agentList=agentsToWaitFor, verbose=True )
   Tac.runActivities( 0 )

   neighborDirByIntfName = arpSmashToNeighborDir( arpSmash )

   karp, _ = readKernelIp6NeighborTable( netNs )

   d = neighborDirByIntfName
   if d is None:
      d = neighborDirByDeviceName

   for key in d.arpIntfStatus:
      t2( "have ARP info for key %s" % ( key, ) )

      arpIntfStatus = d.arpIntfStatus[ key ]
      neighEntries = arpIntfStatus.ip6NeighborEntry

      if d is neighborDirByIntfName:
         deviceName = intfStatusKernel[ key ].deviceName
      else:
         deviceName = key

      if deviceName not in karp:
         if len( neighEntries ):
            message = ( "dir contains a device that is not "
                        " in the kernel neigh table: "
                        "'%s'" % ( deviceName, ) )
            t0( message )
            assert False, message
         continue
      
      karpDeviceNameDict = karp[ deviceName ]

      for ipAddr, neighEntry in neighEntries.iteritems():
         t2( "have neigh entry for %s = %s" % ( ipAddr, neighEntry ) )

         if ipAddr.stringValue not in karpDeviceNameDict:
            message = ( "dir contains an ip address for %s "
                        "that is not in the "
                        "kernel neigh table: '%s'" % ( deviceName, ipAddr ) )
            t0( message )
            assert False, message

         ( ethAddr, _ ) = karpDeviceNameDict[ ipAddr.stringValue ]
         assert ethAddr == neighEntry.ethAddr
         
         
         # remove this ip address so only those that aren't tracked will
         # be in the dictionary
         del karpDeviceNameDict[ ipAddr.stringValue ]
      
      # scrub invalid entries that the kernel still shows
      for ipAddr, ( ethAddr, _ ) in karpDeviceNameDict.items():
         if ethAddr == '00:00:00:00:00:00':
            del karpDeviceNameDict[ ipAddr ]
      
      if len( karpDeviceNameDict ) != 0:
         # there were some arp entries that we did not notice for this device
         message = ( "dir for %s does not contain neigh entries "
                     "for %s" % ( deviceName, karpDeviceNameDict.items() ) )
         t0( message )
         assert False, message

      del karp[ deviceName ]

   if len( karp ) != 0:
      # there were some devices that we did not notice
      message = ( "dir did not contain neigh entries for "
                  "%s" % ( karp.keys() ) )
      t0( message )
      assert False, message

   if ( ( neighborDirByIntfName is not None ) and
        ( neighborDirByDeviceName is not None ) and
        ( intfStatusKernel is not None ) ):
      compareNeighborDirs( neighborDirByDeviceName, neighborDirByIntfName,
                           intfStatusKernel )

def readKernelArpTable( netNs=DEFAULT_NS ):
   # Build a dict containing the kernel's ARP table.
   karp = {}

   arpRe = re.compile( 
      r"(\d+\.\d+\.\d+\.\d+)\s+\S+\s+(\S+)\s+(\S+)\s+\S+\s+(\S+)$" )
   # From include/linux/if_arp.h
   ATF_COM = 0x02
   ATF_PERM = 0x04

   # store just the ip addresses that we have seen
   ips = {}

   #if netNs is DEFAULT_NS:
   #   fo = file( '/proc/net/arp', 'r' )
   #else:
   fo = runMaybeInNetNs( netNs, [ 'cat', '/proc/net/arp' ], 
                         stdout=Tac.CAPTURE, asRoot=True )
   fo = fo.split( '\n' )

   for line in fo:
      if line == '':
         continue
      m = arpRe.match( line )
      if m:
         ip = m.group( 1 )
         flags = int( m.group( 2 ), 16 )
         static = bool( flags & ATF_PERM )
         if flags & ATF_COM:
            eth = m.group( 3 )
         else:
            # Entries that have been manually deleted from the ARP table will be
            # incomplete, but the old Ethernet address still appears in
            # /proc/net/arp.  We need to ignore that old address.
            eth = '00:00:00:00:00:00'
         deviceName = m.group( 4 )
         
         karp.setdefault( deviceName, {} )[ ip ] = ( eth, static )
         ips[ ip ] = ips.get( ip, [] ) + [ deviceName ]
      else:
         if line != ( "IP address       HW type     Flags       HW address" +
                      "            Mask     Device" ):
            t0( "could not process line in kernel arp table: '%s'" % ( line, ) )
   return karp, ips

def compareNeighborsAgainstSysfs( entityManager,
                                  neighborDirByDeviceName,
                                  arpSmash,
                                  intfStatusKernel,
                                  agentsToWaitFor=None,
                                  netNs=DEFAULT_NS ):
   if agentsToWaitFor:
      assert entityManager, "You must pass entityManager for agentsToWaitFor to work"
      WaitForWarmup.wait( entityManager, agentList=agentsToWaitFor, verbose=True )
   Tac.runActivities( 0 )

   neighborDirByIntfName = arpSmashToNeighborDir( arpSmash )

   karp, _ = readKernelArpTable( netNs )

   directory = neighborDirByIntfName
   if directory is None:
      directory = neighborDirByDeviceName

   for key in directory.arpIntfStatus:
      t2( "have ARP info for key %s" % ( key, ) )

      arpIntfStatus = directory.arpIntfStatus[ key ]
      arpEntries = arpIntfStatus.arpEntry

      if directory is neighborDirByIntfName:
         deviceName = intfStatusKernel[ key ].deviceName
      else:
         deviceName = key

      if deviceName not in karp:
         if len( arpEntries ):
            message = ( "dir contains a device that is not "
                        " in the kernel arp table: "
                        "'%s'" % ( deviceName, ) )
            t0( message )
            assert False, message
         continue
      
      karpDeviceNameDict = karp[ deviceName ]

      for ipAddr, arpEntry in arpEntries.iteritems():  # pylint: disable-msg=W0621 
         t2( "have ARP entry for %s = %s" % ( ipAddr, arpEntry ) )

         if ipAddr not in karpDeviceNameDict:
            message = ( "dir contains an ip address for %s "
                        "that is not in the "
                        "kernel arp table: '%s'" % ( deviceName, ipAddr ) )
            t0( message )
            assert False, message

         ( ethAddr, _ ) = karpDeviceNameDict[ ipAddr ]
         assert ethAddr == arpEntry.ethAddr
         
         
         # remove this ip address so only those that aren't tracked will
         # be in the dictionary
         del karpDeviceNameDict[ ipAddr ]
      
      # scrub invalid entries that the kernel still shows
      for ipAddr, ( ethAddr, _ ) in karpDeviceNameDict.items():
         if ethAddr == '00:00:00:00:00:00':
            del karpDeviceNameDict[ ipAddr ]
      
      if len( karpDeviceNameDict ) != 0:
         # there were some arp entries that we did not notice for this device
         message = ( "dir for %s does not contain arp entries "
                     "for %s" % ( deviceName, karpDeviceNameDict.items() ) )
         t0( message )
         assert False, message

      del karp[ deviceName ]

   if len( karp ) != 0:
      # there were some devices that we did not notice
      message = ( "dir did not contain arp entries for "
                  "%s" % ( karp.keys() ) )
      t0( message )
      assert False, message

   if ( ( neighborDirByIntfName is not None ) and
        ( neighborDirByDeviceName is not None ) and
        ( intfStatusKernel is not None ) ):
      compareNeighborDirs( neighborDirByDeviceName, neighborDirByIntfName,
                           intfStatusKernel )

# Modify various properties of the tap device.
def ipLinkSetForNeighborTest( deviceName, params, netNs=DEFAULT_NS ):
   cmd = [ "ip", "link", "set", deviceName ] + params
   t0( "Running", cmd )
   runMaybeInNetNs( netNs, cmd )

def addArpEntry( deviceName, hostname, macAddr, netNs=DEFAULT_NS ):
   '''
   Adds Arp entry to Kernel Arp Table
   Wrapper function for addNdEntry
   '''
   addNdEntry( deviceName, hostname, macAddr, netNs=netNs )
  
def delArpEntry( deviceName, hostname, netNs=DEFAULT_NS ):
   '''
   Deletes Arp entry from Kernel Arp Table
   Wrapper function for delNdEntry
   '''
   delNdEntry( deviceName, hostname, netNs=netNs )

def localAddressAdd( devName, ipAddress, prefixLen, netNs=DEFAULT_NS ):
   runMaybeInNetNs( netNs, [ "ip", "addr", "add", ipAddress  + "/" + 
                             str( prefixLen ), "dev", devName ] )

def arpEntryExistsWithState( deviceName, ipAddr, macAddr,
                             stateRe, ipv6Host=False, netNs=DEFAULT_NS ):
   cmd = [ "ip", "neigh", "show", ipAddr, "dev", deviceName ]
   if ipv6Host:
      cmd.insert( 1, "-6" )
   else:
      cmd.insert( 1, "-4" )
   if netNs != DEFAULT_NS:
      cmd = [ "ip", "netns", "exec", netNs ] + cmd
   arpOutput = Tac.run( cmd, stdout=Tac.CAPTURE, ignoreReturnCode=True )
   matches = re.match( "(\S+) lladdr (\S+) (router )*(\S+)",
                       arpOutput )
   return matches and matches.group( 1 ) == ipAddr and \
          matches.group( 2 ) == macAddr and re.match( stateRe, matches.group( 4 ) )

def arpEntryExists( deviceName, peerIpAddress, peerMacAddress=".*", 
                    netNs=DEFAULT_NS ):
   """for network namespaces created using ip netns"""
   cmd = [ "ip", "neigh", "show" ]
   arpOutput = runMaybeInNetNs( netNs, cmd, stdout=Tac.CAPTURE )
   expectedArpOutputPattern = ( peerIpAddress + ".*" + deviceName +
                                ".*" + peerMacAddress.lower() )
   return( re.search( expectedArpOutputPattern, arpOutput ) )


def confirmArpEntryExists( deviceName, peerIpAddress, peerMacAddress,
                           netNs=DEFAULT_NS ):
   assert arpEntryExists( deviceName, peerIpAddress, peerMacAddress, netNs )

def arpEntryInNetnsExists( hostName, hostIntfName, peerIpAddress, 
                           peerMacAddress=".*" ):
   """for network namespaces created via netns"""
   arpOutput = Tac.run( [ "netns", hostName, "arp", "-n", peerIpAddress ], 
                        stdout=Tac.CAPTURE )
   expectedArpOutputPattern = ( peerIpAddress + ".*" + peerMacAddress.lower() +
                                ".*" + hostIntfName )
   return( re.search( expectedArpOutputPattern, arpOutput ) )

def arpEntryConfirmedTimeValue( deviceName, peerIpAddress ):
   # example: 172.17.0.22 lladdr 00:11:43:1b:81:5f ref 4 used 31/26/1 REACHABLE
   expectedPattern = "used \d*/(\d*)/\d*"

   ipOutput = Tac.run( [ "ip", "-s", "neigh", "show", "to", peerIpAddress,
                         "dev", deviceName ], stdout=Tac.CAPTURE )
   m = re.search( expectedPattern, ipOutput )
   if m:
      confirmedTime = m.group( 1 )
      return confirmedTime

   assert False

def arpEntryUsedValue( deviceName, peerIpAddress ):
   # example: 172.17.0.22 lladdr 00:11:43:1b:81:5f ref 4 used 31/26/1 REACHABLE
   expectedPattern = "used (\d*)/\d*/\d*"

   ipOutput = Tac.run( [ "ip", "-s", "neigh", "show", "to", peerIpAddress,
                         "dev", deviceName ], stdout=Tac.CAPTURE )
   m = re.search( expectedPattern, ipOutput )
   if m:
      usedTime = m.group( 1 )
      return usedTime

   assert False
                    
def localAddressReplace( device, ipAddress, prefixLen, netNs=DEFAULT_NS ):
   runMaybeInNetNs( netNs, [ "ip", "addr", "replace", ipAddress  + "/" + 
                             str( prefixLen ), "dev", device.name ] )
def localAddressChange( device, ipAddress, prefixLen, netNs=DEFAULT_NS ):
   runMaybeInNetNs( netNs, [ "ip", "addr", "change", ipAddress  + "/" + 
                             str( prefixLen ), "dev", device.name ] )


def localAddressDel( device, ipAddress, prefixLen, netNs=DEFAULT_NS ):
   runMaybeInNetNs( netNs, [ "ip", "addr", "del", ipAddress  + "/" + 
                             str( prefixLen ), "dev", device.name ] )

def arpEntry( ipAddr, ethAddr, netNs=DEFAULT_NS ):
   # adds an arp entry without specifying a device, use arp(8) because ip(8), via
   # ip neigh add, requires a device
   runMaybeInNetNs( netNs, [ "arp", "-s", ipAddr, ethAddr ] )

def addNdEntry( intf, ipAddr, macAddr, useV6=False, netNs=DEFAULT_NS ):
   if useV6:
      ipCmdList = [ 'ip', '-6' ]
   else:
      ipCmdList = [ 'ip' ]
   ipCmdList += [ 'neigh', 'replace', ipAddr, 'lladdr', macAddr,
                  'nud', 'permanent', 'dev', intf ] 
   t0( "Running", ipCmdList )
   runMaybeInNetNs( netNs, ipCmdList, asRoot=True )

def delNdEntry( intf, ipAddr, useV6=False, netNs=DEFAULT_NS ):
   if useV6:
      ipCmdList = [ 'ip', '-6' ]
   else:
      ipCmdList = [ 'ip' ]
   ipCmdList += [ 'neigh', 'del', ipAddr, 'dev', intf ]
   t0( "Running", ipCmdList )
   runMaybeInNetNs( netNs, ipCmdList )

def _getNdList( intf, ipAddr, useV6, netNs ):
   if useV6:
      ipCmdList = [ 'ip', '-6' ]
   else:
      ipCmdList = [ 'ip' ]

   ipCmdList = ipCmdList + [ 'neigh', 'show', 'to',
                             ipAddr, 'dev', intf ]
   neighOutput = runMaybeInNetNs( netNs, ipCmdList, asRoot=True,
                                  stdout=Tac.CAPTURE )
   return neighOutput

def _ndEntryPresent( intf, ipAddr, present, useV6, netNs ):
   neighOutput = _getNdList( intf, ipAddr, useV6, netNs ) 
   if ( present == False and neighOutput == "" ) or\
          ( present == True and neighOutput != "" ):
      return True
   return False

def checkNdEntry( intf, ipAddr, present, useV6=False, netNs=DEFAULT_NS ):
   Tac.waitFor( lambda: _ndEntryPresent( intf, ipAddr, present, useV6, netNs ),
                description="check for a neighbor entry" )

def createTapDevice( deviceName, intfName, sysdbRoot, netNs ):
   if sysdbRoot is None:
      tap = Arnet.Device.Tap( deviceName, netNs=netNs )
   else:
      tap = Arnet.Test.Tap( intfName, sysdbRoot, deviceName, netNs )

   return tap

def testDuplicateIpAddress( sysdbRoot, entityManager,
                            neighborDirByDeviceName, arpSmash,
                            intfStatusKernel, agentsToWaitFor=None,
                            netNs=DEFAULT_NS ):

   def checkSysfs( ns ):
      compareNeighborsAgainstSysfs( entityManager,
                                    neighborDirByDeviceName, 
                                    arpSmash,
                                    intfStatusKernel,
                                    agentsToWaitFor, ns )

   t0( "testing when the same ip address is in multiple arp entries" )

   deviceName = 'rtnmTest4Dup'
   intfName = 'Test004'
   tapTest4Dup = createTapDevice( deviceName, intfName, sysdbRoot, netNs )
 
   localAddressAdd( tapTest4Dup.name, "10.20.100.1", 24, netNs )

   print runMaybeInNetNs( netNs, [ "ip", "addr", "show", "dev", deviceName ], 
                          stdout=Tac.CAPTURE )


   tapTest4Dup.arpSet( "10.20.100.19", "18:00:00:00:00:0f" )

   # this one show up as a secondary address because it is in the first
   # address' subnet
   localAddressAdd( tapTest4Dup.name, "10.20.100.4", 24, netNs )
   print runMaybeInNetNs( netNs, [ "ip", "addr", "show", "dev", deviceName ], 
                          stdout=Tac.CAPTURE )
   localAddressReplace( tapTest4Dup, "10.20.101.2", 24, netNs )
   print runMaybeInNetNs( netNs, [ "ip", "addr", "show", "dev", deviceName ], 
                          stdout=Tac.CAPTURE )
   localAddressChange( tapTest4Dup, "10.20.102.3", 24, netNs )
   print runMaybeInNetNs( netNs, [ "ip", "addr", "show", "dev", deviceName ], 
                          stdout=Tac.CAPTURE )
   localAddressAdd( tapTest4Dup.name, "10.20.103.4", 24, netNs )
   print runMaybeInNetNs( netNs, [ "ip", "addr", "show", "dev", deviceName ], 
                          stdout=Tac.CAPTURE )

   deviceName2 = "rtnmTest4Dup2"
   intfName2 = 'Test004/2'
   tapTest4Dup2 = createTapDevice( deviceName2, intfName2, sysdbRoot, netNs )

   localAddressChange( tapTest4Dup2, "10.20.106.1", 24, netNs )

   deviceName3 = "rtnmTest4Dup3"
   tapTest4Dup3 = createTapDevice( deviceName3, "Test004/3", sysdbRoot, netNs )
   localAddressChange( tapTest4Dup3, "10.20.102.4", 24, netNs )

   print runMaybeInNetNs( netNs, [ "ip", "addr", "show", "dev", deviceName ], 
                          stdout=Tac.CAPTURE )
   print runMaybeInNetNs( netNs, [ "ip", "addr", "show", "dev", deviceName2 ], 
                          stdout=Tac.CAPTURE )

   # we can create two ARP entries that are for the same IP address on 
   # two devices
   arpEntry( "10.20.100.9", "08:00:00:00:00:9", netNs )
   tapTest4Dup2.arpSet( "10.20.100.9", "28:00:00:00:00:29" )
   tapTest4Dup3.arpSet( "10.20.100.9", "38:00:00:00:00:39" )
   runMaybeInNetNs( netNs, [ "cat", "/proc/net/arp" ] )

   checkSysfs( netNs )

   # if the only ip address on a device goes away, the arp entries are flushed
   localAddressDel( tapTest4Dup2, "10.20.106.1", 24, netNs )
   print runMaybeInNetNs( netNs, [ "ip", "addr", "show", "dev", deviceName2 ], 
                          stdout=Tac.CAPTURE )
   print runMaybeInNetNs( netNs, [ "cat", "/proc/net/arp" ], 
                          stdout=Tac.CAPTURE )

   # make sure that the the neighbor dir has an entry for the ip address which 
   # used to be present twice
   checkSysfs( netNs )


   # if we remove an ip address, but there are still others on the device,
   # the device's arp entries are not flushed even if they correspond to the 
   # address that went away
   localAddressChange( tapTest4Dup2, "10.20.106.1", 24, netNs )
   localAddressChange( tapTest4Dup2, "10.20.107.1", 24, netNs )
   print runMaybeInNetNs( netNs, [ "ip", "addr", "show", "dev", deviceName2 ], 
                          stdout=Tac.CAPTURE )
   arpEntry( "10.20.106.9", "08:00:00:00:00:9", netNs )
   localAddressDel( tapTest4Dup2, "10.20.106.1", 24, netNs )
   print runMaybeInNetNs( netNs, [ "ip", "addr", "show", "dev", deviceName2 ], 
                          stdout=Tac.CAPTURE )
   print runMaybeInNetNs( netNs, [ "cat", "/proc/net/arp" ], 
                          stdout=Tac.CAPTURE )
   # this will cause errors because the interface does not have an address for
   # this arp entry
   try:
      arpEntry( "10.20.106.10", "08:00:00:00:00:9", netNs )
   except Tac.SystemCommandError:
      # XXX might want to confirm that it is a network is unreachable error
      pass

   checkSysfs( netNs )

   # test what happens when devices go away
   tapTest4Dup2 = None
   checkSysfs( netNs )
   tapTest4Dup3 = None
   checkSysfs( netNs )
   tapTest4Dup = None
   checkSysfs( netNs )
   
   

def runNeighborTests( entityManager, sysdbRoot,
                      neighborDirByDeviceName,
                      arpSmash,
                      intfStatusKernel,
                      agentsToWaitFor=None,
                      netNs=DEFAULT_NS ):
   def checkSysfs( ns ):
      compareNeighborsAgainstSysfs( entityManager,
                                    neighborDirByDeviceName, 
                                    arpSmash,
                                    intfStatusKernel,
                                    agentsToWaitFor, ns )
      
   t0( "Create a tap device for the neighbor tests" )

   deviceName = 'rtnmTest1'
   intfName = 'Test11'
   tap = createTapDevice( deviceName, intfName, sysdbRoot, netNs )
   checkSysfs( netNs )

   t0( "adding arp entries" )
   hostname = "172.17.0.251"
   addArpEntry( deviceName, hostname, "00:11:43:1b:81:5f", netNs )
   checkSysfs( netNs )

   hostname = "172.17.0.128"
   addArpEntry( deviceName, hostname, "00:11:43:1b:89:4f", netNs )
   checkSysfs( netNs )

   t0( "modifying mac address for ip address with an existing arp entry" )
   addArpEntry( deviceName, hostname, "00:11:43:1b:81:50", netNs )
   checkSysfs( netNs )

   t0( "testing deleted arp entries" )
   delArpEntry( deviceName, hostname, netNs )
   checkSysfs( netNs )

   t0( "testing re-add of deleted arp entry" )
   addArpEntry( deviceName, hostname, "00:22:86:1b:81:5f", netNs )
   checkSysfs( netNs )

   t0( "bringing down interface" )
   ipLinkSetForNeighborTest( deviceName, [ "down" ], netNs )
   checkSysfs( netNs )

   t0( "bringing up interface" )
   ipLinkSetForNeighborTest( deviceName, [ "up" ], netNs )
   checkSysfs( netNs )
      
   # XXX: let the arp entries get timed out and see what messages we get,
   #      e.g. DELNEIGH
   # XXX: add arp entries on different interfaces for the same MAC address
   # XXX: trigger handleKernelIntf for a deviceName going away
   
   # sleep to see arp entries get timed out
   t0( "sleeping in hopes that arp entries will expire" )
   t1( "arp entries before sleeping: " + 
       runMaybeInNetNs( netNs, [ "arp" ], stdout=Tac.CAPTURE ) )
   import time
   time.sleep( 1 )
   
   t0( "done sleeping" )
   t1( "arp entries after sleeping: " + 
       runMaybeInNetNs( netNs, [ "arp" ], stdout=Tac.CAPTURE ) )

   t0( "adding arp entries before destroying tap device" )
   addArpEntry( deviceName, hostname, "00:11:43:1b:81:5c", netNs )
   checkSysfs( netNs )
   t0( "Destroy the tap device" )
   Tac.runActivities( 0 )
   t0( "Destroyed the tap device" )

   checkSysfs( netNs )

   # add arp entries on different interfaces for the same IP address
   testDuplicateIpAddress( sysdbRoot, entityManager,
                           neighborDirByDeviceName,
                           arpSmash,
                           intfStatusKernel, agentsToWaitFor, netNs )
   # This avoids "Unused variable" pylint error
   del tap
