#!/usr/bin/env python
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import Tac

def createRoot( entityManager=None, name="Root", cliConfig=None,
                cAgent=None, requireAllValid=False ):
   if requireAllValid or entityManager:
      assert entityManager
      if requireAllValid:
         assert cliConfig
         assert cAgent
      root = entityManager.root().parent[ 'Etba' ].newEntity(
         'Arfa::Root', name )
   else:
      root = Tac.newInstance( "Arfa::Root", name )

   cEntityManager = None
   if entityManager:
      cEntityManager = entityManager.cEntityManager()

   root.cliConfig = cliConfig
   root.cAgent = cAgent
   root.entityManager = cEntityManager
   return root

def setupBridgeRoot( bridgeRoot, brConfig=None, sbrStatus=None, topoStatus=None,
                     ethIntfStatusDir=None, topoConfig=None, ethIntfStatusLocal=None,
                     requestCli=None, replyCli=None, clock=None,
                     requireAllValid=False ):
   if requireAllValid:
      assert brConfig
      assert sbrStatus
      assert ethIntfStatusLocal
      assert topoConfig
      assert ethIntfStatusLocal
      assert requestCli
      assert replyCli
      assert topoStatus
      assert clock
   bridgeRoot.brConfig = brConfig
   bridgeRoot.sbrStatus = sbrStatus
   bridgeRoot.ethIntfStatusDir = ethIntfStatusDir
   bridgeRoot.topoConfig = topoConfig
   bridgeRoot.ethIntfStatusLocal = ethIntfStatusLocal
   bridgeRoot.requestCli = requestCli
   bridgeRoot.replyCli = replyCli
   bridgeRoot.topoStatus = topoStatus
   bridgeRoot.clock = clock

   bridgeRoot.startPipeline()

def createAll( brConfig=None, sbrStatus=None, ethIntfStatusDir=None,
               topoConfig=None, ethIntfStatusLocal=None, entityManager=None,
               cAgent=None, requestCli=None, replyCli=None, topoStatus=None,
               clock=None, cliConfig=None, requireAllValid=False ):
   root = createRoot( entityManager=entityManager, cliConfig=cliConfig,
                      cAgent=cAgent, requireAllValid=requireAllValid )
   setupBridgeRoot( root.bridgeRoot,
                    brConfig=brConfig,
                    sbrStatus=sbrStatus,
                    ethIntfStatusDir=ethIntfStatusDir,
                    topoConfig=topoConfig,
                    ethIntfStatusLocal=ethIntfStatusLocal,
                    requestCli=requestCli,
                    replyCli=replyCli,
                    topoStatus=topoStatus,
                    clock=clock,
                    requireAllValid=requireAllValid )

   pipelineManager = createPipelineManager( root.bridgeRoot.steps )
   setupPluginManager( entityManager=entityManager, cAgent=cAgent,
                       ports=root.ports, pipelineManager=pipelineManager,
                       pluginManager=root.pluginManager,
                       commonData=root.bridgeRoot.commonData,
                       packetProcessor=root.packetProcessor,
                       requireAllValid=requireAllValid )

   setupPacketParsers( root.pluginManager )

   return root

def createPipelineManager( steps ):
   assert steps is not None
   return Tac.newInstance( "Arfa::PipelinePluginManager", steps )

def setupPacketParsers( pluginManager ):
   EthType = Tac.Type( "Arnet::EthType" )
   IpProtoType = Tac.Type( "Arnet::IpProtocolNumber" )

   ethParser = Tac.newInstance( "Arfa::EthParser" )
   ethParser.typeMap[ Tac.enumValue( EthType, "ethTypeIp" ) ] = "Ipv4"
   ethParser.typeMap[ Tac.enumValue( EthType, "ethTypeIp6" ) ] = "Ipv6"
   pluginManager.packetParser.addMember( ethParser )

   ip4Parser = Tac.newInstance( "Arfa::Ip4Parser" )
   ip4Parser.typeMap[ Tac.enumValue( IpProtoType, "ipProtoUdp" ) ] = "Udp"
   pluginManager.packetParser.addMember( ip4Parser )

   ip6Parser = Tac.newInstance( "Arfa::Ip6Parser" )
   ip6Parser.typeMap[ Tac.enumValue( IpProtoType, "ipProtoUdp" ) ] = "Udp"
   pluginManager.packetParser.addMember( ip6Parser )

   udpParser = Tac.newInstance( "Arfa::UdpParser" )
   pluginManager.packetParser.addMember( udpParser )

def setupPluginManager( entityManager=None,
                        cAgent=None,
                        ports=None,
                        pipelineManager=None,
                        pluginManager=None,
                        commonData=None,
                        packetProcessor=None,
                        requireAllValid=False ):
   if requireAllValid:
      assert entityManager
      assert cAgent
      assert ports
      assert pipelineManager
      assert pluginManager
      assert commonData
      assert packetProcessor

   cEntityManager = None
   if entityManager:
      cEntityManager = entityManager.cEntityManager()
      pluginManager.brHwCapabilities = entityManager.entity(
         'bridging/hwcapabilities' )

   pluginManager.entityManager = cEntityManager
   pluginManager.pipelineManager = pipelineManager
   pluginManager.cAgent = cAgent
   pluginManager.ports = ports
   pluginManager.commonData = commonData
   pluginManager.packetProcessor = packetProcessor
