#!/usr/bin/env python
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""Python bindings for Aresolve, the asynchronous host resolver library.
"""

import collections
import socket
import weakref

import Tac
import Tracing

traceHandle = Tracing.Handle( 'AresolvePy' )
tReq = traceHandle.trace5
tResp = traceHandle.trace6
tState = traceHandle.trace7
tReact = traceHandle.trace8

# Portability: Linux specific
# EAI errors used in getaddrinfo[_a] calls; from netdb.h on Linux
EAI_SUCCESS = 0
EAI_BADFLAGS = socket.EAI_BADFLAGS
EAI_NONAME = socket.EAI_NONAME
EAI_AGAIN = socket.EAI_AGAIN  # occurs when for e.g., no nameserver configured
EAI_FAIL = socket.EAI_FAIL
EAI_NODATA = socket.EAI_NODATA
EAI_FAMILY = socket.EAI_FAMILY
EAI_SOCKTYPE = socket.EAI_SOCKTYPE
EAI_SERVICE = socket.EAI_SERVICE
EAI_ADDRFAMILY = socket.EAI_ADDRFAMILY
EAI_MEMORY = socket.EAI_MEMORY
EAI_SYSTEM = socket.EAI_SYSTEM
EAI_OVERFLOW = socket.EAI_OVERFLOW
# Not defined in socket module
EAI_INPROGRESS = -100
EAI_CANCELED = -101
EAI_NOTCANCELED = -102
EIA_ALLDONE = -103
EAI_INTR = -104
EAI_IDN_ENCODE = -105

# The possible errors noted by Aresolve.tac
EAI_ERROR_TABLE = {
   EAI_INPROGRESS: 'Processing request in progress',
   socket.EAI_NONAME: 'Name or service not known',
   socket.EAI_AGAIN: 'Temporary failure in name resolution',
   socket.EAI_NODATA: 'Non-recoverable failure in name resolution',
   EAI_CANCELED: 'Request canceled',
   socket.EAI_ADDRFAMILY: 'Address family for NAME not supported',
   socket.EAI_MEMORY: 'Memory allocation failure',
}

EAI_ERRNO_TO_SYMBOL = {
   EAI_SUCCESS: 'EAI_SUCCESS',
   socket.EAI_BADFLAGS: 'EAI_BADFLAGS',
   socket.EAI_NONAME: 'EAI_NONAME',
   socket.EAI_AGAIN: 'EAI_AGAIN',
   socket.EAI_FAIL: 'EAI_FAIL',
   socket.EAI_NODATA: 'EAI_NODATA',
   socket.EAI_FAMILY: 'EAI_FAMILY',
   socket.EAI_SOCKTYPE: 'EAI_SOCKTYPE',
   socket.EAI_SERVICE: 'EAI_SERVICE',
   socket.EAI_ADDRFAMILY: 'EAI_ADDRFAMILY',
   socket.EAI_MEMORY: 'EAI_MEMORY',
   socket.EAI_SYSTEM: 'EAI_SYSTEM',
   socket.EAI_OVERFLOW: 'EAI_OVERFLOW',
   EAI_INPROGRESS: 'EAI_INPROGRESS',
   EAI_CANCELED: 'EAI_CANCELED',
   EAI_NOTCANCELED: 'EAI_NOTCANCELED',
   EIA_ALLDONE: 'EIA_ALLDONE',
   EAI_INTR: 'EAI_INTR',
   EAI_IDN_ENCODE: 'EAI_IDN_ENCODE',
}


# DnsRecord - object passed to all Querier callbacks; attributes of this object are:
#
# name: str, the name queried.
# valid: bool, was the query successful? If False, consult lastError
# lastError: int (one of EAI_ERROR_TABLE.keys(), the most recent error for name
# lastRefresh: float, Tac::Seconds the last time this record updated
# ipAddress: list of str, zero or more IPv4 addresses for name
# ip6Address: list of str, zero or more IPv6 addresses for name
DnsRecord = collections.namedtuple(
   'DnsRecord',
   'name valid lastError lastRefresh ipAddress ip6Address' )


resolutionRecordTypename = 'Aresolve::ResolutionRecord'

class ResolutionRecordReactor( Tac.Notifiee ):
   """A TACC notifiee for DNS records (results) from Aresolve."""

   notifierTypeName = resolutionRecordTypename

   def __init__( self, aresolveRecordDir, querier, callback ):
      Tac.Notifiee.__init__( self, aresolveRecordDir )
      self.aresolveRecordDir = aresolveRecordDir
      self.callback_ = callback
      self.querier_ = querier

   @Tac.handler( 'lastError' )
   @Tac.handler( 'valid' )
   @Tac.handler( 'lastRefresh' )
   def handler( self ):
      """When attributes change, fire the callback if we have a usable result,

      Aresolve inserts entries into the output directory as it makes
      requests. Initially requests have the valid attribute set False.
      If either valid changes to True or lastError changes from its
      initial value (EAI_SUCCESS), we have a valid DNS record or an error.
      """
      n = self.notifier_
      tReact( 'react name:', repr( n.name ) )
      # Convert the U32 lastError in Sysdb to the signed EAI_* defines in netdb.h
      lastError = getSigned( n.lastError )
      error = False
      if not n.valid:
         # The record not valid, meaning in error or incomplete.
         # If we're in error and not being asked to try again, we can notify.
         if lastError:
            self.querier_.counter[ lastError ] += 1
            if lastError != EAI_AGAIN:
               error = True
      else:
         # Completed and lastError is EAI_SUCCESS
         self.querier_.counter[ lastError ] += 1
      # Notify the registered callback of any completed result.
      if n.valid or error:
         self.callback_( n )


class Querier( object ):
   """Aresolve Querier: An asynchronous DNS querier.

   This object is a thin wrapper around the C++ Aresolve library and
   manages its state machine and request and response directories.

   Querier provides a callback interface (implemented via
   ResolutionRecordReactor) and a polling interface.

   The callback interface is used by specifying a default callback
   supplied as the 'callback' initializer argument or, if you prefer,
   callbacks may be supplied for each query (see the callback argument
   to the host() method). If no per-query callback is specified, the
   default callback passed at initialization will be used.

   When full response are received, your callback is run with the DNS
   result, a DnsRecord named tuple with the same attributes as an
   Aresolve::ResolutionRecord.

   If no default callback or per query callbacks are supplied, the
   only way to receive results is to poll, with the result* methods.

   After first resolution (either successfully, or in error) queries
   will be re-run every longTime seconds and if changes in the
   response occur, your callback will be fired again.

   Use this behaviour to dynamically update configuration files or
   agent state. For example, when a temporary DNS error that happens
   during startup corrects itself, your agent can respond to the new
   DnsRecord received by its callback to start working without retry
   code in any agent.
   """

   def __init__( self, callback=None, shortTime=None, longTime=None ):
      """Creates an Aresolve (TACC DNS resolver) instance and a reactor.

      The shortTime and longTime values default to those defined in the
      Aresolve::AresolveSm implementation, generally 10 and 300 seconds,
      respectively.

      If no callback argument is provided, a callback must be supplied on
      each query (e.g., the host() method), or the polling interface must
      be used (e.g., the resultHost() method) to obtain query results.

      Args:
         callback: a callable, the default callback to be called when a DNS
            result arrives. It is given one argument, a DnsRecord namedtuple
            being the DNS result record. If no callback is provided, polling
            mode is the only way to retrieve DNS result records.
         shortTime: an int, the number of seconds between ticks of the
            Aresolve short clock. This is the retry period on errors while
            Aresolve is still attempting to resolve.
         longTime: an int, the number of seconds between ticks of the
            Aresolve long clock. This is the query refresh period. If
            the query result has changed after this period, an updated
            record will be sent to the query's callback.
      """
      self.dnsOut_ = Tac.newInstance( 'Aresolve::ResolutionRecordDir',
                                      'AresolveResolutionRecord' )
      self.dnsIn_ = Tac.newInstance( 'Aresolve::ResolutionRequestDir',
                                     'AresolveResolutionRequest' )
      self.dnsSm_ = Tac.newInstance( 'Aresolve::AresolveSm',
                                     self.dnsOut_, self.dnsIn_ )
      if shortTime:
         self.dnsSm_.shortTime = shortTime
      if longTime:
         self.dnsSm_.longTime = longTime

      self.counter = collections.Counter()
      self.callback_ = callback
      self.reactor_ = Tac.collectionChangeReactor(
         self.dnsOut_.record,
         ResolutionRecordReactor,
         reactorArgs=( weakref.proxy( self ), self._queryCallback ) )

   def host( self, name ):
      """Starts an A/AAAA DNS query for the host 'name' (str).

      Args:
        name: str, the DNS name to query.
      """
      tReq( 'request name:', repr( name ) )
      self.dnsIn_.request[ name ] = self.dnsIn_.request.get( name,  0 ) + 1

   def resultHost( self, name ):
      """Returns the current response record for host 'name'.

      Returns None if there is no current query for name, else returns a DnsRecord.

      This interface can be used by code not running in a TAC activity loop to poll
      for DNS results.
      """
      resolutionRecord = self.dnsOut_.record.get( name )
      tResp( 'result name:', name, 'record:', resolutionRecord )
      if resolutionRecord is not None:
         return self._convert( resolutionRecord )

   def _convert( self, record ):
      """Converts a Aresolve::ResolutionRecord to a DnsRecord."""
      return DnsRecord( name=record.hostname,
                        lastRefresh=record.lastRefresh,
                        lastError=getSigned( record.lastError ),
                        valid=record.valid,
                        ipAddress=record.ipAddress.keys(),
                        ip6Address=[ i.stringValue for i in record.ip6Address ] )

   def _queryCallback( self, record ):
      """Callback wrapper. Converts Aresolve::ResolutionRecord to DnsRecord."""
      tResp( 'received DNS record for name:', record.name )
      if self.callback_:
         self.callback_( self._convert( record ) )

   def finishHost( self, name ):
      """Finishes an A/AAAA DNS query for (str) name.

      Finishing a query stops it from being re-resolved by the Aresolve
      state machine.
      """
      tState( 'finish:', name )
      val = self.dnsIn_.request.get( name )
      if val is not None:
         if val:
            self.dnsIn_.request[ name ] -= 1
            # Re-set the variable again in case we've hit zero
            val = self.dnsIn_.request[ name ]
         if not val:
            tState( 'delete:', name )
            # No more referers, so kill the query.
            try:
               del self.dnsIn_.request[ name ]
            except KeyError:
               pass

   def clear( self ):
      """Stops all DNS queries created by this instance."""
      tState( "clearing all requests/responses" )
      self.dnsIn_.request.clear()
      # dnsOut_.record is cleared for us by Afetch.tin when we clear requests


def getSigned( number, numBits=32 ):
   """Gets a signed number from an unsigned number by two's complement.

   Args:
      number: int, the unsigned input value
      numBits: int, the length in bits of the input value

   Returns:
      An int, the two's complement of number for length numBits
   """
   mask = ( 2 ** numBits ) - 1
   if number & ( 1 << ( numBits - 1 ) ):
      return number | ~mask
   else:
      return number & mask


def gaiSterror( errno, signErrno=True ):
   """Emulates the C gai_sterror(3) call, returning a str for the int errno."""
   if signErrno:
      errno = getSigned( errno )
   return EAI_ERROR_TABLE.get(
      errno, EAI_ERRNO_TO_SYMBOL.get( errno, 'UNKNOWN_ERROR' ) )
