# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import
import os

import SpaceMgmtLib
import SpaceMgmtLib.Quota
import SpaceMgmtLib.Utils
import Tac

class Filesystem( object ):
   """
   A linux filesystem.

   This class provides an interface to query the filesystem state and interact
   with it.

   Attributes
   ----------
   mntPt: str
      Absolute path of the filesystem mountpoint.
   dev: Device
      Represent the filesystem device (See class Device).

   Properties
   ----------
   quotaUserFilePath

   mounted
   fsInfo
   size
   used
   usedPct
   available
   mntInfo
   mntOpts
   quotaMount

   quotasOn
   quotasOff
   repquotaInfo
   graceTimes
   blkGraceTime
   fileGraceTime
   quotaStatsPerUser
   """

   quotaMntOpts = { 'quota', 'usrquota' }

   def __init__( self, path, device=None ):
      """
      Parameters
      ----------
      path: str
         Existing absolute path. The Filesystem instance returned corresponds
         to the filesystem on which `path` resides.
      device: Debice, optional
         Instance of Device corresponding to this filesystem device. If None, a
         new instance will be created.
      """

      fsInfo = SpaceMgmtLib.Utils.fileSystemInfo( path )

      msg = 'Unable to get filesystem information for %s' % path
      assert fsInfo is not None, msg

      self.mntPt = fsInfo[ 'mntPt' ]

      if device is not None:
         self.dev = device
      else:
         # pylint: disable=cyclic-import
         from ArchiveLib import Device
         self.dev = Device.Device( fsInfo[ 'dev' ], fileSystemInfo=self )

   #################################################################################
   # Paths
   #################################################################################

   @property
   def quotaUserFilePath( self ):
      """Absoute path of the quota user file"""

      return os.path.join( self.mntPt, SpaceMgmtLib.Utils.QUOTA_USER_FILE )

   #################################################################################
   # Mount status properties
   #################################################################################

   @property
   def mounted( self ):
      """True if the filesystem is mounted, False otherwise."""

      return os.path.ismount( self.mntPt )

   @property
   def fsInfo( self ):
      """
      Filesystem information. See SpaceMgmtLib.Utils.fileSystemInfo for details.
      """

      fsInfo = SpaceMgmtLib.Utils.fileSystemInfo( self.mntPt )

      msg = 'Unable to get filesystem information for %s' % self.mntPt
      assert fsInfo is not None, msg

      msg = ( 'Filesystem for path %s changed: mounpoint was %s, now is %s'
              % ( self.mntPt, self.mntPt, fsInfo[ 'mntPt' ] ) )
      assert fsInfo[ 'mntPt' ] == self.mntPt, msg

      return fsInfo

   @property
   def size( self ):
      """Filesystem size in KiB."""

      return self.fsInfo[ 'size' ]

   @property
   def used( self ):
      """Used space in KiB."""

      return self.fsInfo[ 'used' ]

   @property
   def available( self ):
      """Available space in KiB."""

      return self.fsInfo[ 'available' ]

   @property
   def usedPct( self ):
      """Percentage of used space."""

      return self.fsInfo[ 'available' ]

   @property
   def mntInfo( self ):
      """Mountpoint information. See SpaceMgmtLib.Utils.mountInfo for details."""

      mntInfo = SpaceMgmtLib.Utils.mountInfo( mntPt=self.mntPt )

      msg = 'Unable to get mountpoint information for %s' % self.mntPt
      assert mntInfo is not None, msg

      return mntInfo

   @property
   def fsType( self ):
      """Filesystem type as reported in /proc/mounts."""

      return self.mntInfo[ 'fsType' ]

   @property
   def mntOpts( self ):
      """Set of mount options used for mounting the filesystem."""

      return self.mntInfo[ 'mntOpts' ]

   @property
   def quotaMount( self ):
      """True if the filesystem is mounted with quota options, False otherwise."""

      mntOpts = self.mntOpts
      return all( opt in mntOpts for opt in Filesystem.quotaMntOpts )

   #################################################################################
   # Quota properties
   #################################################################################

   @property
   def quotasOn( self ):
      """True if user quotas are on, False otherwise."""

      return SpaceMgmtLib.Quota.checkQuotaOn( self.mntPt )

   @property
   def quotasOff( self ):
      """True if user quotas are off, False otherwise."""

      return SpaceMgmtLib.Quota.checkQuotaOff( self.mntPt )

   @property
   def repquotaInfo( self ):
      """
      Quota statistics inluding global grace times and per user statistics.

      Returns
      -------
      tuple( graceTimes, statsPerUser )
         See SpaceMgmtLib.Quota.repquotaInfo for details.
      """

      return SpaceMgmtLib.Quota.repquotaInfo( self.mntPt )

   @property
   def graceTimes( self ):
      """
      Quotas grace times

      Returns
      -------
      tuple(blkGraceTime, fileGraceTime):
         blkGraceTime: int
            Grace time for block quotas in seconds.
         fileGraceTime: int
            Grace time for file quotas in seconds.
      """

      quotaStats = self.repquotaInfo

      return quotaStats.blkGraceTime, quotaStats.fileGraceTime

   @property
   def blkGraceTime( self ):
      """Grace time for block quotas in seconds."""

      return self.repquotaInfo.blkGraceTime

   @property
   def fileGraceTime( self ):
      """Grace time for file quotas in seconds."""

      return self.repquotaInfo.fileGraceTime

   @property
   def quotaStatsPerUser( self ):
      """
      Quota limits and usage statistics by username.

      Returns
      -------
      dict of str: dict
         Same as statsPerUser in SpaceMgmtLib.Quota.repquotaInfo.
      """

      return self.repquotaInfo.statsPerUser

   #################################################################################
   # Utils
   #################################################################################

   def pctFromSize( self, sizeKiB ):
      """Percentage of filesystem size corresponding to `sizeKiB`."""

      assert sizeKiB >= 0 and sizeKiB <= self.size, 'Invalid size: %s' % sizeKiB
      return SpaceMgmtLib.Utils.ratioToPct( sizeKiB, self.size )

   def sizeFromPct( self, pct ):
      """Space in KiB coresponding to `pct` percent of filesystem size."""

      assert SpaceMgmtLib.Utils.validPct( pct ), 'Invalid percentage.'
      return SpaceMgmtLib.Utils.pctToRatio( pct, self.size )

   #################################################################################
   # Quota control methods
   #################################################################################

   def enableQuotas( self ):
      """Set linux user quota to on."""

      SpaceMgmtLib.Quota.enableQuota( self.mntPt )

   def disableQuotas( self ):
      """Set linux user quota to off."""

      SpaceMgmtLib.Quota.disableQuota( self.mntPt )

   def setQuotaPct( self, quotaPct, userName ):
      """
      Set quota block limits to a percentage of the filesystem size.

      Parameters
      ----------
      quotaPct: int
         constaints: value in the range [0-100]
         Percentage of the filesystem size the block quota limits should be set to.
      userName: str
         Name of the user block quota limits should be set for.
      """

      SpaceMgmtLib.Quota.setQuotaPct( quotaPct,
                                      userName,
                                      self.mntPt,
                                      totalSizeKiB=self.size )

   def quotaCheck( self, createFiles=True, remountRO=False ):
      """
      Run quotacheck on the filesystem.

      Notes
      -----
      Quota needs to be turned off before running quotacheck. See man quotacheck
      for details.

      Parameters
      ----------
      createFiles: bool, optional
         If True, run quotacheck with --create-files. See man quotacheck for details.
      """

      SpaceMgmtLib.Quota.quotaCheck( self.mntPt,
                                     createFiles=createFiles,
                                     remountRO=remountRO )

   def disableGraceTimes( self ):
      """Disable quota grace times."""

      SpaceMgmtLib.Quota.setQuotaGraceTimes( self.mntPt, 0, 0 )
