# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import
import os
import re

import SpaceMgmtLib
import SpaceMgmtLib.Quota
import SpaceMgmtLib.Utils
import Tac

class Device( object ):
   """
   A linux block device with a filesystem.

   This class provides an interface to query a device state and interact with it.

   Attributes
   ----------
   name: str
      Device name.
   physical: bool
      Is the device physical or virtual.
   fs: Filesystem
      Represent the device filesystem (See class Filesystem).

   Properties
   ----------
   present
   devAndPart
   devId
   isSSD
   isFlash
   forceDisabled

   mounted
   mntPt
   mntOpts
   """

   # Marker file. If it exists, the switch's ssd should not be used.
   _noSsdUseOverrideFilePath = '/mnt/flash/no_ssd_var'

   def __init__( self, name, fileSystemInfo=None ):
      """
      Parameters
      ----------
      name: str
         Device name
      fileSystemInfo: Filesystem, optional
         Instance of Filesystem corresponding to this device filesystem. If None,
         a new instance will be instantiated.
      """

      self.name = name
      self.physical = ( name not in { 'none', 'devtmpfs',
                                      'tmpfs', 'overlay', 'sysfs' }
                        and not name.startswith( '/dev/md' )
                        and not name.startswith( '/dev/loop' ) )

      if fileSystemInfo is not None:
         self.fs = fileSystemInfo
      else:
         self.fs = None
         mntInfo = SpaceMgmtLib.Utils.mountInfo( devName=name )
         if mntInfo is not None:
            # pylint: disable=cyclic-import
            from ArchiveLib import Filesystem
            self.fs = Filesystem.Filesystem( mntInfo[ 'mntPt' ] )

   #################################################################################
   # General properties
   #################################################################################

   @property
   def present( self ):
      """True if device is present on system, False otherwise."""

      return ( not self.physical
               or os.path.exists( os.path.join( '/dev', self.name ) ) )

   @property
   def devAndPart( self ):
      """Name section and part section of the device name."""

      basename = os.path.basename( self.name )

      if basename.startswith( 'mmc' ):
         pattern = r'(?P<name>mmcblk[0-9]*)(?P<part>.*$)'
      else:
         pattern = r'(?P<name>[^0-9]+)(?P<part>[0-9]*$)'

      match = re.match( pattern, basename )

      if match is not None:
         return match.group( 'name' ), match.group( 'part' )
      return basename, ''

   @property
   def devId( self ):
      """Device id."""

      # /sys/block/<base device name>/device is a symbolic link the corresponding
      # device in /sys/devices fs. The full path in /sys/devices is formed using
      # the device id after /sys/devices/
      dev, _ = self.devAndPart
      sysDevicesPath = os.path.realpath( '/sys/block/%s/device' % dev )
      return os.path.join( *( sysDevicesPath.split( os.path.sep )[ 3 : ] ) )

   def _knownBlockDevIds( self ):
      """
      Known block devices ids from /etc/blockdev
         key: str
            Name of block devie, e.g. drive, flash, usb1, ...
         value: str
            Regex that match the device id corresponding to that device.
      """

      try:
         with open( '/etc/blockdev' ) as f:
            lines = f.readlines()
      except IOError:
         return {}

      return dict( reversed( line.split() ) for line in lines )

   def _isKnownDev( self, name ):
      """Helper method to check if it is a known device, i.e. flash or drive."""

      # If mounted, only need to check the mountpoint
      if self.mounted:
         return self.fs.mntPt == '/mnt/%s' % name

      try:
         devIdPattern = self._knownBlockDevIds()[ name ]
      except KeyError:
         return False

      _, part = self.devAndPart

      return ( re.match( devIdPattern,
                         os.path.join( self.devId, part ) ) is not None )

   @property
   def isSSD( self ):
      """True if the device is the ssd drive, False otherwise."""

      return self._isKnownDev( 'drive' )

   @property
   def isFlash( self ):
      """True if the device is the flash, False otherwise."""

      return self._isKnownDev( 'flash' )

   @property
   def forceDisabled( self ):
      """True if the device is force disabled, False otherwise."""

      return self.isSSD and os.path.exists( Device._noSsdUseOverrideFilePath )

   #################################################################################
   # Mount properties
   #################################################################################

   @property
   def mounted( self ):
      """True if the device is mounted, False otherwise."""

      return self.fs is not None and self.fs.mounted

   @property
   def mntPt( self ):
      """Mountpoint path if device is mounted, None otherwise."""

      return None if not self.mounted else self.fs.mntPt

   @property
   def mntOpts( self ):
      """Set of mount options used."""

      if not self.mounted:
         return set()
      return self.fs.mntOpts

   #################################################################################
   # Mount methods
   #################################################################################

   def _updateMntOpts( self, mntOpts, enableQuota ):
      if enableQuota is None:
         return mntOpts

      quotaOpts = { 'quota', 'usrquota' }

      if enableQuota:
         mntOpts.discard( 'noquota' )
         mntOpts.update( quotaOpts )
      else:
         for opt in quotaOpts:
            mntOpts.discard( opt )
         mntOpts.add( 'noquota' )

      return mntOpts

   def remount( self, enableQuota ):
      """
      Remount the device.

      Parameters
      ----------
      enableQuota: bool
         True if quota options should be added when remounting, False if quota
         options should be removed. None to reuse current options.
      """

      if not self.mounted:
         return

      # pylint: disable=cyclic-import
      from ArchiveLib import Filesystem
      mntOpts = self._updateMntOpts( self.mntOpts, enableQuota ).copy()
      mntOpts.add( 'remount' )

      try:
         cmd = [ 'mount', '-o', ','.join( mntOpts ), self.mntPt ]
         Tac.run( cmd, asRoot=True, stdout=Tac.DISCARD, stderr=Tac.DISCARD )
      finally:
         self.fs = Filesystem.Filesystem( self.fs.mntPt, device=self )
