# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import os
import six

OVERRIDE_ENV_VAR = 'TOGGLE_OVERRIDE'

strToBool = { 'true' : True, 'True' : True, '1': True,
              'false' : False, 'False' : False, '0': False }

def validateToggle( toggleName, toggleState ):
   '''Validates toggle name and state'''
   # No spaces allowed in toggle names
   assert ' ' not in toggleName, \
         "No space allowed in toggle name " + toggleName 

   # toggleState values are either true or false.  Assert otherwise
   assert toggleState in strToBool, \
         "Override value should be true or false for toggle " + toggleName

def createOverrideDictFromEnv():
   '''Parses the environment variable and creates a dictionary with
   toggle name as the key and state as value
   '''
   overrideDict = {}

   try:
      envStr = os.environ[ OVERRIDE_ENV_VAR ]
   except KeyError:
      # Environment variable not defined yet.  Return
      # empty dict
      return overrideDict

   # First split all the toggles
   toggles = envStr.split( ',' )

   # For each toggle get teh state and add to the dict
   for entry in toggles:
      tokens = entry.split( '=' )
      # validate toggle name and override value
      validateToggle( tokens[ 0 ], tokens[ 1 ] )

      # Add to the dictionary
      overrideDict[ tokens[ 0 ] ] = strToBool[ tokens[ 1 ] ]

   return overrideDict

def createToggleOverrideStr( toggleName, toggleState ):
   '''This function builds and returns the toggle override environment variable
   string based on the current value set in os.environ, but with toggleName set
   to toggleState'''

   try:
      # If this toggle was already overriden, we need to change the value
      # in the environment
      overrideDict = createOverrideDictFromEnv()
      overrideDict[ toggleName ] = toggleState

      # Now iterate over the dict and create the string
      newOverride = ",".join( "%s=%s" % i for i in six.iteritems( overrideDict ) )
   except KeyError:
      newOverride = '%s=%s' % ( toggleName, toggleState )
   return newOverride

def _setToggleState( toggleName, toggleState, force=False ):
   '''This function updates the environment variable and invokes
   the C binding function to update the toggles.
   '''
   # We import ArTogglesPyAgent here instead of at the top of the file to avoid
   # loading toggles too early, see BUG182156.
   import ArTogglesPyAgent
   ArTogglesPyAgent._pySetToggleState( toggleName, # pylint: disable=W0212
                                       toggleState, force )
   newOverride = createToggleOverrideStr( toggleName, toggleState )

   os.environ[ OVERRIDE_ENV_VAR ] = newOverride

def getAllToggles():
   # We import ArTogglesPyAgent here instead of at the top of the file to avoid
   # loading toggles too early, see BUG182156.
   import ArTogglesPyAgent
   return ArTogglesPyAgent.pyGetAllToggles()
