# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
from contextlib import contextmanager
import six

###
# Base Exception class
###

class ArException( Exception ):
   '''Useful base class exception, that allows adding any kwargs, and prints it in a
   nice easy to read, and abug format.

   Arguments:
      message - A string to be printed when the exception is thrown
      kwargs - Named arguments that will be printed after the message is printed

   Additionally, this class overrides the __str__ method so that the string rep of
   the exception consists of the message and the dictionary
   '''

   def __init__( self, message, **kwargs ):
      self.message = message
      self.messageDict = kwargs
      super( ArException, self ).__init__()

   def __str__( self ):
      stringRep = [ self.message ]
      for key in sorted( self.messageDict ):
         value = self.messageDict[ key ]
         try:
            stringRep.append( ( key + " : " + str( value ) ) )
         except Exception as e: # pylint: disable=broad-except
            stringRep.append( "ERROR:\n   {}:{}{}".format( key, type( e ), e ) )
      return '\n'.join( stringRep )

###
# Factory class
###

# Exceptions for the Factory class

class FactoryException( ArException ):
   pass

class ExpectedValueError( FactoryException ):
   def __init__( self, fieldName, expectedVal, realVal, **kwargs ):
      message = ( "Unexpected val in {}, expected: {}, got: {}".format( fieldName,
                                                                        expectedVal,
                                                                        realVal ) )
      super( ExpectedValueError, self ).__init__( message, **kwargs )

class ExtraMetaAttributesError( FactoryException ):
   def __init__( self, cls, attribute, **kwargs ):
      message = ( "Extra meta attributes: {} specified that is not required/used by"
                  " generated test class {}" ).format( attribute, cls.__name__ )
      super( ExtraMetaAttributesError, self ).__init__( message, **kwargs )

class IncompatibleMixinError( FactoryException ):
   def __init__( self, mixin, incompatibility, reason, **kwargs ):
      message = "Mixin: {} incompatible with: {} because {}".format(
         mixin, incompatibility, reason )
      super( IncompatibleMixinError, self ).__init__( message, **kwargs )

class MissingMetaAttributesError( FactoryException ):
   def __init__( self, cls, attributes, **kwargs ):
      message = ( "Required meta attributes: {} have not been set on the test "
                  "class : {}" ).format( attributes, cls.__name__ )
      super( MissingMetaAttributesError, self ).__init__( message, **kwargs )

# Meta types for the Factory class

class MetaAttributeError( FactoryException ):
   pass

class MetaAttribute( object ):
   '''This class is used to denote test class attributes that are designed to be
   overridden by the test generation code.

   This class offers the facility to specify a range of acceptable values that can be
   set in place of this attribute:

   Please note that pylint checking for this object is almost disabled due to the
   definition of __getattr__. This allows users to use MetaAttribute and its
   subclasses in place of any other class without pylint complaining
   '''

   options = None

   def __init__( self, options=None ):
      self.options = options

   def append( self, value ):
      '''Dummy function used to allow this class to be used in place of lists'''
      raise MetaAttributeError( "MetaAttribute append should never be called!" )

   def extend( self, value ):
      '''Dummy function used to allow this class to be used in place of lists'''
      raise MetaAttributeError( "MetaAttribute extend should never be called!" )

   def __getattr__( self, name ):
      '''This definition stops pylint from complaining that MetaAttribute* does not
      have attribute x when using in place of other classes
      '''
      self.__getattribute__( name )

class MetaAttributeRequired( MetaAttribute ):
   '''This is a helper class designed to designate meta attributes that are required
   to be set by the test generation code. If these attributes are not set at the
   creation time of the derived test class then MissingMetaAttributesError is
   raised.

   Additionally, this class allows the framework developer to restrict the value of
   meta attributes to a specific list of values.
   '''

   def __call__( self ):
      '''Dummy function required to allow this class to be used in place of
      callables.
      '''
      raise MetaAttributeError( "MetaAttributeRequired should never be called!" )

class MetaAttributeOptional( MetaAttribute ):
   '''This is a helper class designed to designate meta attributes that can be
   optionally set by the framework user at derived class generation time.

   Additionally, the user can also specify a default value for this meta attribute as
   well a list to restrict the value of the meta attributes to.
   '''

   default = None

   def __init__( self, default=None, # pylint: disable-msg=keyword-arg-before-vararg
         *args, **kwargs ):
      self.default = default
      super( MetaAttributeOptional, self ).__init__( *args, **kwargs )

      if self.options and default not in self.options:
         raise ExpectedValueError( 'default', default, self.options )

   def __call__( self ):
      return self.default

# The Factory!

class Factory( object ):
   '''A factory class to create new classes.

   This class provides a set of routines that are useful for generating classes at
   runtime.  The generateChildClass is the method that does this.  The MetaAttributes
   allow the mixins and base class to define a set of requirements for how the final
   class should be generated, and all MetaAttributes are passed as kwargs in the
   generateChildClass method.  Generated classes can then be used as a new base
   class, and this can be called recursively.

   The use case for this, and/or the original use case at Arista, is to generate a
   unique test class for parity testing on each table on the forwarding ASIC, we can
   also then add in mixins for different methods of triggering that parity error.
   For example the base class may trigger by reading the table through the control
   path, while a traffic mixin could create a UDP packet that would cause a lookup in
   that table to trigger the parity error.

   Meta Attributes:
      name ( optional ) - A simple print friendly name for the generated child class.
   '''

   # Meta Attribute
   name = MetaAttributeOptional()

   # Set in a derived class for tracing
   t0 = lambda *args, **kwargs: None
   t1 = lambda *args, **kwargs: None

   def __str__( self ):
      if self.name:
         return str( self.name )
      else:
         return self.__class__.__name__

   def __repr__( self ):
      ret = []
      mro = type( self ).mro()
      # ignore this class (1st item in MRO), and object (last item in MRO)
      mro = mro[ 1 : -1 ]

      ret.append(  'Class "{}" Info:'.format( self.__class__.__name__ ) )
      ret.append( "MRO:" )
      for cls in mro:
         ret.append( "     {}".format( cls ) )
      ret.append( "Meta:" )
      for attr in sorted( self.getMetaAttributes() ):
         val = str( getattr( self, attr ) )
         if len( val ) > 300:
            # This makes abugging hard, so strip off the end
            val = val[ 0 : 300 ] + " **** VALUE TOO LONG TO DISPLAY ****"
         ret.append( "   {}: {}".format( attr, val ) )
      return '\n'.join( ret )

   @contextmanager
   def overrideAttrs( self, trace=True, **kwargs ):
      '''Used in a with: block to override specific attributes temporarily.'''
      backup = {}
      try:
         for attrName, attrVal in six.iteritems( kwargs ):
            backup[ attrName ] = getattr( self, attrName )
            setattr( self, attrName, attrVal )
            if trace:
               self.t1( "Overriding attr {} from {} to {}".format(
                        attrName, backup[ attrName ], attrVal ) )
         yield
      finally:
         for attrName, attrVal in six.iteritems( backup ):
            setattr( self, attrName, attrVal )
            if trace:
               self.t1( "Restoring attr {} to {}".format( attrName, attrVal ) )

   @classmethod
   def getMetaAttributes( cls ):
      '''This method returns a dict of every valid MetaAttribute defined on the class
      or its bases.

      The returned dict follows the format:
         attributeName : attributeCls

      Even if meta attributes have been overwritten with actual values this function
      will discover them.

      Additionally, the returned MetaAttributes are guaranteed to follow python's
      MRO. Thus, if cls subclasses from B which subclasses from A and both these
      bases define the same meta attribute with different restrictions, B's
      attribute will be returned.
      '''
      return { attrName : attrCls for base in cls.__mro__[ ::-1 ]
                                  for attrName, attrCls
                                  in six.iteritems( base.__dict__ )
                                  if isinstance( attrCls, MetaAttribute ) }

   @classmethod
   def generateChildClass( cls, mixins=None, **kwargs ):
      '''Generates a subclass of the base class cls with the mixins applied and
      the contents of kwargs being added as attributes on the resultant object.

      Throws AleParityLibError if a key in kwargs does not matched a defined meta
      attribute on either the base class or any of the mixins.
      '''
      if mixins is None:
         mixins = []

      # If we are applying a mixin that already exists in the class hierarchy then we
      # can cause all sorts of issues with python's MRO
      if any( issubclass( cls, mixin ) for mixin in mixins ):
         raise IncompatibleMixinError( mixins, cls,
                                       "One or more mixins already exist on the "
                                       "class" )

      # For debugging purposes we need to generate a usefull class name
      generatedClassName = getattr( cls, "prettyName", cls.__name__ ) + "_"
      for mixin in mixins:
         generatedClassName += getattr( mixin, "prettyName", mixin.__name__ ) + "_"

      generatedTestClass = type( generatedClassName, tuple( mixins + [ cls ] ), {} )

      # Because this function is meant to support consecutive calls it is possible
      # that the caller wishes to override a meta attribute that has already been
      # set. Thus we have to query the python type system to check what was a meta
      # attribute and what was not.
      # It is important to pay attention to MRO because a mixin might have overridden
      # the optional meta attribute of a base to restrict its options, we need to
      # respect this ordering
      metaAttributes = generatedTestClass.getMetaAttributes()

      # Set all meta attributes from the supplied dict on the generated class
      # ensuring that each attribute has already been defined as a meta attribute
      # on the class. This enforces a certain "typeness" to the tests
      extraAttributes = []
      for key, value in six.iteritems( kwargs ):
         metaAttr = metaAttributes.get( key )
         if metaAttr is None:
            extraAttributes.append( key )
            continue

         # Here we check to ensure that if the attribute is a meta attribute then it
         # obeys any given value constraints
         if metaAttr.options and value not in metaAttr.options:
            raise ExpectedValueError( key, metaAttr.options, value )

         setattr( generatedTestClass, key, value )

      if extraAttributes:
         raise ExtraMetaAttributesError( generatedTestClass, extraAttributes,
                                         **kwargs )
      # Loop through all items in the object dictionary checking that all required
      # attributes are set and resolving defaults
      missingAttributes = []
      attributes = dir( generatedTestClass )
      for attribute in attributes:
         value = getattr( generatedTestClass, attribute )

         if isinstance( value, MetaAttributeRequired ):
            missingAttributes.append( attribute )
         elif isinstance( value, MetaAttributeOptional ):
            setattr( generatedTestClass, attribute, value() )

      if missingAttributes:
         raise MissingMetaAttributesError( generatedTestClass, missingAttributes,
                                           **kwargs )

      return generatedTestClass

###
# Singleton
###

class Singleton( type ):
   '''To make a singleton class, just add "__metaclass__ = Singleton" to any class'''
   _instances = {}

   def __call__( cls, *args, **kwargs ):
      if cls not in cls._instances:
         cls._instances[ cls ] = super( Singleton, cls ).__call__( *args, **kwargs )
      return cls._instances[ cls ]
