#!/usr/bin/env python
# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""
Generic library for source navigation related functionality.
"""

from __future__ import absolute_import, division, print_function
import os

PRUNE_FILE_NAME = '.no_source_navigation'
INCLUDE_FILE_NAME = '.source_navigation'

def defaultPruneFileName():
   """
   Name of file w/ files/dirs to be excluded from src navigation DB generation.
   Returns the one in the /src directory (if exists). Falls back to the one
   in the home directory otherwise.
   """
   workspacePruneFile = os.path.join( "/src", PRUNE_FILE_NAME )
   if os.path.exists( workspacePruneFile ):
      return workspacePruneFile
   return os.path.join( os.environ[ 'HOME' ], PRUNE_FILE_NAME )

def defaultIncludeFileName():
   """
   Name of file w/ files/dirs that are the only to be included in
   src navigation DB generation.
   Returns the one in the /src directory (if exists). Falls back to the one
   in the home directory otherwise.
   """
   workspaceIncludeFile = os.path.join( "/src", INCLUDE_FILE_NAME )
   if os.path.exists( workspaceIncludeFile ):
      return workspaceIncludeFile
   return os.path.join( os.environ[ 'HOME' ], INCLUDE_FILE_NAME )

def readPatternFile( fname ):
   """
   Read pattern file, returning list of files/dirs to be excluded/included from/in
   DB generation.

   This file, if present, contains list of files/directories to be ignored/included
   by Arista source navigation database generation tools
   (e.g., a4 mkid, taccscope -b).

   Pattern file format:

     Paths here should be *relative* to the base directory of the source
     navigation database generation. Example:

       - base directory:          /src
       - elements to be excluded: /src/Foo, /src/Bar, /src/Fii/blah.h

       => entries in this file should be: Foo, Bar, Fii/blah.h

     Lines starting with '#' are treated as comments.
     Elements can be separated by a mix of spaces, comma, or newlines.

   """
   lines = []
   try:
      with open( fname ) as f:
         lines = f.readlines()
   except IOError:
      return []
   patternList = []
   for line in lines:
      # seperators can be a mix of spaces, commas, and newlines
      line = line.strip()
      if line.startswith( '#' ):
         # ignore commented line
         continue
      patternList += line.replace( ',', ' ' ).split()
   return patternList

def writePatternFile( fname, patternList ):
   """Write a pattern file from a list."""
   with open( fname, "w+" ) as f:
      for p in patternList:
         f.write( "%s\n" % p )
