# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import copy
from abc import ABCMeta

import simplejson
import Tracing

traceHandle = Tracing.Handle( 'JsonUwsgiServer' )
log = traceHandle.trace0
warn = traceHandle.trace1
info = traceHandle.trace2
trace = traceHandle.trace3
debug = traceHandle.trace4

class BaseType( object ):
   __metaclass__ = ABCMeta

   def __init__( self, apiName, tacName, fieldType, versions, description, inputOk,
                outputOk ):
      assert apiName, "No API name given"
      assert fieldType, "No type given"
      # The name in the JSON API
      self.apiName = apiName
      # The field name used in Sysdb
      self.tacName = tacName
      # The python type of the field, i.e. int, bool, float, str
      self.fieldType = fieldType
      # A description of what the field is used for
      self.description = description
      # The API model versions that this field is used in
      self.versions = versions
      # Whether the field is mutable as input
      self.inputOk = inputOk
      # Whether the field is shown in output
      self.outputOk = outputOk
      # Set to true once this model field gets populated
      self.hasBeenSet = False
      # The actual value once it has been set
      self.value = None
      # The value type if fieldType is complex (i.e. List)
      self.valueType = None

   def checkType( self, value, apiType ):
      # Treat ints as floats
      if apiType == float and type( value ) is int:
         return

      if not type( value ) is apiType:
         msg = 'Type %s does not match API type %s' % ( type( value ),
                                                        apiType )
         warn( msg )
         raise AttributeError( msg )

   def fromSysdb( self, t ):
      # Don't auto generate types that have no tac name
      if not self.tacName:
         return

      value = getattr( t, self.tacName )
      if self.outputOk:
         # Do not assert for complex types
         if not self.valueType:
            self.checkType( value, self.fieldType )
         self.value = value
         self.hasBeenSet = True
      else:
         raise IOError( "Attribute not allowed as output" )

class Str( BaseType ):
   def __init__( self, apiName, tacName, versions, inputOk, outputOk, description ):
      super( Str, self ).__init__( apiName, tacName, str, versions, description,
                                   inputOk, outputOk )

class Int( BaseType ):
   def __init__( self, apiName, tacName, versions, inputOk, outputOk, description ):
      super( Int, self ).__init__( apiName, tacName, int, versions, description,
                                   inputOk, outputOk )

class Float( BaseType ):
   def __init__( self, apiName, tacName, versions, inputOk, outputOk, description ):
      super( Float, self ).__init__( apiName, tacName, float, versions, description,
                                   inputOk, outputOk )

class Bool( BaseType ):
   def __init__( self, apiName, tacName, versions, inputOk, outputOk, description ):
      super( Bool, self ).__init__( apiName, tacName, bool, versions, description,
                                    inputOk, outputOk )

class List( BaseType ):
   def __init__( self, apiName, tacName, versions, inputOk, outputOk, description,
                 valueType ):
      super( List, self ).__init__( apiName, tacName, list, versions, description,
                                    inputOk, outputOk )
      self.valueType = valueType
      self.value = []

   def fromSysdb( self, t ):
      super( List, self ).fromSysdb( t )

      if issubclass( self.valueType, BaseModel ):
         parsedVals = []
         # pylint: disable-msg=E1103
         for val in self.value.values():
            model = self.valueType()
            model.fromSysdb( val )
            parsedVals.append( model )
         self.value = parsedVals
      else:
         # pylint: disable-msg=E1103
         for v in self.value.keys():
            self.checkType( v, self.valueType )
         self.value = self.value.keys()

   def append( self, item ):
      if not self.value:
         self.value = [ item ]
      else:
         self.value.append( item )

class ModelJsonSerializer( simplejson.JSONEncoder ):
   """
   Serializes all subclasses of BaseModel by iterating
   through their BaseTypes and setting the apiName
   as the key and the value.
   """
   # pylint: disable-msg=E0202
   def default( self, obj ):
      if issubclass( type( obj ), BaseModel ):
         d = {}
         for field in obj.__attributes__.values():
            if field.valueType and issubclass( field.valueType, BaseModel ):
               d[ field.apiName ] = self.default( field )
            else:
               d[ field.apiName ] = field.value
         return d
      elif issubclass( type( obj ), List ):
         l = []
         for v in obj.value:
            l.append( self.default( v ) )
         return l
      else:
         simplejson.JSONEncoder.default( self, obj )

class BaseModel( object ):
   __metaclass__ = ABCMeta
   fields = []

   def __init__( self ):
      """
      Reads all the class variables and if they derive from BaseType we create
      copies in the __attributes__ dictionary and stores it in the fields list.
      When the model gets populated we look up the real value of the field there.
      """
      self.__attributes__ = {}
      for field in dir( self ):
         attr = getattr( self, field )
         if ( isinstance( type( attr ), BaseType ) or
              issubclass( type( attr ), BaseType ) ):
            self.fields.append( attr )
            self.__attributes__[ field ] = copy.copy( attr )

   def fromSysdb( self, t ):
      """
      Reads in a Tacc entity and populates all the model fields
      who's tacName matches the tacc attribute.
      """
      for field in self.__attributes__.values():
         field.fromSysdb( t )

   def getModelField( self, name ):
      """Gets the model field based on the JSON API key"""
      for field in self.__attributes__.values():
         if field.apiName == name:
            return field
      return None

   def getPopulatedModelFields( self ):
      """Returns model fields that have been assigned values."""
      return [ f for f in self.__attributes__.values() if f.hasBeenSet ]

   def setModelField( self, name, value, version=1 ):
      """Sets the model field to a given value and performs type checking."""
      field = self.getModelField( name )
      valueType = field.valueType
      if valueType and issubclass( valueType, BaseModel ):
         if type( field ) is List:
            # Lists are special case where we iterate each value in the list
            # and check its type independently.
            for item in value:
               model = valueType() # instantiate new model from the complex type
               model.populateModelFromJson( item )
               field.append( model )
         else:
            field.checkType( value, field.fieldType )
            model = valueType() # instantiate new model from the complex type
            model.populateModelFromJson( value )
            field.value = model
      else:
         field.checkType( value, field.fieldType )
         field.value = value

      field.hasBeenSet = True

   def populateModelFromJson( self, jsondata ):
      """Populates model fields form JSON input."""
      for k, v in jsondata.iteritems():
         field = self.getModelField( k )
         if not field:
            # Since the API is meant to handle unkown params we continue without
            # an error
            warn( 'Unkown attribute %s in %s' % ( k, jsondata ) )
            continue
         if field.inputOk:
            self.setModelField( k, v )
         else:
            msg = 'Tried to write non-writable attribute %s' % field.apiName
            warn( msg )
            raise AttributeError( msg )

   def toSysdb( self, entity ):
      """
      Subclasses should override this method to serialize
      the model to Sysdb. The entity parameter must be the root
      of anything that needs to be written to Sysdb.
      """
      pass

   def __dict__( self ):
      """Overloaded for easier printing of the model."""
      d = {}
      for field in self.__attributes__.values():
         d[ field.apiName ] = field.value
      return d

   def __repr__( self ):
      """Overloaded for easier printing of the model."""
      return "%s" % self.__dict__()
