# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import simplejson as json

import AgentDirectory
import PyClient
import QuickTrace
import Tac
import Tracing

t0 = Tracing.trace0
t1 = Tracing.trace1
t2 = Tracing.trace2
# Trace level 3 is reserved for unusual events, and is always enabled.
t3 = Tracing.trace3

qv = QuickTrace.Var
qt0 = QuickTrace.trace0
qt1 = QuickTrace.trace1
qt2 = QuickTrace.trace2

def dictCopy( srcDict, destDict ):
   '''
   Works only for value collections
   '''
   for key, value in srcDict.items():
      destDict[ key ] = value

class DictCmp( object ):
   '''
   Compare srcDict and destDict. Find new items added and old items
   deleted in srcDict. Return items where value changed in srcDict
   '''

   def __init__( self, srcDict, destDict ):
      self.srcDict, self.destDict = srcDict, destDict
      self.srcSet, self.destSet = set( srcDict.keys() ), set( destDict.keys() )
      self.intersect = self.srcSet.intersection( self.destSet )
      t2( "DictCmp with src ", srcDict.keys(), "\n    dst ", destDict.keys() )
      # Commenting following qt for now because it triggers a crash in Quicktrace
      # BUG227330
      #qt2( "DictCmp with src ", srcDict.keys(), "\n    dst ", destDict.keys() )

   def added( self ):
      return self.srcSet - self.intersect 

   def deleted( self ):
      return self.destSet - self.intersect 

   def modified(self):
      '''
      Applicable only for value collection.
      '''
      return set( key for key in self.intersect if 
         self.destDict[ key ] != self.srcDict[ key ])

class AlertBaseImporter( object ):
   ''' Class to handle AlertBase input 
   1. Read AlertBase from the json file
   2. Parse json and update Sysdb AlertBase 
   '''

   def __init__( self, alertBaseFile, sysname ):
      t0( "Starting AlertBaseImporter" )
      qt0( "Starting AlertBaseImporter" )
      self.alertBaseFile = alertBaseFile
      # Local copy of parsed json
      self.alertBaseLocal = None 
      self.alertBaseSysdb = None # Sysdb copy of AlertBase
      self.jsonData = {} # raw JSON data

      # Check if BugAlert is running
      if not AgentDirectory.agent( sysname, 'BugAlert' ):
         t0( "Please enable BugAlert. Exiting.." )
         qt0( "Please enable BugAlert. Exiting.." )
         return

      pc = PyClient.PyClient( sysname, "BugAlert",
                                execMode=PyClient.Rpc.execModeThreadPerConnection )
      self.alertBaseSysdb = \
                  pc.root()[ sysname ][ 'BugAlert' ][ 'BugAlert' ].alertBase

   def entityCopyAlertBase( self ):

      def entityCopyBugs():

         def versionDictCopy( verSrc, verDest ):
            verCmp = DictCmp( verSrc, verDest )

            # Copy new entries and update modified ones
            for version in ( verCmp.added().union( verCmp.modified() ) ):
               verDest[ version ] = verSrc[ version ]

            # Delete any stale entries
            for version in verCmp.deleted():
               del verDest[ version ]

         bugCmp = DictCmp( self.alertBaseLocal.bug, self.alertBaseSysdb.bug )
         # Add new bugs info
         for bugId in bugCmp.added():
            t1( "Adding new bug ", bugId, " to Sysdb" )
            qt1( "Adding new bug ", bugId, " to Sysdb" )
            bugSrc = self.alertBaseLocal.bug[ bugId ]
            bugDest = self.alertBaseSysdb.newBug( bugId )
            # Since we are not going to change the src, shallow copy
            # should be fine here
            dictCopy( bugSrc.versionIntroduced, bugDest.versionIntroduced )
            dictCopy( bugSrc.versionFixed, bugDest.versionFixed )

            # Copy BugRule conjuction
            for conjuctionId, conjuctionSrc in bugSrc.bugRuleConjunction.items():
               conjuctionDest = bugDest.newBugRuleConjunction( conjuctionId )
               dictCopy( conjuctionSrc.tag, conjuctionDest.tag )

            bugDest.alertSummary = bugSrc.alertSummary
            bugDest.alertNote = bugSrc.alertNote
            bugDest.lastBiteTime = bugSrc.lastBiteTime
            bugDest.bites = bugSrc.bites

         # Delete stale bugs info
         for bugId in bugCmp.deleted():
            t1( "Deleting bug ", bugId, " from Sysdb" )
            qt1( "Adding new bug ", bugId, " to Sysdb" )
            del self.alertBaseSysdb.bug[ bugId ]

         # Check and update modified bugs
         for bugId in bugCmp.intersect:
            t1( "Checking bug ", bugId )
            qt1( "Checking bug ", bugId )
            bugSrc = self.alertBaseLocal.bug[ bugId ]
            bugDest = self.alertBaseSysdb.bug[ bugId ]
            # Update versionIntroduced
            if bugSrc.versionIntroduced.keys() != bugDest.versionIntroduced.keys():
               t1( "  Updating versionIntroduced" )
               qt1( "  Updating versionIntroduced" )
               versionDictCopy( bugSrc.versionIntroduced, 
                  bugDest.versionIntroduced )

            # Update versionFixed
            if bugSrc.versionFixed.keys() != bugDest.versionFixed.keys():
               t1( "  Updating versionFixed" )
               qt1( "  Updating versionFixed" )
               versionDictCopy( bugSrc.versionFixed, bugDest.versionFixed )

            # Update bugRuleConjunction
            # We assume ordering here. e.g. Below code will treat 2 
            # conjuctions ( mlagEnabled or stpEnabled ) and ( stpEnabled or
            # mlagEnabled ) as different. I think we can live with that for now.
            conjuctionCmp = DictCmp( bugSrc.bugRuleConjunction, 
                  bugDest.bugRuleConjunction )
            for conjuctionId in conjuctionCmp.intersect:
               srcConjunctionTag = bugSrc.bugRuleConjunction[ conjuctionId ].tag
               destConjunctionTag = bugDest.bugRuleConjunction[ conjuctionId ].tag
               tagCmp = DictCmp( srcConjunctionTag, destConjunctionTag )

               for tag in tagCmp.added().union( tagCmp.modified() ):
                  t1( "Adding/Updating tag ", tag )
                  qt1( "Adding/Updating tag ", tag )
                  destConjunctionTag[ tag ] = srcConjunctionTag[ tag ]
               for tag in tagCmp.deleted():
                  t1( "Deleting tag ", tag )
                  qt1( "Deleting tag ", tag )
                  del destConjunctionTag[ tag ]

            for conjuctionId in conjuctionCmp.deleted():
               t1( "  Deleting conjuction ", conjuctionId )
               qt1( "  Deleting conjuction ", conjuctionId )
               del bugDest.bugRuleConjunction[ conjuctionId ]

            for conjuctionId in conjuctionCmp.added():
               t1( "  Adding conjuction ", conjuctionId )
               qt1( "  Adding conjuction ", conjuctionId )
               conjuctionDest = bugDest.newBugRuleConjunction( conjuctionId )
               dictCopy( bugSrc.bugRuleConjunction[ conjuctionId ].tag,
                     conjuctionDest.tag )

            # Update alertSummary
            if bugDest.alertSummary != bugSrc.alertSummary:
               t1( "  Updating alertSummary" )
               qt1( "  Updating alertSummary" )
               bugDest.alertSummary = bugSrc.alertSummary

            # Update alertNote
            if bugDest.alertNote != bugSrc.alertNote:
               t1( "  Updating alertNote" )
               qt1( "  Updating alertNote" )
               bugDest.alertNote = bugSrc.alertNote

            # Update lastBiteTime
            if bugDest.lastBiteTime != bugSrc.lastBiteTime:
               t1( "  Updating lastBiteTime" )
               qt1( "  Updating lastBiteTime" )
               bugDest.lastBiteTime = bugSrc.lastBiteTime

            # Update bites
            if bugDest.bites != bugSrc.bites:
               t1( "  Updating bites" )
               qt1( "  Updating bites" )
               bugDest.bites = bugSrc.bites

      def entityCopyTagRules():

         tagRuleCmp = DictCmp( self.alertBaseLocal.tagRule, 
               self.alertBaseSysdb.tagRule )

         # Create new tag rules
         for tag in tagRuleCmp.added():
            t1( "Adding new tag ", tag )
            qt1( "Adding new tag ", tag )
            tagRuleDest = self.alertBaseSysdb.newTagRule( tag )
            dictCopy( self.alertBaseLocal.tagRule[ tag ].implied,
                  tagRuleDest.implied )

         # Delete stale tag rules
         for tag in tagRuleCmp.deleted():
            t1( "Deleting tag ", tag )
            qt1( "Deleting tag ", tag )
            del self.alertBaseSysdb.tagRule[ tag ]

         # Check and update modified tag rules
         for tag in tagRuleCmp.intersect:
            t1( "Checking tag ", tag )
            qt1( "Checking tag ", tag )
            tagRuleSrc = self.alertBaseLocal.tagRule[ tag ]
            tagRuleDest = self.alertBaseSysdb.tagRule[ tag ]

            if cmp( tagRuleSrc.implied, tagRuleDest.implied ):
               impliedCmp = DictCmp( tagRuleSrc.implied, tagRuleDest.implied )
               for tag in impliedCmp.added():
                  t1( "  Adding implied tag ", tag )
                  qt1( "  Adding implied tag ", tag )
                  tagRuleDest.implied[ tag ] = True
               for tag in impliedCmp.deleted():
                  t1( "  Deleting implied tag ", tag )
                  qt1( "  Deleting implied tag ", tag )
                  del tagRuleDest.implied[ tag ]

      def entityCopyCliRules():

         cliRuleCmp = DictCmp( self.alertBaseLocal.cliRule,
               self.alertBaseSysdb.cliRule )

         # Delete stale Cli Rules
         for cliRuleId in cliRuleCmp.deleted():
            t1( "Deleting Cli Rule ", cliRuleId )
            qt1( "Deleting Cli Rule ", cliRuleId )
            del self.alertBaseSysdb.cliRule[ cliRuleId ]

         # Check and update modified Cli rules
         cliRulesToAdd = []
         for cliRuleId in cliRuleCmp.intersect:
            t1( "Checking Cli Rule ", cliRuleId )
            qt1( "Checking Cli Rule ", cliRuleId )
            cliRuleSrc = self.alertBaseLocal.cliRule[ cliRuleId ]
            cliRuleDest = self.alertBaseSysdb.cliRule[ cliRuleId ]

            # Update tag
            if cliRuleSrc.tag != cliRuleDest.tag:
               t1( "  Updating tag" )
               qt1( "  Updating tag" )
               cliRuleDest.tag = cliRuleSrc.tag
            else: 
               # command, revision and pythonExpression are constructor
               # arguments so we need to delete and add the entity if 
               # these fields are modified
               del self.alertBaseSysdb.cliRule[ cliRuleId ]
               cliRulesToAdd.append( cliRuleId )

         # Create new Cli Rules
         for cliRuleId in list( cliRuleCmp.added() ) + cliRulesToAdd:
            t1( "Adding new Cli Rule ", cliRuleId )
            qt1( "Adding new Cli Rule ", cliRuleId )
            cliRuleSrc = self.alertBaseLocal.cliRule[ cliRuleId ]
            cliRuleDest = self.alertBaseSysdb.newCliRule( cliRuleId, 
                  cliRuleSrc.showCmd, cliRuleSrc.revision, cliRuleSrc.responseFormat,
                  cliRuleSrc.pythonExpression )
            cliRuleDest.tag = cliRuleSrc.tag 

      def copyAlertBaseInfo():
         self.alertBaseSysdb.genId = self.alertBaseLocal.genId
         self.alertBaseSysdb.releaseDate = self.alertBaseLocal.releaseDate         

      t0( "Entity copy AlertBase to Sysdb" )
      qt0( "Entity copy AlertBase to Sysdb" )
      entityCopyBugs()
      entityCopyTagRules()
      entityCopyCliRules()
      copyAlertBaseInfo()

   def loadAlertBase( self ):
      t0( "loadAlertBase ", self.alertBaseFile )
      qt0( "loadAlertBase ", self.alertBaseFile )

      if os.environ.get( 'BUGALERT_BTEST' ):
         return

      with open( self.alertBaseFile ) as alertBase:
         try:
            self.jsonData = json.load( alertBase )
            alertBase.close()
         except json.JSONDecodeError as e:
            t0( "JSONDecodeError exception:", e )
            return
      
      self.alertBaseLocal = Tac.newInstance( "BugAlert::AlertBase", "alertBase" ) 
      self.parseJson()
      self.entityCopyAlertBase()

   def parseBugs( self ):
      t0( "Parsing Bugs from json" )
      qt0( "Parsing Bugs from json" )
      bugCount = 0
      for bugData in self.jsonData[ 'bugs' ]:
         bug = self.alertBaseLocal.newBug( int( bugData[ 'bugId' ] ) )

         for ver in  bugData[ 'versionIntroduced' ]:
            bug.versionIntroduced[ Tac.Type( "BugAlert::Version" )( ver )] = True 
         for ver in bugData[ 'versionFixed' ]:
            bug.versionFixed[ Tac.Type( "BugAlert::Version" )( ver ) ] = True 

         bug.alertSummary = bugData[ 'alertSummary' ] if bugData[ 'alertSummary' ] \
               else ""
         bug.alertNote = bugData[ 'alertNote' ]
         bug.lastBiteTime = bugData[ 'lastBiteTime' ]
         bug.bites = int( bugData[ 'bites' ] )

         if not bugData.get( 'conjunction'):
            continue

         conjuctionId = 0
         negation = Tac.Type( "BugAlert::Negation" ) 
         for bugRule in bugData[ 'conjunction' ]:
            conjunction = bug.newBugRuleConjunction( conjuctionId )
            for clause in bugRule:
               conjunction.tag[ clause[ 'tag' ] ] = negation.negative \
                   if clause.get( 'negation' ) else negation.positive 
            conjuctionId += 1
         bugCount += 1
      t0( "Parsed ", bugCount, " bugs from AlertBase" )
      qt0( "Parsed ", bugCount, " bugs from AlertBase" )

   def parseCliRules( self ):
      t0( "Parsing Cli Rules from json" )
      qt0( "Parsing Cli Rules from json" )
      cliRuleCount = 0
      for ruleData in self.jsonData[ 'cliRule' ]:
         rule = self.alertBaseLocal.newCliRule( 
            ruleData[ 'id' ], ruleData[ 'command' ],
            ruleData[ 'revision' ], ruleData[ 'responseFormat' ],
            ruleData[ 'expression' ] )
         rule.tag = ruleData[ 'tag' ]
         cliRuleCount += 1
      t0( "Parsed ", cliRuleCount, " cli rules from AlertBase" )
      qt0( "Parsed ", cliRuleCount, " cli rules from AlertBase" )

   def parseTagImplications( self ):
      t0( "Parsing tag implications from json" )
      qt0( "Parsing tag implications from json" )
      tagImplicationCount = 0
      for tagData in self.jsonData[ 'tagImplication' ]:
         tagRule = self.alertBaseLocal.newTagRule( tagData[ 0 ] )
         tagRule.implied[ tagData[ 1 ] ] = True
         tagImplicationCount += 1
      t0( "Parsed ", tagImplicationCount, " tag implications from AlertBase" )

   def parseReleaseGraph( self ):
      return

   def parseAlertBaseInfo( self ):
      self.alertBaseLocal.releaseDate = self.jsonData[ 'releaseDate' ]
      self.alertBaseLocal.genId = self.jsonData[ 'genId' ]

   def parseJson( self ):
      self.parseAlertBaseInfo()
      self.parseBugs( )
      self.parseCliRules( )
      self.parseTagImplications( )
      self.parseReleaseGraph( ) #the release graph should be a list of tuples ?

