# Copyright (c) 2006-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import Tac, Tracing
import LauncherLib
import LauncherUtil
from StageMgr import defaultStageInstance
from enum import Enum
import StageGraphs

__defaultTraceHandle__ = Tracing.Handle( 'LauncherRunnabilityEval' )

t0 = Tracing.trace0
t3 = Tracing.trace3
t8 = Tracing.trace8

"""
This module registers reactors for ProgressDir and CompletionStatusDir.
As StageMgr publishes new stages, ProgressSm in this module reacts and
evaluates runnability of Agents participating in published stage. If 
runnability condition has not been met, event is marked as complete.
"""
activeSupervisorRoleName = LauncherUtil.activeSupervisorRoleName
allSupervisorsRoleName = LauncherUtil.allSupervisorsRoleName
allCellsRoleName = LauncherUtil.allCellsRoleName

class CompletionStatusDirSm( Tac.Notifiee ):
   notifierTypeName = "Stage::CompletionStatusDir"

   def __init__( self, entMan, stageClass, completionStatusDir ):
      self.entMan_ = entMan
      self.stageClass_ = stageClass
      self.completionStatusDir_ = completionStatusDir
      self.completionStatusDirSm_ = {}
      Tac.Notifiee.__init__( self, completionStatusDir )

      for key in self.completionStatusDir_:
         self.handleCompletionStatusDir( key )

   class CompletionStatusSm( Tac.Notifiee ):
      notifierTypeName = "Stage::CompletionStatus"

      def __init__( self, entMan, completionStatus ):
         self.entMan_ = entMan
         self.completionStatus_ = completionStatus
         self.agentCfgs_ = entMan.launcherSysdbState.agentConfigDir.agent
         self.agentSts_ = entMan.launcherSysdbState.agentStatusDir.agent
         self.launchCandidateDir_ = entMan.launcherSysdbState.launchCandidateDir
         Tac.Notifiee.__init__( self, completionStatus )

         self.handleCompletionStatus()

      @Tac.handler( "complete" )
      def handleCompletionStatus( self ):
         t0( "handleCompletionStatus" )
         if not self.completionStatus_.complete:
            return

         t0( "stage progress completed" )
         if self.entMan_.launcherSysdbState.bootComplete:
            return

         self.entMan_.launcherSysdbState.bootComplete = True
         if self.entMan_.launcherSysdbState.redModReactor:
            self.entMan_.launcherSysdbState.redModReactor.maybeCreateCliReactor()

         for agentStatusEntity in self.agentSts_.itervalues():
            agentConfigEntity = self.agentCfgs_[ agentStatusEntity.agentName ]
            LauncherUtil.maybeUpdateSupeRoleAgentConfig(
               self.entMan_.redundancyStatus(), self.launchCandidateDir_,
               agentConfigEntity, agentStatusEntity )

   @Tac.handler( "status" )
   def handleCompletionStatusDir( self, key ):
      t0( "handleCompletionStatusDir Key: %s " % key )
      if key is None:
         for k in self.completionStatusDir_.keys():
            self.handleCompletionStatusDir( k )
         return
      if self.stageClass_ == 'switchover' or \
         self.stageClass_ == 'boot':
         assert key == defaultStageInstance

      if key in self.completionStatusDir_:
         self.completionStatusDirSm_[ key ] = \
               self.CompletionStatusSm( self.entMan_,
                                        self.completionStatusDir_.status[ key ] )
      else:
         del self.completionStatusDirSm_[ key ]

class ProgressDirReactor( Tac.Notifiee ):
   notifierTypeName = 'Stage::ProgressDir'

   def __init__( self, entMan, stageClass, progressDir, config, fatalError, 
                       agentStatus, completionStatus ):
      self.entMan_ = entMan
      self.stageClass_ = stageClass
      self.progressDir_ = progressDir
      self.config_ = config
      self.fatalError_ = fatalError
      self.agentStatus_ = agentStatus
      self.completionStatus_ = completionStatus
      self.progressSm_ = {}
      Tac.Notifiee.__init__( self, progressDir )

      for key in self.progressDir_:
         self.handleProgressDir( key )

   class ProgressSm( Tac.Notifiee ):
      notifierTypeName = 'Stage::Progress'

      def __init__( self, entMan, instance, stageClass, 
                          progress, config, fatalError, agentStatus, 
                          completionStatus ):
         self.entMan_ = entMan
         self.instance_ = instance
         self.stageClass_ = stageClass
         self.progress_ = progress
         self.config_ = config
         self.fatalError_ = fatalError
         self.agentStatus_ = agentStatus
         self.completionStatus_ = completionStatus
         self.agentCfgs_ = entMan.launcherSysdbState.agentConfigDir.agent
         self.agentSts_ = entMan.launcherSysdbState.agentStatusDir.agent
         self.agentShutdownCfgs = \
                        self.entMan_.launcherSysdbState.agentShutdownConfigDir.agent
         # pylint: disable=W0212 
         self.bootMode_ = StageGraphs.BootModeConfig( entMan )._determineBootMode()
         self.redStatus_ = entMan.redundancyStatus()
         self.launchCandidateDir_ = entMan.launcherSysdbState.launchCandidateDir
         Tac.Notifiee.__init__( self, progress )

         for key in self.progress_.stage:
            self.handleProgress( key )

      def doStageComplete( self, agent, stage ):
         agentStatusKey = Tac.newInstance( "Stage::AgentStatusKey", 
                              agent, stage, self.instance_ )
         self.agentStatus_.complete[ agentStatusKey ] = True

      @Tac.handler( "stage" )
      def handleProgress( self, key ):
         t0( "handleProgress" )
         if self.instance_ not in self.completionStatus_.status:
            # There are bunch of btests that are testing stage progression and don't
            # want ProgressDir reactor to interfere with testing. Rather than setting
            # SYSDB_LOADS_RUNNABILITY_EVALUATOR to 0 for each individual test, 
            # returning  without assert is ok.
            if os.environ.get( 'SYSDB_LOADS_RUNNABILITY_EVALUATOR' ) == '1':
               t0( "ERROR: Stage progression without instance of completionstatus" )
               return
            else:
               assert False, \
                    "ERROR: Stage progression without instance of completionstatus" 
         if self.completionStatus_.status[ self.instance_ ].complete:
            return

         assert key
         agents = set()
         stage = key

         # Get list of agents that are participating in given stage
         for agent in self.config_.keys():
            if stage in self.config_[ agent ].event:
               agents.update( [ agent ] )
         for a in agents:
            if a in self.agentCfgs_:
               agentConfig = self.agentCfgs_[ a ]
               agentStatus = self.agentSts_[ a ]
               t3( 'ProgressDirReactor : try update', a )
               LauncherUtil.maybeUpdateSupeRoleAgentConfig(
                  self.redStatus_, self.launchCandidateDir_,
                  agentConfig, agentStatus )
               # If the agent is not runnable at the time the stage is started, 
               # then we  want to autocomplete the stage.
               runnable = False
               if LauncherUtil.agentConfigIsSane( agentConfig ):
                  t3( 'ProgressDirReactor agentConfigIsSane' )
                  if LauncherUtil.agentIsRunnable(
                     agentConfig,
                     self.entMan_.launcherSysdbState.agentShutdownConfigDir,
                     agentStatus.roleUsed, None, self.entMan_ ):
                     t3( 'ProgressDirReactor agentIsRunnable' )
                     runnable = True

               t3( 'ProgressDirReactor runnable:', runnable )
               if not runnable:
                  # If the agent is in shutdown then complete the stage for this
                  # agent.
                  if LauncherUtil.agentIsShutdown(
                        self.agentShutdownCfgs.get( a ),
                        a, self.entMan_.redundancyStatus().mode, None ):
                     t3( 'Stage complete in shutdown', a )
                     self.doStageComplete( a, stage )

                  # For switchover and cold boot,
                  # check if completeNotRunnable is enabled
                  if self.stageClass_ == 'switchover' or \
                     ( self.bootMode_ == 'supervisorColdBoot' and \
                       self.stageClass_ == 'boot' ) or \
                     self.stageClass_ == 'shutdown':
                     if self.config_[ a ].event[ stage ].completeNotRunnable:
                        self.doStageComplete( a, stage )
                        t3( 'ProgressDirReactor completeNotRunnable', a )
                     elif not self.config_[ a ].event[ stage ].complete:
                        t3( 'ProgressDirReactor require completeNotRunnable', a )
                  else:
                     self.doStageComplete( a, stage )
            elif self.config_[ a ].event[ stage ].completeNotRunnable:
               # This agent should be platform dependent and currently is not
               # installed on the system. We need to complete this stage for
               # the agent.
               t3( 'ProgressDirReactor completeNotRunnable', a )
               self.doStageComplete( a, stage )

         # If the stage "PriorityBootComplete" is encountered, then launcher
         # will create all the role config to allow all agents to be launched.
         # If this stage is not encountered, then the role config for the agents
         # is created after stage progression is completed.
         if key == "PriorityBootComplete":
            for agentStatusEntity in self.agentSts_.itervalues():
               agentConfigEntity = self.agentCfgs_[ agentStatusEntity.agentName ]
               LauncherUtil.maybeUpdateSupeRoleAgentConfig( self.redStatus_,
                                                            self.launchCandidateDir_,
                                                            agentConfigEntity,
                                                            agentStatusEntity )

         # Check that there is no core files otherwise we need to request a fatal
         # Error. This should only be checked before boot stages are complete.
         if( key == "BootSanityCheck" and
             self.stageClass_ == "boot" and
             self.completionStatus_.status[ self.instance_ ].complete == False ):
            coreDir = os.environ.get( "COREDIR", "/var/core" )
            coreFiles = os.listdir( coreDir )
            if coreFiles:
               t3( "Detected cores. Requesting FatalError" )
               t3( "CoreDir %s contains these files: %s" % ( coreDir, coreFiles ) )
               coreFilesFullPath = [ os.path.join( coreDir, f ) for f in coreFiles ]
               coreFilesFullPath.sort( key=os.path.getmtime )
               coreFileLatest = os.path.basename( coreFilesFullPath[ - 1 ] )
               self.fatalError_.rebootReason = ( "BootSanityCheck detected cores "
                     "(Total: %d Latest: %s)" % ( len( coreFiles ), coreFileLatest )
                     )
               rcConstants = Tac.Value( "ReloadCause::ReloadCauseConstants" )
               self.fatalError_.reloadCauseDesc = rcConstants.fatalErrorDesc
               fatalErrorResetMode = Tac.Type( "Stage::FatalError::ResetMode" )
               self.fatalError_.requestReboot = fatalErrorResetMode.resetLocal
            else:
               # mark the event complete
               agentCfg = self.config_.get( "Launcher" )
               if agentCfg:
                  eventCfg = agentCfg.event.get( "BootSanityCheck" )
                  if eventCfg:
                     self.doStageComplete( "Launcher", eventCfg.name )

   @Tac.handler( "progress" )
   def handleProgressDir( self, key ):
      t0("handleProgressDir  Key: %s" % key)
      if key is None:
         for k in self.progressDir_.keys():
            self.handleProgressDir( k )
         return
      if self.stageClass_ == 'switchover' or \
         self.stageClass_ == 'boot':
         assert key == defaultStageInstance

      if key in self.progressDir_:
         self.progressSm_[ key ] = \
               self.ProgressSm( self.entMan_, key, self.stageClass_, 
                                self.progressDir_.progress[ key ], 
                                self.config_, self.fatalError_, 
                                self.agentStatus_, self.completionStatus_ )
      else:
         del self.progressSm_[ key ]

class stageClassEnum( Enum ):
   boot = 'boot'
   shutdown = 'shutdown'
   switchover = 'switchover'
   maintEnter = 'maintEnter'
   maintExit = 'maintExit'

class AgentsRunnabilityEvaluator( Tac.Notifiee ):
   notifierTypeName = 'Redundancy::RedundancyStatus'

   def __init__( self, stageCompletionstatus, stageProgress, stageInputDir,
                   stageAgentStatus, fatalError, entMan, redStatus ):
      self.stageCompletionstatus_ =  stageCompletionstatus
      self.stageProgress_ = stageProgress
      self.stageInputDir_ = stageInputDir
      self.stageAgentStatus_ = stageAgentStatus
      self.fatalError_ = fatalError
      self.entMan_ = entMan
      self.redStatus_ = redStatus
      self.progressSms_ = set()
      self.completionStatusSms_ = set()
      Tac.Notifiee.__init__( self, self.redStatus_ )

      self.handleMode()

   @Tac.handler( 'mode' )
   def handleMode( self ):
      t8( 'handleMode: %s' % self.redStatus_.mode )

      # AgentsRunnabilityEvaluator shouldn't perform any stage class
      # logic on sso standby
      if LauncherLib.isSsoStandby( self.notifier_ ):
         self.clearStageClassReactors()
      else:
         self.createStageClassReactors()

   def createStageClassReactors( self ):
      if len( self.completionStatusSms_ ) > 0:
         # We already have our reactors created, don't overwrite them.
         return

      for stageClass in self.stageProgress_.keys():
         t0( "Creating reactors for stage class %s" % stageClass )
         self.progressSms_.add(
            ProgressDirReactor( self.entMan_, stageClass,
                                self.stageProgress_[ stageClass ],
                                self.stageInputDir_[ stageClass ],  self.fatalError_,
                                self.stageAgentStatus_[ stageClass ],
                                self.stageCompletionstatus_[ stageClass ] ) )
         self.completionStatusSms_.add(
            CompletionStatusDirSm( self.entMan_, stageClass,
                                   self.stageCompletionstatus_[ stageClass ] ) )

   def clearStageClassReactors( self ):
      self.progressSms_.clear()
      self.completionStatusSms_.clear()
