#!/usr/bin/env python
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# This module gathers the agent dump from Rib/IGP agents over stream sockets
# and is run within the context of a VRF namespace by connecting
# to the Rib/IGP  agent in that network namespace

import socket, errno
import sys
import RibAgent
import IsisAgt
import OspfAgt
import Ospf3Agt
import RipAgt

AGENT_DUMP_RECV_BUF = 4096

def doShowAgentDump( agentName, debug=False, ribout=False, verbose=False ):
   if ( agentName == RibAgent.name() ):
      if debug or ribout:
         AGENT_DUMP_PORT = RibAgent.RIB_DUMP_PORT_DEBUG
      else:
         AGENT_DUMP_PORT = RibAgent.RIB_DUMP_PORT
   elif ( agentName == IsisAgt.name() ):
      # Ignore the debug argument even if it was passed in as true. This is
      # used for 'rib consistency checker' which doesn't make sense in
      # multi-agent mode
      debug = False
      ribout = False
      AGENT_DUMP_PORT = IsisAgt.ISIS_DUMP_PORT
   elif ( agentName == OspfAgt.name() ):
      # Ignore the debug and ribout arguments even if passed in as true. This is
      # used for 'rib/ribout consistency checker' which doesn't make sense in
      # multi-agent mode
      debug = False
      ribout = False
      AGENT_DUMP_PORT = OspfAgt.OSPF_DUMP_PORT
   elif ( agentName == Ospf3Agt.name() ):
      # Ignore the debug and ribout arguments even if passed in as true. This is
      # used for 'rib/ribout consistency checker' which doesn't make sense in
      # multi-agent mode
      debug = False
      ribout = False
      AGENT_DUMP_PORT = Ospf3Agt.OSPF3_DUMP_PORT
   elif ( agentName == RipAgt.name() ):
      # Ignore the debug and ribout arguments even if passed in as true. This is
      # used for 'rib/ribout consistency checker' which doesn't make sense in
      # multi-agent mode
      debug = False
      ribout = False
      AGENT_DUMP_PORT = RipAgt.RIP_DUMP_PORT

   sockFd = None
   try:
      sockFd = socket.socket()
   except socket.error as e:
      exit( 1 )

   try:
      # a blocking connect socket with low time-out
      sockFd.settimeout( 1.0 )
      sockFd.connect( ( 'localhost', AGENT_DUMP_PORT ) )
      command_str = ''
      if debug or ribout:
         if debug:
            command_str = 'debug '
         if ribout:
            command_str += 'ribout '
         if verbose:
            command_str += 'verbose '
         if command_str:
            sockFd.sendall( command_str.strip() + "\n" )
         else:
            sockFd.sendall( "quiet\n" )
   except socket.error as e:
      sockFd.close()
      exit( 2 )

   sockFd.settimeout( 5.0 )
   while True:
      try:
         data = sockFd.recv( AGENT_DUMP_RECV_BUF )
         if not data:
            # reached EOF, all data has been received
            sys.stdout.flush()
            break
         sys.stdout.write( data )
      except socket.error as e:
         if e.errno == errno.ETIMEDOUT:
            # we encountered an err.ETIMEDOUT
            # not all data from the sender could be received in time
            sockFd.close()
            exit( 3 )
      except IOError as e:
         # If someone pipes the output of command closes it without reading the
         # whole output, it's fine
         if e.errno == errno.EPIPE:
            sockFd.close()
            exit(0)
   sockFd.close()
   exit( 0 )
