#!/usr/bin/env python
# Copyright (c) 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Tracing
import Cell
import TacSigint
import os
import socket
import sys
import select
import PyClient
import threading

t0 = Tracing.trace0

threadLock = threading.Lock()
requestNum = 1

def getRequestNum():
   global requestNum
   threadLock.acquire()
   rNo = requestNum
   requestNum += 1
   threadLock.release()
   return rNo

def _configPath( dirName ):
   return Cell.path( "agent/commandRequest/config/" ) + dirName

class RunSocketCommandException( Exception ):
   pass

# Create a socket to be used by processes to communicate. The most typical usage
# usage for this is the execution of CLI commands. The sequence of events in this
# case is the following:
# - the cli creates the socket, specifying the command to run
# - the platform agent reacts to the socket creation and run the desired command
# - the platform agent send the command output to the socket
# - the cli process receives the output and prints it, then closes the socket
def createSocket( pyClientOrEm, sysname, dirName, socketName,
                  command, commandType,
                  keepalive, asyncCommand, timeout, outputFormat, revision=1 ):

   socketFile = "/tmp/" + socketName

   # Make sure the socket does not already exist
   try:
      os.unlink( socketFile )
   except OSError:
      if os.path.exists( socketFile ):
         t0( "Error: existing socket" )
         return None

   # Create the socket and bind to the name
   t0( "Initializing socket" )
   try:
      sock = socket.socket( socket.AF_UNIX, socket.SOCK_STREAM )
   # pylint: disable = W0702
   except:
      e = sys.exc_info()[ 0 ]
      t0( "Error on socket creation : %s" % e )
      return None

   # pylint: enable = W0702
   sock.bind( socketFile )
   sock.settimeout( timeout )
   sock.listen( 1 )

   # Add socket to the agentCommandRequest directory. This directory must be
   # mounted by the agent using the cli. It is not created by Sysdb.

   t0( "Creating sysdb entity" )

   assert isinstance( revision, int )
   assert revision > 0
   if isinstance( pyClientOrEm, PyClient.PyClient ):
      cmd = """import Tac
r = Tac.root.entity[ '%s/Sysdb/%s' ].newEntity(
      'Agent::AgentCommandRequest', %r)
r.commandType = %r
r.commandString = %r
r.keepalive = %r
r.asyncCommand = %r
r.outputFormat = %r
r.revision = %r
r.initialized = True""" % ( sysname,
                            _configPath( dirName ),
                            socketName,
                            commandType,
                            command,
                            keepalive,
                            asyncCommand,
                            "of" + outputFormat.title(),
                            int( revision ) )
      pyClientOrEm.execute( cmd )
   else:
      agentRoot = pyClientOrEm.root().entity[ _configPath( dirName ) ]
      request = agentRoot.newEntity( 'Agent::AgentCommandRequest', socketName )
      request.commandType = commandType
      request.commandString = command
      request.keepalive = keepalive
      request.asyncCommand = asyncCommand
      request.revision = revision
      # used by cli show commands with a capi model based on 'deferredModel'
      request.outputFormat = "of" + outputFormat.title()
      # This must be the last attribute to change as the agentCommandRequest
      # react to it
      request.initialized = True

   # Listen for incoming connections
   t0( "Waiting for connection" )

   try:
      ( connection, client_address ) = sock.accept()
      t0( "Received connection from %s" % client_address )
   except socket.timeout:
      t0( "Error: connection timeout" )
      deleteSocket( pyClientOrEm, sysname, dirName, socketName )
      return None

   t0( "Connection received" )
   return connection

def deleteSocket( pyClientOrEm, sysname, dirName, socketName ):

   t0( "Deleting socket" )
   socketFile = "/tmp/" + socketName
   os.unlink( socketFile )

   t0( "Deleting sysdb entity" )
   if isinstance( pyClientOrEm, PyClient.PyClient ):
      cmd = """import Tac
Tac.root.entity[ '%s/Sysdb/%s' ].deleteEntity( '%s' )""" % ( sysname,
                                                          _configPath( dirName ),
                                                          socketName )
      pyClientOrEm.execute( cmd )
   else:
      agentRoot = pyClientOrEm.root().entity[ _configPath( dirName ) ]
      agentRoot.deleteEntity( socketName )

# Same as runSocketCommand, but sets up the environment for 'cliprint', a combined
# ways of printing text or json, in a steamed fashion (no intermediate model).
# The print APIs (CliPrint.h) used by the handler function, running in either Cli or
# Agent context, will pick the relevant args from the env (unbeknownst/transparent
# to the API user.
def runCliPrintSocketCommand( entityManager, dirName, commandType, command, mode,
                              keepalive=False, asyncCommand=False, timeout=120,
                              connErrMsg=None, stringBuff=None,
                              forceOutputFormat=None ):

   outputFormat = forceOutputFormat if (
         forceOutputFormat ) else mode.session_.outputFormat_
   runSocketCommand( entityManager, dirName, commandType, command,
                     keepalive=keepalive, asyncCommand=asyncCommand, timeout=timeout,
                     stringBuff=stringBuff, outputFormat=outputFormat,
                     connErrMsg=connErrMsg,
                     revision=mode.session_.requestedModelRevision() )

def _handleOutput( chunks, stringBuff, errorResponses,
                   throwException, error=False ):
   if stringBuff:
      map( stringBuff.write, chunks )
      if throwException:
         response = stringBuff.getvalue()
         for er in errorResponses:
            if er in response:
               error = True
               break
         if error:
            return ( True, response )
   elif error:
      if throwException:
         return ( True, "".join( chunks ) )
      map( sys.stderr.write, chunks )
      sys.stderr.flush()
   else:
      if throwException:
         response = "".join( chunks )
         for er in errorResponses:
            if er in response:
               return ( True, response )
      map( sys.stdout.write, chunks )
      sys.stdout.flush()
   return ( False, None )

def _handleConnection( connection, chunks, stringBuff, errorResponses,
                       throwException ):
   data = connection.recv( 4096 )
   error, message = False, None
   if data:
      if stringBuff is not None or throwException:
         chunks.append( data )
      else:
         _handleOutput( [ data ], stringBuff, errorResponses, throwException )
   else:
      t0( "Receive completed" )
      if stringBuff is not None or throwException:
         error, message = _handleOutput( chunks, stringBuff, errorResponses,
                                         throwException )
      return True, error, message
   return False, None, None

def _cohabLoop( connection, chunks, stringBuff, errorResponses, throwException,
                  timeout ):
   startTime = Tac.now()
   error, message = False, None
   while True:
      # small timeout value to run activities if select gets blocked
      ready = select.select( [ connection ], [], [], 0.1 )
      if ready[ 0 ]:
         msgComplete, error, message = _handleConnection( connection, chunks,
               stringBuff, errorResponses, throwException )
         if msgComplete:
            return error, message
      else:
         if Tac.now() - startTime < timeout:
            # These mini timeouts are only to flush activity loop
            Tac.runActivities( 0 )
            continue
         chunks.append( "% Cli connection timeout\n" )
         error, message = _handleOutput( chunks, stringBuff, errorResponses,
                                         throwException, error=True )
         return error, message

def _nonCohabLoop( connection, chunks, stringBuff, errorResponses, throwException,
                  timeout ):
   while True:
      # Check if we have been interrupted before blocking.
      # Note we'd raise select.error with EINTR or KeyboardInterrupt if we are
      # interrupted by a signal in select(), so no need to check afterwards.
      TacSigint.check()
      ready = select.select( [ connection ], [], [], timeout )
      if ready[ 0 ]:
         msgComplete, error, message = _handleConnection( connection, chunks,
               stringBuff, errorResponses, throwException )
         if msgComplete:
            return error, message
      else:
         chunks.append( "% Cli connection timeout\n" )
         error, message = _handleOutput( chunks, stringBuff, errorResponses,
                                         throwException, error=True )
         return error, message

def _dirExists( pyClientOrEm, sysname, dirName ):
   if isinstance( pyClientOrEm, PyClient.PyClient ):
      errStr = "Error: no Tac::Dir"
      dirCmd = """import Tac
if Tac.root.entity.get( '%s/Sysdb/%s' ) is None:
   print \"%s\" """ % ( sysname, _configPath( dirName ), errStr )

      output = pyClientOrEm.execute( dirCmd )
      if errStr in output:
         t0( "%s does not exist" % _configPath( dirName ) )
         return False
   elif pyClientOrEm.root().entity.get( _configPath( dirName ) ) is None:
      t0( "%s does not exist" % _configPath( dirName ) )
      return False
   return True

def runSocketCommand( entityManager, dirName, commandType, command, keepalive=False,
                      asyncCommand=False, timeout=120, stringBuff=None,
                      outputFormat="unset", throwException=False, errors=None,
                      connErrMsg=None, revision=1 ):
   """Wrapper around createSocket/deleteSocket. It is mainly used by the Cli.
   It creates a socket and a corresponding entity of type AgentCommandRequest
   in sysdb (the socket name and the sysdb directory are passed as parameter).
   The platform agent has to create the sysdb directory, and it reacts to the
   entity creation through a AgentCommandRequestDir state machine. This state
   machine will call a callback on the agent to run the command and send the
   output via the socket.
   RunSocketCommand also prints the output received and deletes the socket once
   the command has completed.
   If stringBuff (a file-like object) is specified, all output will be sent to it
   rather than to stdout/stderr.
   if throwException is True then it can raise RunSocketCommandException as well,
   also caller can specify errors string or list of error message agent can spit.
   *** NOTE: Maximum response size supported is 1Mb. Please refer to Bug113060
   """

   # Used only for breadth test verification
   if 'AGENT_COMMAND_REQUEST_DRY_RUN' in os.environ:
      print "Dir: %s, Type: %s, Command: %s" % \
         ( dirName, commandType, command )
      return

   assert throwException or errors is None, \
         "throwException can't be False when errors is not None"

   errorResponses = []
   if errors is not None:
      if type( errors ) != list:
         errorResponses.append( errors )
      else:
         errorResponses = errors

   sysdbSafeCommandType = commandType.replace( '/', '_' )
   socketName = sysdbSafeCommandType + '-' + str( os.getpid() ) + '-' + \
       str( getRequestNum() )

   if entityManager.local():
      t0( "Running in cohab mode" )
      # local entity manager avoids pyClient for cohabiting tests
      pc = entityManager
   else:
      pc = PyClient.PyClient( entityManager.sysname(), "Sysdb" )

   # Make sure AgentCommandRequest directory exists
   if not _dirExists( pc, entityManager.sysname(), dirName ):
      errMsg = "% Error: Agent has not been started\n"
      error, message = _handleOutput( [ errMsg ], stringBuff, errorResponses,
                                      throwException, error=True )
      if error:
         raise RunSocketCommandException( message )
      return
    
   connection = createSocket( pc, entityManager.sysname(), dirName, socketName,
                              command, commandType, keepalive=keepalive,
                              asyncCommand=asyncCommand, timeout=timeout,
                              outputFormat=outputFormat, revision=revision )

   error, message = False, None
   if not connection:
      if outputFormat == 'json':
         connErrMsg = '{"errors": ["%s"]}' % ( connErrMsg or "Cli connection error" )
      else:
         connErrMsg = "%% %s\n" % ( connErrMsg or "Cli connection error" )
      error, message = _handleOutput( [ connErrMsg ], stringBuff, errorResponses,
                                      throwException, error=True )
      if error:
         raise RunSocketCommandException( message )
      return

   t0( "Receiving output" )
   chunks = []
   with TacSigint.immediateMode():
      try:
         if entityManager.local():
            error, message = _cohabLoop( connection, chunks, stringBuff,
                                        errorResponses, throwException, timeout )
         else:
            error, message = _nonCohabLoop( connection, chunks, stringBuff,
                                           errorResponses, throwException, timeout )
      except KeyboardInterrupt:
         raise
      except Exception, e: # pylint: disable=W0703
         t0( "Failed to read: %s" % str( e ) )
         if outputFormat == 'json':
            errorMsg = '{"errors": ["%s"]}' % ( connErrMsg or
                                                "Cli connection exception" )
            chunks.append( errorMsg )
         else:
            chunks.append( "% Cli connection exception\n" )
         error, message = _handleOutput( chunks, stringBuff, errorResponses,
                                         throwException, error=True )
         if error:
            raise RunSocketCommandException( message )
      else:
         if error:
            raise RunSocketCommandException( message )
      finally:
         t0( "Closing connection" )
         connection.close()

         t0( "Deleting socket" )
         deleteSocket( pc, entityManager.sysname(), dirName, socketName )
