# Copyright (c) 2006-2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""
Framework for defining and running agents.

Use these classes as in the following example::

   import Agent

   class Xxx( Agent.Agent ):
      def __init__( self, entityManager ):
         Agent.Agent.__init__( self, entityManager )

      def doInit( self ):
         mg = entityManager.mountGroup()
         self.config_ = mg.mount( 'Xxx/config', 'Xxx::Config', mode='r' )
         self.status_ = mg.mount( 'Xxx/status', 'Xxx::Status', mode='w' )
         def _finish():
            self.configReactor_ = XxxConfigReactor( self.config_ )
         mg.close( _finish )

   def main():
      container = Agent.AgentContainer( [ Xxx ] )
      container.runAgents()
"""

from __future__ import absolute_import, division, print_function

from collections import Counter
import gc
import optparse
import os
import resource
from resource import (
   RLIMIT_NOFILE,
   getrlimit,
   setrlimit,
)
import socket
import sys
import time
import traceback
import weakref

import Cell
import EntityManager
import PassiveMountNamingLib
import PyServer
import SharedMem
import Tac
from Tracing import (
   t0,
   t1,
   t4,
)
import Valgrind

PROC_THREAD_FILE = '/proc/self/stat'
PROC_SMAPS_FILE = '/proc/self/smaps'
THREADS_FIELD_IDX = -33

def garbageReport():
   ''' Get the top 100 most common ojbect types from the garbage collector '''
   gc.collect()
   try:
      counter = Counter( type( obj ) for obj in gc.get_objects() )
      return { '%s.%s' % ( cls.__module__, cls.__name__ ): count
               for cls, count in counter.most_common( 100 ) }
   except MemoryError: # BUG495384; May blow up; `get_objects` is a big list.
      return {}

def getMemAndThreadCount():
   ''' Compute private memory usage from /proc/self/smaps, also thread count '''
   memory = 0
   with open( PROC_SMAPS_FILE ) as f:
      for line in f:
         if line.startswith( 'Private_' ):
            memory += int( line.split()[ 1 ] )

   with open( PROC_THREAD_FILE ) as f:
      threads = int( f.read().split()[ THREADS_FIELD_IDX ] )

   return memory, threads

def getDataForModel():
   ''' Update data and return model-ready data '''
   memory, threads = getMemAndThreadCount()
   top100 = garbageReport()
   return { 'memory': memory,
            'threads': threads,
            'mostCommonObjects': top100 }

def inValgrind( agentName ):
   return ( Valgrind.isValgrindAgent( agentName ) or
            'valgrind' in os.environ.get( 'LD_PRELOAD', '' ) )

def setOpenFileLimit( limit ):
   setrlimit( RLIMIT_NOFILE, ( limit, limit ) )
   t0( "setrlimit RLIMIT_NOFILE to", limit )

def setMaxOpenFileLimit( agentName ):
   value = Tac.run( "sysctl -n fs.nr_open".split(), stdout=Tac.CAPTURE )
   maxOpenFiles = int( value )
   try:
      setOpenFileLimit( maxOpenFiles )
      return
   except ( ValueError, resource.error ):
      t0( "warning: setrlimit(RLIMIT_NOFILE) failed with", maxOpenFiles )
      if inValgrind( agentName ):
         # While running under valgrind, ignore the failure. Read
         # https://bugs.kde.org/show_bug.cgi?id=73146 for more details
         return
      # Fall through `except`

   _, hard = getrlimit( RLIMIT_NOFILE )
   setOpenFileLimit( hard )

class Agent( object ):
   usingAgentContainer = False

   def __init__( self, entityManager, agentName=None, warmupWarningInterval=60 ):
      """Every agent program should provide a class derived from Agent to
      providing warm-up reporting that will allow other components in the
      system to monitor the warm-up status of the agent.  Typically Agent
      classes are instantiated by ``AgentContainer``, so the typical agent
      code does not explicitly construct an Agent class.

      Parameters:

      * ``entityManager`` -- an :mod:`EntityManager`
      * ``agentName`` -- name of the agent.  (default: the class name)
      """
      self.agentName = agentName or self.__class__.__name__
      sysname = entityManager.sysname()
      t0( "Starting agent %s in system %s" % ( self.agentName, sysname ) )
      self.entityManager = entityManager
      self.shmemEntityManager = SharedMem.entityManager( sysdbEm=entityManager )
      if 'agentDirName' in dir( self ):
         self.agentDirName_ = getattr( self, 'agentDirName' )
      else:
         self.agentDirName_ = self.agentName

      self.warmupWarningInterval_ = warmupWarningInterval

      sysnameDir = Tac.root.newEntity( 'Tac::Dir', sysname )
      self.agentRoot_ = sysnameDir.newEntity( 'Tac::Dir', self.agentDirName_ )

      if os.geteuid() == 0:
         # Set the maximum open file descriptor limit as high as we can to support
         # VRF scale of the order of 1K.  The Linux kernel allocates the necessary
         # memory dynamically, so there is no memory overhead in increasing this to
         # the maximum value. If an agent wants to set its own open fd limit, it
         # should call setrlimit explicitly with a more appropriate value.
         setMaxOpenFileLimit( self.agentName )

      if not self.usingAgentContainer:
         # ie we are being created outside the agent container mechanism
         # create the associated CAgent object which handles the agent config
         self.cAgent_ = Tac.newInstance( "AgentBase::PyAgent", self.agentName )
         # pass a weakref to the c++ object so that it does not keep the agent
         # alive if it is deleted, this only happens in tests (when this was written)
         weakSelf = weakref.proxy( self )
         self.cAgent_.pythonObj = id( weakSelf )
         self.cAgent_.agentSysdbName = self.agentName
         self.cAgent_.agentDirName = self.agentDirName_
         self.cAgent_.warmupReportingInterval = self.warmupWarningInterval_
         self.cAgent_.entityManager = self.entityManager.cEntityManager()

         self.cAgent_.initialized = True

         mg = self.entityManager.mountGroup()
         # Note: CAgent mounts agent/config with flags "rfc", so copying
         #       that here to avoid any conflict.
         agentGlobalConfigDir = mg.mount( "agent/config",
                                          "Agent::GlobalConfigDir", "rfc" )

         def finish( ):
            from AgentGlobalConfigReactor import AgentGlobalConfigReactor
            agentName = self.agentName
            # pylint: disable=attribute-defined-outside-init
            self.agentGlobalConfigDirReactor = Tac.collectionChangeReactor(
                  agentGlobalConfigDir.agentGlobalConfig, AgentGlobalConfigReactor,
                  reactorFilter=lambda x: x == agentName )

         mg.close( finish )

   def getDefaultTrace( self ):
      return ""

   def defaultSchedStatsEnabled( self ):
      return True

   def doInit( self, entityManager ):
      """ This function is called from PyAgent during initialization, an agent
      should override this method to do its Sysdb mounts. Do mounts in this
      function (instead of in __init__()) allows the interlock between an agent
      instance running on a standby and active supervisor to work.
      This function is named after the CAgent doInit() function as it is called
      at the same point in agent initialization as a c++ agent."""
      pass

   def __del__( self ):
      # delete the CAgent
      # NOTE: At this time this only happens in tests as there is no method
      # to remove the agent references from an AgentContainer
      self.cAgent_ = None

   def warm( self ):
      """Returns whether or not the Agent is warm.  An agent is said to be warm
      if it is behaving in accordance with its configuration.  Subclasses should
      override this method if they need to do any processing (other than that in
      their constructor) before being considered warm."""
      return self.entityManager.cEm_.allMountsComplete()

   def agentInitCallback( self ):
      """This function is called from PyAgent during initialization."""
      try:
         self.doInit( self.entityManager )
      except:
         sys.excepthook( *sys.exc_info() )
         # Tac.pdb()
         raise

   def agentWarmupIntervalCallback( self ):
      """This function is called from PyAgent during initialization."""
      try:
         return long( self.warmupWarningInterval_ )
      except:
         sys.excepthook( *sys.exc_info() )
         # Tac.pdb()
         raise

   def agentNameCallback( self ):
      """This function is called from PyAgent::doUpdateWarm()."""
      try:
         return str( self.agentName )
      except:
         sys.excepthook( *sys.exc_info() )
         Tac.pdb()
         raise

   def warmCallback( self ):
      """This function is called from PyAgent::doUpdateWarm().
      We call this function instead of warm so that we can get a backtrace when
      the warm function crashes, each Agent can override warm and we don't want
      to have to put this try...except code in every instance.
      """
      try:
         rc = self.warm()
         # force return type to the expected type
         return bool( rc )
      except:
         sys.excepthook( *sys.exc_info() )
         # Tac.pdb()
         raise

   def handleMountError( self, failed ):
      ''' helper function for agents that support hotplug hardware, there is a race
      when hotplug hardware can be removed just as an agent is starting and this
      causes the mount of the agents config to fail, this handler provide a common
      way for agents to handle this error and not produce a core or a backtrace and
      logs that can alarm users and cause tests to fail
      '''
      sys.stderr.write( 'Mounts failed, assuming hardware removed, exiting\n' )
      sys.stderr.write( str( failed ) )
      # sleep to give Launcher a chance to deconfigure the agent and reduce churn
      time.sleep( 1 )
      sys.exit( 1 )

   def createName( self, *params ):
      ''' Create a hyphen-separated name using the class name + the instance specific
          params to create a unique name for this particular instance of the agent.
          If an agent needs a more sophisticated naming scheme, use your own code, or
          override this class, and let this base function be agnostic of non-generic
          concepts.'''
      params = [ str( p ) for p in params if p is not None and p != "" ]
      params.insert( 0, self.__class__.__name__ )
      return '-'.join( params )

   def _cAgent( self ):
      if hasattr( self, "cAgent_" ):
         return self.cAgent_
      else:
         return None

   def _createLocalRedundancyStatus( self ):
      # pylint: disable=attribute-defined-outside-init
      self.redundancyStatusMgr_ = \
          Tac.newInstance( "Redundancy::Agent::RedundancyStatusManager",
                           self.entityManager.cEm_,
                           self.entityManager.redundancyStatus() )
      self.redundancyStatus_ = self.redundancyStatusMgr_.filteredStatus

   def redundancyStatus( self ):
      cagent = self._cAgent()
      if cagent:
         return cagent.redundancyStatus
      if not getattr( self, "redundancyStatus_", None ):
         self._createLocalRedundancyStatus()
      return self.redundancyStatus_

   def redundancyStatusMgr( self ):
      cagent = self._cAgent()
      if cagent:
         return cagent.redundancyStatusMgr
      if not getattr( self, "redundancyStatusMgr_", None ):
         self._createLocalRedundancyStatus()
      return self.redundancyStatusMgr_

   def flushEntityLog( self, callBack, timeout=180 ):
      """ Calls the callBack when all outgoing entity log messages have been sent
      and acknowledged by the peer. If flush is not successful with in timeout,
      then calls the callBack with timedOut arguement set to True. """
      # In order to correctly pick the Sysdb/Archer mounts, use the
      # cEntityManager's getEntity, and don't directly index off the
      # root of EntityManager.
      cEm = self.entityManager.cEntityManager()
      req = cEm.getEntity[ Cell.path( "agent/sync/config/" ) + self.agentName ]
      reqSysdb = cEm.getEntity[ Cell.path( "agent/sync/configSysdb/" ) +
                                self.agentName ]
      rspRoot = cEm.getEntity[ Cell.path( "agent/sync/status" ) ]
      t0( 'SyncConfig', req, 'SyncConfigSysdb', reqSysdb, 'RspRoot', rspRoot )
      # We have to increment syncRequestRound in syncConfigSysdb that goes via Sysdb
      # to ensure that the write has been completed.
      req.syncRequestRound += 1
      if reqSysdb:
         reqSysdb.syncRequestRound += 1
         assert req.syncRequestRound == reqSysdb.syncRequestRound

      def doWait():
         if self.agentName not in rspRoot.status:
            return False
         return req.syncRequestRound == \
                rspRoot.status[ self.agentName ].syncResponseRound
      Tac.Poller( doWait,
                  lambda ignored: callBack( timedOut=False ),
                  timeoutHandler=lambda: callBack( timedOut=True ),
                  description="entity log to be flushed",
                  timeout=timeout )

   def agentScheduler( self ):
      cAgent = self._cAgent()
      if cAgent is None:
         cAgent = self.agentRoot_[ self.agentName ]
         assert cAgent is not None
      assert cAgent.scheduler is not None
      return cAgent.scheduler

   def agentCmdRequestDirSm( self ):
      cAgent = self._cAgent()
      if cAgent is None:
         cAgent = self.agentRoot_[ self.agentName ]
         assert cAgent is not None
      return cAgent.cmdRequestDirSm

   def createLocalEntity( self, entityName, typeName, path ):
      # self.cAgent_ is None if the agent is created using AgentContainer
      # In that case, fetch the agent from the entityManager
      cAgent = self._cAgent()
      if cAgent is None:
         cAgent = self.agentRoot_[ self.agentName ]
      return cAgent.createLocalEntity( entityName, typeName, path )

   def localEntitiesCreatedIs( self, done ):
      # self.cAgent_ is None if the agent is created using AgentContainer
      # In that case, fetch the agent from the entityManager
      cAgent = self._cAgent()
      if cAgent is None:
         cAgent = self.agentRoot_[ self.agentName ]
      cAgent.localEntitiesCreated = done

   def doStage( self, stageClass, stage ):
      ''' This is called by PyStagesHelper when the agent registers to wait for a
          stage of the default and such stage is published by the StageMgr. Agents
          that participate in a stage progression should override this function.'''
      raise Exception

   def doStageForInstance( self, stageClass, stage, instanceName ):
      ''' This is called by PyStagesHelper when the agent registers to wait for a
          stage of a particulat instance and such stage is published by the
          StageMgr. Agents that participate in a stage progress and do not use the
          stage default instance should override this function.'''
      raise Exception

   def handleStageInstance( self, instanceName ):
      # no-ops
      pass

   def handleStageProgressionComplete( self, stageClass, instanceName ):
      ''' This is called by PyStagesHelper when the agent registers to wait for a
          stage instance and such instance has been complete. '''
      raise Exception

class AgentContainer( object ):
   def __init__( self, agentClasses,
                 sysname=None, rootFlags=None, agentTitle=None,
                 scheduledAttrLog=False, threadingOk=True,
                 startupWaitTime=None, mountTimeout=None,
                 pyServerUnixAddr=None, passiveMount=False, optionalMount=False,
                 implicitMount=False, bypassSysdbInit=False, logProcessStatus=True,
                 entityManager=None, hybridMounts=False, mountProfilePath=None,
                 redundancyStatusProvider=False ):
      """A container for the specified list of agent classes.  This provides
      a mechanism for running one or more agents in a single process, as well
      as registering the agents in the :mod:`AgentDirectory` and starting a
      :mod:`PyServer` in the process.

      Parameters:

      * ``agentClasses`` -- A list of a class derived from :class:`Agent`
        Note only one class allowed in the list.
      * ``sysname`` -- The sysname in which to register the agents.  If None is
        specified, defaults to 'ar'
      * ``rootFlags`` -- passed to the :mod:`EntityManager` constructor.
      * ``agentTitle`` -- if provided, is used as the process title.
        Otherwise, the process title is set by --agenttitle, Tac.setproctitle()
        , or is constructed from command line arguments.
      * ``startupWaitTime`` -- amount of time to wait for agent to connect to Sysdb.
        Note this does not affect mount timeout and timeout for Sysdb initialization.
        Instead it just affects reconnectRetries for EntityManager.
      * ``mountTimeout`` -- timeout for mounts, passed to the entityManager used
        by the agents.
      * ``logProcessStatus`` -- whether or not to syslog the process status
      """
      # Remove HEAPCHECK from environment so child processes (such as spawned
      # by Tac.run()) won't inherit it. This won't affect the current process
      # since the checker would have been activated at this point.
      os.environ.pop( 'HEAPCHECK', None )
      # Build two dicts mapping agentName -> agentClass and agentClass -> list of
      # command-line options to pass to the agent constructor.
      self.agentClasses_ = {}
      self.agentOptions_ = {}
      self.agentDirName_ = {}
      self.rootFlags_ = rootFlags
      for agentClass in agentClasses:
         agentName = agentClass.__name__
         self.agentClasses_[ agentName ] = agentClass
         self.agentOptions_[ agentClass ] = []
         if "agentDirName" in agentClass.__dict__:
            self.agentDirName_[ agentName ] = agentClass.agentDirName

      # Figure out the process title,
      self.explicitAgentTitle = agentTitle is not None
      self.agentTitle_ = agentTitle
      self.scheduledAttrLog_ = scheduledAttrLog
      self.passiveMount_ = passiveMount
      self.optionalMount_ = optionalMount
      self.implicitMount_ = implicitMount
      self.useLegacyName_ = False

      if passiveMount:
         t0( "PASSIVE MOUNT IS ENABLED : agentTitle", self.agentTitle_ )
         if self.rootFlags_:
            self.rootFlags_ += "p"
         else:
            self.rootFlags_ = "p"

      if self.optionalMount_:
         self.rootFlags_ += "O"

      if self.implicitMount_:
         self.rootFlags_ += "M"

      self.startupWaitTime = startupWaitTime
      self.mountTimeout = mountTimeout

      # Set whether we are going to wait for remote Sysdb initialization to complete
      self.bypassSysdbInit_ = bypassSysdbInit

      # Initialize the command-line option parser.  This may be modified by calls to
      # addOption().
      self.parser_ = optparse.OptionParser()
      self.sysname_ = sysname
      if self.sysname_ is None:
         self.parser_.add_option( "-s", "--sysname", action="store",
                                  default=os.environ.get( "SYSNAME", "ar" ),
                                  help="system name (default: %default)" )
      self.parser_.add_option( "--daemonize", dest="daemonize", action="store_true",
                               help="Run as daemon (forked into background)" )
      self.parser_.add_option( "--pidfile", dest="pidfile", action="store",
                               help="Write pid into the given file" )
      self.parser_.add_option( "--ignore", action="store",
                               help="Use this parameter if you need to lengthen the "
                               "command-line.  Its value will be ignored" )
      self.parser_.add_option( "--sysdbport", action="store",
                               help="Specify the host:port of the upstream"
                               " Sysdb process" )
      self.parser_.add_option( "--agenttitle", action="store",
                               help=optparse.SUPPRESS_HELP )
      self.parser_.add_option( "--scheduled", dest="scheduled", action="store_true",
                               default=False, help="Use Scheduled AttrLog" )
      self.parser_.add_option( "--sysdbsockname", action="store",
                               help="Specify the unix domain socket of the upstream"
                               " Sysdb process" )
      self.parser_.add_option( "--pyserverport", dest="pyServerPort", type="int",
                               default=None,
                               help="Port number for PyServer to listen on" )
      self.parser_.add_option( "--pyserveripaddr", dest="pyServerIpAddr",
                               default='127.0.0.1',
                               help="IP Addr for PyServer to listen on" )
      self.parser_.add_option( "--pyserverinterface", dest="pyServerInterface",
                              default='', help="Interface for PyServer to bind on" )
      self.pyServerUnixAddr_ = pyServerUnixAddr
      if self.pyServerUnixAddr_ is None:
         self.parser_.add_option( "--pyserverunixaddr", dest="pyServerUnixAddr",
                                  default='',
                                  help="Unix Addr for PyServer to listen on" )
      self.options = None
      self.entityManager_ = entityManager
      self.redundancyStatusProvider_ = redundancyStatusProvider
      self.threadingOk_ = threadingOk
      self.logProcessStatus_ = logProcessStatus
      self.agenttitle_ = None
      self.hybridMounts_ = hybridMounts
      self.mountProfilePath_ = mountProfilePath

   def __str__( self ):
      return 'AgentContainer( [ %s ] )' % Tac.proctitle()

   def addOption( self, *args, **kwargs ):
      r"""Adds a command-line option.  The \*args and \*\*kwargs are passed
      through to the ``optparse.OptionParser.add_option()`` method, except for
      the ``agentClass`` keyword parameter.  If this parameter is specified,
      then the option value will be passed as a keyword argument to the
      constructor of this agent."""
      if 'agentClass' in kwargs:
         agentClass = kwargs[ 'agentClass' ]
         del kwargs[ 'agentClass' ]
      else:
         agentClass = None
      option = self.parser_.add_option( *args, **kwargs )
      if agentClass:
         self.agentOptions_[ agentClass ].append( option.dest )
      t1( 'Registered command-line option %s with agent %s' %
          ( option.dest, agentClass ) )

   def addSliceIdOption( self, agentClass ):
      """Called to indicate that the given agent class supports slices. This
      requires that the agentClass takes ``sliceId`` as a constructor parameter."""
      if not self.parser_.has_option( "--sliceId" ):
         self.parser_.add_option( "--sliceId", dest="sliceId", action="store",
                                  default="",
                                  help="Agent slice identifier" )
      self.agentOptions_[ agentClass ].append( "sliceId" )

   def parseOptions( self ):
      """Parses the command-line arguments, exiting on a syntax error."""
      ( self.options, args ) = self.parser_.parse_args()
      if args:
         self.parser_.error( "unexpected arguments: %s" % args )

   def _createEntityMgr( self, procTitle ):
      if self.entityManager_:
         return

      self.cAgentContainer_.isLocal = False
      if self.options.sysdbport:
         sysdbHostPort = EntityManager.parseHostPort( self.options.sysdbport )
         self.cAgentContainer_.host, self.cAgentContainer_.sysdbPort = \
                                                                  sysdbHostPort
      else:
         sysdbHostPort = None
         self.cAgentContainer_.sysdbPort = 0
         self.cAgentContainer_.host = ""

      if self.options.sysdbsockname:
         sysdbSockname = self.options.sysdbsockname
         self.cAgentContainer_.sysdbSockName = sysdbSockname
      else:
         sysdbSockname = None

      entityManager = EntityManager.Sysdb(
         sysname=self.sysname_,
         dieOnDisconnect=True,
         waitForSysdbToInitialize=False,
         sysdbhostport=sysdbHostPort, sysdbsockname=sysdbSockname,
         rootFlags=self.rootFlags_,
         connectionTimeout=self.startupWaitTime,
         mountTimeout=self.mountTimeout,
         scheduled=self.options.scheduled or self.scheduledAttrLog_,
         sysdbServer=False,
         verbose=True,
         agentName=procTitle,
         hybridMounts=self.hybridMounts_,
         mountProfilePath=self.mountProfilePath_
         )

      self.entityManager_ = entityManager

   def _initRedundancy( self ):
      return

   def useLegacyName_DO_NOT_COPY_THIS( self ):
      if self.entityManager_ and self.entityManager_.started:
         assert False, "Call useLegacyName_DO_NOT_COPY_THIS() after EntityManager"\
                       " already started"
      self.useLegacyName_ = True

   def startAgents( self ):
      """Instantiates the list of agents, returning control to you.
      Note: you probably want runAgents() instead, because most agents
      do not work correctly unless the activity manager has control.

      Create the underlying AgentContianer entity that does all the work
      and register the agent creation callback function required to create
      python agent classes """

      if not self.options:
         self.parseOptions()

      if self.sysname_ is None:
         self.sysname_ = self.options.sysname
      sysname = self.sysname_

      # pylint: disable=attribute-defined-outside-init
      self.cAgentContainer_ = Tac.newInstance( "AgentBase::AgentContainer",
                                               "agentCont" )
      # pylint: enable=attribute-defined-outside-init
      self.cAgentContainer_.logProcessStatus = self.logProcessStatus_
      self.cAgentContainer_.sysname = sysname
      self.cAgentContainer_.passiveMount = self.passiveMount_
      self.cAgentContainer_.redundancyStatusProvider = self.redundancyStatusProvider_
      self.cAgentContainer_.pythonAgent = True
      self.cAgentContainer_.useLegacyName = self.useLegacyName_
      if self.useLegacyName_:
         assert len( self.agentClasses_ ) == 1
         self.cAgentContainer_.pythonLegacyName = \
               list( self.agentClasses_.values() )[ 0 ].__name__

      # Set agentTitle passed from the ProcMgr.  This is for consistency.
      if self.options.agenttitle:
         self.cAgentContainer_.agentTitle = self.options.agenttitle

      # add all the agents to the agent container
      for agent in self.agentClasses_:
         agentDirName = self.agentDirName_.get( agent, agent )
         self.cAgentContainer_.doAddAgent( agent, "AgentBase::PyAgent",
                                           agentDirName )

      # set the agent creation callback, this also stores the real argv & argc
      # in a singleton, required so that the c++ code can change the proc title
      import SysdbPyAgent
      # pylint: disable=c-extension-no-member
      SysdbPyAgent.setAgentCreateCallback( self.agentCreateCallback )

      if self.options.daemonize:
         # this option has an immediate effect, the process will deamonize now
         self.cAgentContainer_.daemonize = True

      # Setup the appropriate procTitle and do sanity check
      # The order of precedence of APIs that affect procTitle:
      # 1. AgentContainer's agentTitle constructor argument
      # 2. --agenttitle passed from ProcMgr
      # 3. Calling Tac.setproctitle() explicitly before startAgents()
      # 4. By default, the procTitle is derived from argv
      procTitle = Tac.proctitle()
      if self.agentTitle_ is not None:
         procTitle = self.agentTitle_
         t0( 'Set procTitle = agentTitle = ' + self.agentTitle_ )
      elif self.options.agenttitle:
         # Use agentTitle passed from it.
         # If the agent use scheduledAttrLog, the --scheduled argument must match
         # TODO: --scheduled can be removed later if there is no use
         assert self.scheduledAttrLog_ == self.options.scheduled
         procTitle = self.options.agenttitle
         t0( 'Set procTitle = --agenttitle = ' + self.options.agenttitle )
      elif procTitle != "no title":
         t0( 'procTitle is already set : ' + procTitle )
      else:
         # This is where the agenttitle is not passed from ProcMgr.  In this case,
         # try to create the procTitle from basename of exe and argv by
         # PassiveMountNamingLib
         tmpargv = sys.argv
         t4( 'Original sys.argv = ' + str( sys.argv ) )
         if sys.argv:
            argvExeName = os.path.basename( sys.argv[ 0 ] )
         else:
            argvExeName = None
         # In the breadth test, we saw that sys.argv doesn't have the binary name
         # We get it from the traceback instead.
         exeName = next( n[ 0 ] for n in traceback.extract_stack()
                         if os.path.exists( n[ 0 ] ) )
         exeName = os.path.basename( exeName ).split( '.' )[ 0 ]
         t4( 'exeName from traceback = ' + exeName )
         if not exeName == argvExeName:
            tmpargv = [ exeName ] + tmpargv
         t4( 'argv used for createAgentProcTitle = ' + str( tmpargv ) )
         procTitle = PassiveMountNamingLib.createAgentProcTitle( tmpargv )
         t0( 'Set procTitle = ' + procTitle )
      t0( 'Python Agent set procTitle = ' + procTitle )
      # Set the process title, so that messages sent to the syslog are prefixed with
      # the correct title, not just 'python'. This change is instantaneous.
      # This also used for passive mount to indicate the mount profile
      self.cAgentContainer_.setProcTitle( procTitle )

      # Create an entity manager.
      Tac.sysnameIs( sysname )
      self._createEntityMgr( procTitle )
      assert self.entityManager_

      self.cAgentContainer_.entityManager = self.entityManager_.cEntityManager()

      mg = self.entityManager_.mountGroup()
      # Note: CAgent mounts agent/config with flags "rfc", so copying
      #       that here to avoid any conflict.
      agentGlobalConfigDir = mg.mount( "agent/config",
                                       "Agent::GlobalConfigDir", "rfc" )

      def finish( ):
         from AgentGlobalConfigReactor import AgentGlobalConfigReactor
         agentList = [ self.agentDirName_.get( agent, agent )
                       for agent in self.agentClasses_ ]
         # pylint: disable=attribute-defined-outside-init
         self.agentGlobalConfigDirReactor = Tac.collectionChangeReactor(
               agentGlobalConfigDir.agentGlobalConfig, AgentGlobalConfigReactor,
               reactorFilter=lambda x: x in agentList )

      mg.close( finish )

      if self.cAgentContainer_.pyServerRunnable:
         # pylint: disable=attribute-defined-outside-init
         # create the PyServer, we cannot use the PyServer code in the c++
         # AgentContainer because it only works when python is not already running
         self.pyServerInet_ = None
         if self.options.pyServerPort is not None:
            # Start INET PyServer for the process.
            self.pyServerInet_ = PyServer.PyServer( "PyServerInet",
                                             port=self.options.pyServerPort,
                                             listenAddr=self.options.pyServerIpAddr,
                                             domain='internet',
                                             threadingOk=self.threadingOk_ )
            if self.options.pyServerInterface:
               # Note that we need to run as root for setting sockopt.
               self.pyServerInet_.socket_.setsockopt( socket.SOL_SOCKET,
                     socket.SO_BINDTODEVICE, self.options.pyServerInterface )

         self.pyServerUnix_ = None
         if self.pyServerUnixAddr_ is None:
            self.pyServerUnixAddr_ = self.options.pyServerUnixAddr
         # Start UNIX PyServer for the process.
         self.pyServerUnix_ = PyServer.PyServer( "PyServerUnix",
                                               port=self.pyServerUnixAddr_,
                                               threadingOk=self.threadingOk_ )

         # After creating the server, we better have activity lock before
         # doing anything with tacc. Make sure we release the lock before
         # doStartAgents(). It grabs the lock again when needed.
         with Tac.ActivityLockHolder():
            if self.pyServerInet_:
               self.cAgentContainer_.pyServerPort = self.pyServerInet_.port()
            if self.pyServerUnix_:
               self.cAgentContainer_.pyServerUnixAddress = \
                   self.pyServerUnix_.sockname()
            # prevent the c++ AgentContainer from starting a
            # pyserver listen, we have taken care of it here
            self.cAgentContainer_.pyServerAlreadyRunning = True
            if self.options.pidfile:
               self.cAgentContainer_.pidFile = self.options.pidfile

      self.agents_ = [] # pylint: disable=attribute-defined-outside-init

      self.cAgentContainer_.bypassSysdbInit = self.bypassSysdbInit_
      self.cAgentContainer_.doStartAgents()
      self._initRedundancy()

   def agentCreateCallback( self, agentName ):
      # This function is called from the PyAgent init handler.
      try:
         kwargs = {}
         agentClass = self.agentClasses_[ agentName ]
         # We want the agent to know if it ws created via the
         # AgentContainer or not so it can create the AgentBase::PyAgent itself
         # when not created via this agent container
         agentClass.usingAgentContainer = True
         for optionName in self.agentOptions_[ agentClass ]:
            kwargs[ optionName ] = getattr( self.options, optionName )
         agent = agentClass( self.entityManager_, **kwargs )
         self.agents_.append( agent )
         return agent
      except:
         sys.excepthook( *sys.exc_info() )
         Tac.pdb()
         raise

   def runAgents( self, **kargs ):
      """Instantiates the list of agents, and runs activities forever."""
      self.startAgents( **kargs )
      self.cAgentContainer_.doRunAgents()

   def entityManager( self ):
      """Returns the :mod:`EntityManager` used by this instance. Valid
      only after ``startAgents()`` or ``runAgents()`` has been called."""
      return self.entityManager_
