#!/usr/bin/env python
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""Python bindings for Afetch.

A callback-based Afetch client is provided, allowing one
to make HTTP(S) requests and receive updates via callback when
complete as well as receive callbacks for further updates.
"""

import Tac


Error = Tac.Type( 'Afetch::Error' )
Method = Tac.Type( 'Afetch::Method' )
RequestState = Tac.Type( 'Afetch::RequestState' )
Scheme = Tac.Type( 'Afetch::Scheme' )
AuthType = Tac.Type( 'Afetch::HttpAuthenticationType' )

# HTTP method name shortcuts
CONNECT = Method.CONNECT
DELETE = Method.DELETE
GET = Method.GET
HEAD = Method.HEAD
OPTIONS = Method.OPTIONS
POST = Method.POST
PUT = Method.PUT
TRACE = Method.TRACE
PATCH = Method.PATCH

# Attributes to ignore when making dictionaries of counters and timers
TAC_ATTR_IGNORE = frozenset(
   ( 'entity', 'fullName', 'isNondestructing', 'name', 'parent', 'parentAttrName' ) )

def _valueToDict( value ):
   result = {}
   for key in value.attributes:
      if key not in TAC_ATTR_IGNORE:
         result[ key ] = getattr( value, key )
   return result


class ResponseReactor( Tac.Notifiee ):
   """A TACC notifiee for Afetch responses, used to initiate client callbacks."""

   notifierTypeName = 'Afetch::Response'

   def __init__( self, afetchResponseDir, callback ):
      Tac.Notifiee.__init__( self, afetchResponseDir )
      self.afetchResponseDir_ = afetchResponseDir
      self.callback_ = callback

   @Tac.handler( 'requestState' )
   def handleRequestState( self ):
      if self.notifier_.finished():
         # Only notify when we've actually changed the result, which will be
         # on the first response and then when any important response field changes
         if self.notifier_.lastChangeTime >= self.notifier_.lastUpdateTime:
            self.callback_( self.notifier_ )


class Client( object ):
   """The Python Afetch Client interface."""

   def __init__( self, baseDir=None ):
      if baseDir:
         self.root = baseDir.newEntity( 'Afetch::Root', 'afetchRoot' )
      else:
         self.root = Tac.newInstance( 'Afetch::Root', 'afetch' )
      self.aresolveSm = self.root.aresolveSm
      self.afetchSm = self.root.afetchSm
      self.requestDir = self.root.afetchRequest
      self.responseDir = self.root.afetchResponse
      self.callbacks_ = {}
      self.reactor_ = Tac.collectionChangeReactor(
         self.responseDir.response,
         ResponseReactor,
         reactorArgs=( self._handleResponse, ) )

   @property
   def numRequestsRunning( self ):
      return len( [ r for r in self.responseDir.response
                    if r.requestState > RequestState.stateNotInit ] )

   @property
   def numRequests( self ):
      return len( self.requestDir.request )

   def request( self, method, uri, callback=None, headers=None, running=False ):
      """Builds an Afetch::Request object to call 'uri' with HTTP 'method'.

      By default, this method starts the request running by placing it
      in the Afetch request collection; use running=False as a keyword
      argument to not start the request straight away (useful for POST
      requests to set a body).

      The response to this request, once running, will be received by your callback.

      Args:
         method: str, an enum value from Afetch.Method (e.g., Afetch.Method.GET)
         uri: str, the URI to retrieve
         callback: A callable, the callback for this request (key); if not specified,
            the response will only be available by calling getResponse for the key
            and polling the response object manually.
         headers: A dict, HTTP request headers
         running: bool, if True, start the request immediately

      Returns:
        An Afetch::Request object. Add it to the Afetch request
        collection to start the request if running=False was
        passed. See module docstring for details on the Request and
        Response objects.
      """
      key = Tac.Value( 'Afetch::RequestKey', method, uri )
      request = Tac.Value( 'Afetch::Request', key )
      if headers:
         for hkey in headers:
            request.header[ hkey ] = headers[ hkey ]
      self.callbacks_[ key.hash() ] = callback

      if running:
         self.requestDir.addRequest( request )
      return request

   def connect( self, uri, **kwargs ):
      return self.request( CONNECT, uri, **kwargs )

   def get( self, uri, **kwargs ):
      return self.request( GET, uri, **kwargs )

   def head( self, uri, **kwargs ):
      return self.request( HEAD, uri, **kwargs )

   def post( self, uri, **kwargs ):
      return self.request( POST, uri, **kwargs )

   def put( self, uri, **kwargs ):
      return self.request( PUT, uri, **kwargs )

   def delete( self, uri, **kwargs ):
      return self.request( DELETE, uri, **kwargs )

   def options( self, uri, **kwargs ):
      return self.request( OPTIONS, uri, **kwargs )

   def trace( self, uri, **kwargs ):
      return self.request( TRACE, uri, **kwargs )

   def patch( self, uri, **kwargs ):
      return self.request( PATCH, uri, **kwargs )

   def getResponse( self, requestKey ):
      """Returns the current value of the reponse for request."""
      return self.responseDir.response[ requestKey ]

   def requestCounters( self, requestKey ):
      """Returns the current dictionary of response counters for a request."""
      response = self.responseDir.response[ requestKey ]
      return _valueToDict( response.counter )

   def requestTimers( self, requestKey ):
      """Returns the current dictionary of response timers for a request."""
      response = self.responseDir.response[ requestKey ]
      return _valueToDict( response.timer )

   def start( self, request ):
      """Starts a request (such as that created by makeRequest) running."""
      self.requestDir.addRequest( request )

   def stop( self, requestKey ):
      """Cancels the request, also deleting the response object immediately."""
      self.deleteRequest( requestKey )

   def deleteRequest( self, requestKey ):
      """Erases the request/response object pair for the given request."""
      del self.requestDir.request[ requestKey ]

   def _handleResponse( self, response ):
      """Select the appropriate callback for the response and call it."""
      callback = self.callbacks_.get( response.requestKey.hash() )
      if callback:
         callback( response )

