#!/usr/bin/env python
# Copyright (c) 2009,2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac, Arnet
from eunuchs.in_h import IPPROTO_UDP, IPPROTO_TCP

NON_DEFAULT_VRFNAME = 'management'

# All types of ACLs
aclTypes = [ 'ip', 'ipv6', 'mac' ]
aclTypeDisplayNames = {
   'ip' : 'IPv4',
   'ipv6' : 'IPv6',
   'mac' : 'MAC'
   }
serviceAclTypes = [ 'ip', 'ipv6' ]

# All directions of ACLs
aclDirections = [ 'in', 'out' ]

# max rule sequence number (32-bit)
MAX_SEQ = 0xFFFFFFFF # 32-bit integer
# max DSCP value
MAX_DSCP = 63
# max IP length value
MAX_IP_PACKET_LEN = 65535

IPPROTO_IGMP = 2
IPPROTO_RSVP = 46
IPPROTO_OSPF = 89
IPPROTO_PIM = 103
IPPROTO_VRRP = 112
IPPROTO_AHP = 51
ICMP_ALL = 0xFFFF
ICMP6_ALL = 0xFFFF

IPPROTO_GRE = 0x2F
IPPROTO_SCTP = 132
IPPROTO_MPLSOVERGRE = 137
TNI_MAX = ( 2 ** 24 ) -1
VNI_MAX = ( 2 ** 24 ) -1
GRE_ALL = ( 2 ** 16 ) - 1
GREPROTO_MAX = ( 2 ** 16 ) - 1 
NVGRE_PROTO = 0x6558
MPLS_PROTO = 0x8847
MPLSLABEL_MAX = ( 2** 20 ) - 1
TEID_MAX = ( 2 ** 32 ) - 1
USERL4_MAX = ( 2 ** 32 ) - 1
PAYLOAD_MAX = ( 2 ** 32 ) - 1

MACVLAN_ALL = 0xFFFF
MACCOS_ALL = 0xFF
MACPROTO_ALL = 0xFFFFFFFF
MACPROTO_IP4 = 0x0800

# convenient Tac.Value type definition
AclDirection = Tac.Type( 'Acl::AclDirection' )
AclType = Tac.Type( 'Acl::AclType' )
AclAction = Tac.Type( 'Acl::Action' )
CopyToCpuDst = Tac.Type( 'Acl::CopyToCpuDst' )
TtlValue = Tac.Type( "Acl::TtlSpec" )
PortValue = Tac.Type( "Acl::PortSpec" )
IpLenValue = Tac.Type( "Acl::IpLenSpec" )
PayloadValue = Tac.Type( "Acl::PayloadValue" )
PayloadSpec = Tac.Type( "Acl::PayloadSpec" )
PayloadStringConstants = Tac.Type( "Acl::PayloadStringConstants" )
MetadataValue = Tac.Type("Acl::MetadataValue")
MetadataSpec = Tac.Type( "Acl::MetadataSpec" )
IpFilterValue = Tac.Type( "Acl::IpFilter" )
IpRuleValue = Tac.Type( "Acl::IpRuleConfig" )
Ip6FilterValue = Tac.Type( "Acl::Ip6Filter" )
Ip6RuleValue = Tac.Type( "Acl::Ip6RuleConfig" )
MacFilterValue = Tac.Type( "Acl::MacFilter" )
MplsFilterValue = Tac.Type( "Acl::MplsFilter" )
MplsLabelFilterValue = Tac.Type( "Acl::MplsLabelFilter" )
MacRuleValue = Tac.Type( "Acl::MacRuleConfig" )
TcpFlagValue = Tac.Type( "Acl::TcpFlag" )
TcamBankSharingMode = Tac.Type( "Acl::TcamBankSharingMode" )
RemarkConfigValue = Tac.Type( "Acl::RemarkConfig" )
Icmp6RuleType = Tac.Type( "Acl::ImplicitIcmp6RulesType" )
EcnValue = Tac.Type( "Acl::Ecn" )
AclPayloadHdrStart = Tac.Type( "Acl::AclPayloadHdrStart" )
HwAclMechanism = Tac.Type( "Acl::HwAclMechanism" )
ActionDuringAclUpdate = Tac.Type( "Acl::ActionDuringAclUpdate" )

# constants
anyIpAddr = Arnet.AddrWithFullMask( '0.0.0.0', 0 )
anyIp6Addr = Arnet.Ip6AddrWithMask( '::/0' )
anyIp6AddrWithFullMask = Arnet.Ip6AddrWithFullMask( '::/::' )
zeroMacAddr = Tac.Value( "Arnet::EthAddr", 0, 0, 0 )
zeroMacAddrString = '00:00:00:00:00:00'
anyTtlValue = TtlValue( 'any', 0 )
anyIpLenValue = IpLenValue('any', '')
anyPortValue = PortValue( 'any', '' )
vxlanPortValue = PortValue( 'eq', '4789' )

defaultVrf = Tac.newInstance( 'L3::VrfName', 'temp-status' ).defaultVrf

def gtpStrFromProto( gtpProto ):
   for gtpStr, val in gtpProtoByName.iteritems():
      ( portNum, _ ) = val
      if( portNum == gtpProto ):
         return gtpStr
   return ''

def gtpPortFromStr( gtpProto ):
   port, _  = gtpProtoByName[ gtpProto ]
   return port

# Generate a 64-bit ID
def genUniqueId( ):
   return Tac.Value("Acl::UniqueId")

# service name/port number

# services common to both TCP and UDP
commonServiceByName_ = {
   "tcpmux": ( 1, "TCP Port Service Multiplexer" ),
   "rje": ( 5, "Remote Job Entry" ),
   "echo": ( 7, "Echo" ),
   "discard": ( 9, "Discard" ),
   "systat": ( 11, "Active users" ),
   "daytime": ( 13, "Daytime" ),
   "qotd": ( 17, "Quote of the Day" ),
   "msp": ( 18, "Message Send Protocol" ),
   "chargen": ( 19, "Character Generator" ),
   "nsw-fe": ( 27, "NSW User System FE" ),
   "msg-icp": ( 29, "MSG ICP" ),
   "dsp": ( 33, "Display Support Protocol" ),
   "time": ( 37, "Time" ),
   "rlp": ( 39, "Resource Location Protocol" ),
   "tacacs": ( 49, "TAC Access Control System" ),
   "re-mail-ck": ( 50, "Remote Mail Checking Protocol" ),
   "la-maint": ( 51, "IMP Logical Address Maintenance" ),
   "xns-time": ( 52, "XNS (Xerox Network Systems) Time Protocol" ),
   "domain": ( 53, "Domain Name Service" ),
   "xns-ch": ( 54, "XNS (Xerox Network Systems) Clearinghouse" ),
   "isi-gl": ( 55, "ISI Graphics Language" ),
   "xns-mail": ( 58, "XNS (Xerox Network Systems) Mail" ),
   "kerberos": ( 88, "Kerberos Authentication System" ),
   "acr-nema": ( 104, 
                 "ACR-NEMA Digital Imaging and Communications in Medicine" ),
   "csnet-ns": ( 105, "CCSO Name Server Protocol" ),
   "snagas": ( 108, "SNA Gateway Access Server" ),
   "sunrpc": ( 111, "Sun Remote Procedure Call" ),
   "sqlserv": ( 118, "SQL Services" ),
   "bftp": ( 152, "Background File Transfer Program" ),
   "sgmp": ( 153, "Simple Gateway Monitoring Protocol" ),
   "sqlsrv": ( 156, "SQL Service" ),
   "snmp": ( 161, "Simple Network Management Protocol" ),
   "snmptrap": ( 162, "SNMP Traps" ),
   "smux": ( 199, "SNMP Unix Multiplexer" ),
   "at-rtmp": ( 201, "AppleTalk Routing Maintenance" ),
   "qmtp": ( 209, "The Quick Mail Transfer Protocol" ),
   "z39-50": ( 210, "ANSI Z39.50" ),
   "ipx": ( 213, "Internetwork Packet Exchange" ),
   "mpp": ( 218, "Netix Message Posting Protocol" ),
   "imap3": ( 220, "Interactive Mail Access Protocol v3" ),
   "esro-gen": ( 259, "Efficient Short Remote Operations" ),
   "bgmp": ( 264, "Border Gateway Multicast Protocol" ),
   "http-mgmt": ( 280, "http-mgmt" ),
   "pkix-timestamp": ( 318, "PKIX Timestamp" ),
   "ptp-event": ( 319, "Precision Time Protocol Event" ),
   "ptp-general": ( 320, "Precision Time Protocol General" ),
   "matip-type-a": ( 350, "MATIP Type A" ),
   "matip-type-b": ( 351, "MATIP Type B" ),
   "odmr": ( 366, "On Demand Mail Retry" ),
   "rpc2portmap": ( 369, "Rpc2portmap" ),
   "clearcase": ( 371, "Clearcase albd" ),
   "hp-alarm-mgr": ( 383, "HP Performance Data Alarm Manager" ),
   "arns": ( 384, "A Remote Network Server System" ),
   "aurp": ( 387, "Appletalk Update-Based Routing Protocol" ),
   "ldap": ( 389, "Lightweight Directory Access Protocol" ),
   "ups": ( 401, "Uninterruptible Power Supply" ),
   "svrloc": ( 427, "Server Location Protocol" ),
   "snpp": ( 444, "Simple Network Paging Protocol" ),
   "kpasswd": ( 464, "Kerberos Change/Set Password" ),
   "tcpnethaspsrv": ( 475, 
                      "Aladdin Knowledge Systems Hasp services, TCP/IP version" ),
   "pim-auto-rp": ( 496, "PIM Auto-RP" ),
   "citadel": ( 504, "Citadel" ),
   "talk": ( 517, "Talk" ),
   "ncp": ( 524, "NetWare Core Protocol" ),
   "courier": ( 530, "Remote Procedure Call" ),
   "commerce": ( 542, "Commerce Applications" ),
   "dhcpv6-client": ( 546, "DHCPv6 Client" ),
   "dhcpv6-server": ( 547, "DHCPv6 Server" ),
   "new-rwho": ( 550, "new-who" ),
   "rtsp": ( 554, "Real Time Streaming Protocol" ),
   "nntps": ( 563, "Network News Transfer Protocol Over TSL/SSH" ),
   "http-rpc-epmap": ( 593, "HTTP RPC Ep Map" ),
   "ipp": ( 631, "Internet Printing Protocol" ),
   "rlzdbase": ( 635, "RLZ DBase" ),
   "ldaps": ( 636, "LDAP Over TLS/SSH" ),
   "msdp": ( 639, "Multicast Source Discovery Protocol" ),
   "repcmd": ( 641, "SupportSoft Nexus Remote Command" ),
   "ldp": ( 646, "Label Distribution Protocol" ),
   "rmc": ( 657, "Remote Monitoring and Control Protocol" ),
   "ha-cluster": ( 694, "Linux-HA Heartbeat" ),
   "kerberos-adm": ( 749, "Kerberos Administration" ),
   "godi": ( 848, "Group Domain of Interpretation Protocol" ),
   "ftps-data": ( 989, "FTPS Protocol (data)" ),
   "ftps": ( 990, "FTPS Protocol (control)" ),
   "nas": ( 991, "Netnews Administration System" ),
   "openvpn": ( 1194, "OpenVPN" ),
   "pkt-krb-ipsec": ( 1293, "Internet Protocol Security" ),
   "ms-sql-m": ( 1434, "Microsoft SQL Monitor" ),
   "pptp": ( 1723, "Microsoft Point-to-Point Tunneling Protocol" ),
   "radius": ( 1812, "Radius Authentication Protocol" ),
   "radius-acct": ( 1813, "Radius Accounting Protocol" ),
   "nfs": ( 2049, "Network File System" ),
   "mlag": ( 4432, "MLAG Protocol" ),
   "nat": ( 4532, "Nat Sync Protocol" ),
   }

tcpServiceByName = {
   "chargen": ( 19, "Character generator" ),
   "ftp-data": ( 20, "FTP data connections" ),
   "ftp": ( 21, "File Transfer Protocol" ),
   "ssh": ( 22, "Secure Shell Protocol" ),
   "telnet": ( 23, "Telnet Protocol" ),
   "smtp": ( 25, "Simple Mail Transport Protocol" ),
   "whois": ( 43, "Nicname" ),
   "gre": ( 47, "Generic Routing Encapsulation" ),
   "gopher": ( 70, "Gopher" ),
   "netrjs-1": ( 71, "Remote Job Service" ),
   "netrjs-2": ( 72, "Remote Job Service" ),
   "netrjs-3": ( 73, "Remote Job Service" ),
   "netrjs-4": ( 74, "Remote Job Service" ),
   "finger": ( 79, "Finger" ),
   "www": ( 80, "World Wide Web (HTTP)" ),
   "hostname": ( 101, "NIC hostname server" ),
   "iso-tsap": ( 102, "ISO-TSAP Class 0" ),
   "rtelnet": ( 107, "Remote Telnet Service" ),
   "pop2": ( 109, "Post Office Protocol v2" ),
   "pop3": ( 110, "Post Office Protocol v3" ),
   "ident": ( 113, "Ident Protocol" ),
   "uucp-path": ( 117, "UUCP Path Service" ),
   "nntp": ( 119, "Network News Transport Protocol" ),
   "imap": ( 143, "Interim Mail Access Protocol" ),
   "print-srv": ( 170, "Network PostScript" ),
   "vmnet": ( 175, "VMNET" ),
   "bgp": ( 179, "Border Gateway Protocol" ),
   "irc": ( 194, "Internet Relay Chat" ),
   "asip-webadmin": ( 311, "AppleShare IP Web Administration" ),
   "https": ( 443, "HTTP Secure (HTTPS)" ),
   "microsoft-ds": ( 445, "Microsoft-DS SMB File Sharing" ),
   "exec": ( 512, "Remote Process Execution/Rexec" ),
   "login": ( 513, "Rlogin" ),
   "cmd": ( 514, "Remote Shell/Rsh" ),
   "lpd": ( 515, "Line Printer Daemon" ),
   "efs": ( 520, "Extended File Name Server" ),
   "netwnews": ( 532, "Readnews" ),
   "uucp": ( 540, "Unix-to-Unix Copy Program" ),
   "klogin": ( 543, "Kerberos login" ),
   "kshell": ( 544, "Kerberos shell" ),
   "afpovertcp": ( 548, "Apple Filing Protocol Over TCP" ),
   "remotefs": ( 556, "RFS Server" ),
   "submission": ( 587, "Email Message Submission" ),
   "http-alt": ( 591, "Filemaker, Inc. -HTTP Alternate" ),
   "tunnel": ( 604, "TUNNEL Profile" ),
   "ldp": ( 646, "Label Distribution Protocol" ),
   "mac-srvr-admin": ( 660, "MacOS Server Admin" ),
   "acap": ( 674, "Application Configuration Access Protocol" ),
   "msexch-routing": ( 691, "MS Exchange Routing" ),
   "ieee-mms-ssl": ( 695, "IEEE Media Management System Over SSL" ),
   "epp": ( 700, "Extensible Provision Protocol" ),
   "lmp": ( 701, "Link Management Protocol" ),
   "iris-beep": ( 702, 
                  "Internet Registry Information Service Over BEEP" ),
   "silc": ( 706, "Secure Internet Live Conferencing" ),
   "cisco-tdp": ( 711, "Cisco Tag Distribution Protocol" ),
   "tbrpf": ( 712, 
              "Topology Broadcast based on Reverse-Path Forwarding Protocol" ),
   "netconf-ssh": ( 830, "NETCONF over SSH" ),
   "gnmi": ( 6030, "gNMI default port" ),
   "dhcp-failover2": ( 847, "DHCP Failover Protocol" ),
   "iscsi": ( 860, "Internet Small Computers Systems Interface" ),
   "rsync": ( 873, "rysnc File Synchronization Protocol" ),
   "imaps": ( 993, "Internet Message Access Protocol over SSL" ),
   "pop3s": ( 995, "Post Office Protocol 3 over TLS/SSL" ),
   "ms-sql-s": ( 1433, "Microsoft SQL Server" ),
   "mlag-arp-sync": ( 50002, "ARP file transfer server port" ),
   }
udpServiceByName = {
   "nameserver": ( 42, "IEN116 Nameserver Service (obsolete)" ),
   "bootps": ( 67, "Bootstrap Protocol (BOOTP) server" ),
   "bootpc": ( 68, "Bootstrap Protocol (BOOTP) client" ),
   "tftp": ( 69, "Trivial File Transfer Protocol" ),
   "auth": ( 113, "Authentication Service" ),
   "ntp": ( 123, "Network Time Protocol" ),
   "netbios-ns": ( 137, "NetBios name service" ),
   "netbios-dgm": ( 138, "NetBios datagram service" ),
   "netbios-ss": ( 139, "NetBios session service" ),
   "xdmcp": ( 177, "X Display Manager Control Protocol" ),
   "dnsix": ( 195, "DNSIX security protocol auditing" ),
   "mobile-ip": ( 434, "Mobile IP registration" ),
   "isakmp": ( 500,
               "Internet Security Association and Key Management Protocol" ),
   "biff": ( 512, "Biff (mail notification, comsat)" ),
   "who": ( 513, "Who service, rwho" ),
   "syslog": ( 514, "System Logger" ),
   "rip": ( 520, "Routing Information Protocol" ),
   "timed": ( 525, "Timeserver" ),
   "netwall": ( 533, "For Emergency Broadcasts" ),
   "rmonitor": ( 560, "Remote Monitord" ),
   "monitor": ( 561, "Monitord" ),
   "asf-rmcp": ( 623, "ASF Remote Management and Control Protocol" ),
   "olsr": ( 698, "Optimized Link State Routing" ),
   "l2tp": ( 1701, "Layer 2 Tunneling Protocol" ),
   "gtp-u": ( 2152, "GPRS Tunneling Protocol User Data" ),
   "gtp-c": ( 2123, "GPRS Tunneling Protocol Control Data" ),
   "gtp-prime": ( 3386, "GPRS Tunneling Prime Protocol" ),
   "bfd": ( 3784, "Bidirectional Forwarding Detection" ),
   "bfd-echo": ( 3785, "BFD Echo" ),
   "multihop-bfd": ( 4784, "Multihop Bfd" ),
   "micro-bfd": ( 6784, "RFC7130 BFD session over each LAG member link" ),
   "sbfd": ( 7784, "Seamless BFD" ),
   "sbfd-initiator": ( 47657, "Seamless BFD initiator port" ),
   "non500-isakmp": (
      4500, "Internet Security Association and Key Management Protocol" ),
   "lsp-ping": ( 3503, "MPLS LSP-echo Port" ),
   }

gtpProtoByName = {
   "gtp-u": ( 2152, "GTP User Data" ),
   "gtp-c": ( 2123, "GTP Control Data" ),
   "gtp-prime": ( 3386, "GTP Prime Protocol" ),
   }

sctpServiceByName = {
   "discard": ( 9, "Discard" ), 
   "ftp-data": ( 20, "FTP data connections" ),
   "ftp": ( 21, "File Transfer Protocol" ), 
   "ssh": ( 22, "Secure Shell Protocol" ),
   "www": ( 80, "World Wide Web (HTTP)" ),
   "bgp": ( 179, "Border Gateway Protocol" ),
   "https": ( 443, "HTTP Secure (HTTPS)" ),
   "nfs": ( 2049, "Network File System" ),
   }

serviceMap = { IPPROTO_TCP: tcpServiceByName,
               IPPROTO_UDP: udpServiceByName,
               IPPROTO_SCTP: sctpServiceByName,
             }

tcpServiceByName.update( commonServiceByName_ )
udpServiceByName.update( commonServiceByName_ )

aptp = Tac.Type( "Arnet::PrivateTcpPorts" )

aristaTcpServiceByName = {
   "lanz" : ( aptp.lanzStreamingPort, "Lanz Streaming" ),
   "cvx" : ( aptp.cvxOobPort, "CVX" ),
   "cvx-cluster" : ( aptp.cvxClusterPort, "CVX Cluster" ),
   "cvx-inband" : ( aptp.controllerdbDefaultPort, "CVX Inband" ),
   "cvx-inband-ssl" : ( aptp.controllerdbDefaultSslPort, "CVX SSL" ),
   "switch-cvx-ssl" : ( aptp.sysdbDefaultSslPort, "Sysdb SSL listener for CVX" ),
   "eco-dhcp" : ( aptp.ecoDhcpPort, "ECO DHCP Fingerprint Streaming" ),
   "eco-ipfix" : ( aptp.ecoIpfixPort, "ECO IPFIX Record Streaming" )
}

tcpServiceByName.update( aristaTcpServiceByName )

def getServByName( proto, name ):
   try:
      return serviceMap[ proto ][ name ][ 0 ]
   except KeyError:
      return None

def getServiceMap( proto ):
   return serviceMap.get( proto )

# We can't access certain const defs in .tac, so redefine here
tcpFlagFin = 1
tcpFlagSyn = 2
tcpFlagRst = 4
tcpFlagPsh = 8
tcpFlagAck = 16
tcpFlagUrg = 32

tcpFlagTokens = dict( fin=tcpFlagFin,
                      syn=tcpFlagSyn,
                      rst=tcpFlagRst,
                      psh=tcpFlagPsh,
                      ack=tcpFlagAck,
                      urg=tcpFlagUrg )

def setTcpFlag( flag, flagName ):
   if flagName == 'fin':
      flag.fin = True
   elif flagName == 'syn':
      flag.syn = True
   elif flagName == 'rst':
      flag.rst = True
   elif flagName == 'psh':
      flag.psh = True
   elif flagName == 'ack':
      flag.ack = True
   elif flagName == 'urg':
      flag.urg = True
   else:
      assert False, "Unknown flag!"

def tcpFlag( flagName ):
   flag = TcpFlagValue()
   setTcpFlag( flag, flagName )
   return flag

def tcpFlags( flagNameList ):
   flag = TcpFlagValue()
   for flagName in flagNameList:
      setTcpFlag( flag, flagName )
   return flag

tcpFlagAll = tcpFlags( [ 'syn', 'fin', 'rst', 'ack', 'psh', 'urg' ] )

def tcpFlagFromString( token ):
   return tcpFlagTokens[ token ]

def tcpFlagsFromValue( flag, split=' ' ):
   # turn a TcpFlag into a string
   flags = [ k for k, v in tcpFlagTokens.iteritems( ) if flag & v ]
   return split.join( sorted( flags ) )

# helper function to compare and copy ACLs
def copyNonInstantiatingCollection( dest, source ):
   # first add/update new members
   for k, v in source.iteritems( ):
      if not k in dest or dest[ k ] != v:
         dest[ k ] = v

   # then delete non-existing members         
   if len( dest ) > len (source ):
      for k, v in dest.iteritems( ):
         if not k in source:
            del dest[ k ]
            if len( dest ) == len (source ):
               break

def collEq( c1, c2 ):
   if len( c1 ) != len( c2 ):
      return False
   for k, v in c1.iteritems( ):
      if not k in c2 or c2[ k ] != v:
         return False
   return True

def copyAclSubConfig( dest, source, aclType ):
   assert dest != source

   dest.countersEnabled = source.countersEnabled

   if aclType == 'ip':
      copyNonInstantiatingCollection( dest.ipRuleById,
                                      source.ipRuleById )
   elif aclType == 'ipv6':
      copyNonInstantiatingCollection( dest.ip6RuleById,
                                      source.ip6RuleById )
   else:
      assert aclType == 'mac'
      copyNonInstantiatingCollection( dest.macRuleById,
                                      source.macRuleById )      

   copyNonInstantiatingCollection( dest.remarkBySequence,
                                   source.remarkBySequence )

   copyNonInstantiatingCollection( dest.ruleBySequence, source.ruleBySequence )

def inverseMacMask( mask ):
   import Ethernet, re
   # inverse bits of a canonical mac address
   m = re.match( Ethernet.colonPattern, mask )
   assert m is not None
   words = [ ]
   for i in range( 6 ):
      words.append( int( m.group( i + 1 ), 16 ) )
   return "%02x:%02x:%02x:%02x:%02x:%02x" % (
      words[ 0 ] ^ 0xFF, words[ 1 ] ^ 0xFF, words[ 2 ] ^ 0xFF,
      words[ 3 ] ^ 0xFF, words[ 4 ] ^ 0xFF, words[ 5 ] ^ 0xFF )

def createIPAcl( config, aclName, aclType='ip' ):
   id0 = Tac.Value( 'Acl::UniqueId', 12345, 765, 0 )
   id1 = Tac.Value( 'Acl::UniqueId', 12345, 765, 1 )
   id2 = Tac.Value( 'Acl::UniqueId', 12345, 765, 2 )

   acl = config.config[ aclType ].newAcl( aclName,
         config.config[ aclType ].type, True, False )

   acl.newSubConfig( id0 )
   if aclType == 'ip':
      acl.subConfig[ id0 ].ipRuleById[ id1 ] = Tac.Value( 'Acl::IpRuleConfig' )
      acl.subConfig[ id0 ].ipRuleById[ id2 ] = Tac.Value( 'Acl::IpRuleConfig' )
   else:
      assert aclType == 'ipv6'
      acl.subConfig[ id0 ].ip6RuleById[ id1 ] = Tac.Value( 'Acl::Ip6RuleConfig' )
      acl.subConfig[ id0 ].ip6RuleById[ id2 ] = Tac.Value( 'Acl::Ip6RuleConfig' )
   acl.currCfg = config.config[ aclType ].acl[ aclName ].subConfig[ id0 ]
   acl.currCfg.ruleBySequence[ 0 ] = id1
   acl.currCfg.ruleBySequence[ 1 ] = id2
   return ( acl, id0, id1, id2 )

def payloadOptToPayloadSpec( payloadOpt=None, headerStart=None ):
   # Remove duplicate, sort and create payloadSpec
   payloadOptDict = {}
   if payloadOpt is not None:
      for payload in payloadOpt:
         if payload.mask != 0:
            payloadOptDict.update( { payload.offset : payload } )
   offsetPattern = ''
   list1 = []
   for offset in sorted(payloadOptDict.keys()):
      payload = payloadOptDict[ offset ]
      list1.append( payload.offset )
      list1.append( payload.pattern )
      list1.append( payload.mask )
      alias = payload.alias

      if payload.patternOverride:
         alias += PayloadStringConstants.overrideDelim + \
                  PayloadStringConstants.patternOverride
      if payload.maskOverride:
         alias += PayloadStringConstants.overrideDelim + \
                  PayloadStringConstants.maskOverride
      list1.append( alias )

   offsetPattern = ' '.join( str(e) for e in list1 ) 
   payloadSpec = PayloadSpec( offsetPattern )
   if headerStart:
      payloadSpec.headerStart = headerStart
   return payloadSpec

def metadataListToMetadataSpec( metadataList=None ):
   # Remove duplicate, sort by source and offset
   metadataDict = {}
   if metadataList is not None:
      for metadata in metadataList:
         if metadata.mask != 0:
            metadataDict.update( 
                  { ( metadata.metadataSource, metadata.offset ) : metadata } )
   offsetPattern = ''
   l = []
   for key in sorted( metadataDict.keys() ):
      metadata = metadataDict[ key ]
      l.append( metadata.metadataSource )
      l.append( metadata.offset )
      l.append( metadata.pattern )
      l.append( metadata.mask )

   offsetPattern = ' '.join( str( e ) for e in l )
   metadataSpec = MetadataSpec( offsetPattern )
   return metadataSpec

def _getFilter( proto, srcIp=None, dstIp=None,
                dport=None, dportOper='eq',
                sport=None, sportOper='eq',
                tracked=False, ttl=None,
                ipLen=None, ipLenOper='eq',aclType='ip' ):
   portSpecs = []
   for port, oper in [ ( dport, dportOper ), ( sport, sportOper ) ]:
      if port is not None:
         ports = []
         for p in port:
            serv = getServByName( proto, p )
            if serv is None:
               serv = p
            ports.append( str( serv ) )
         ports = ' '.join( ports )
         portSpec = PortValue( oper, ports )
      else:
         portSpec = anyPortValue
      portSpecs.append( portSpec ) 

   if ipLen is not None:
      ipLens = []
      for p in ipLen:
         ipLens.append( p )
      ipLens = ' '.join( ipLens )
      ipLenSpec = IpLenValue( ipLenOper, ipLens )
   else:
      ipLenSpec = anyIpLenValue
   if ttl is None:
      ttlSpec = anyTtlValue
   else:
      ttlSpec = TtlValue( 'eq', ttl )
   if aclType == 'ip':
      if srcIp is None:
         srcIp = anyIpAddr
      if dstIp is None:
         dstIp = anyIpAddr
      return IpFilterValue( proto=proto,
                            source=srcIp,
                            destination=dstIp,
                            innerSource=anyIpAddr,
                            innerDest=anyIpAddr,
                            innerSource6=anyIp6Addr,
                            innerDest6=anyIp6Addr,
                            dport=portSpecs[ 0 ],
                            sport = portSpecs[ 1 ],
                            tracked=tracked,
                            ttl=ttlSpec,
                            ipLen=ipLenSpec )
   else:
      if srcIp is None:
         srcIp = anyIp6Addr
      if dstIp is None:
         dstIp = anyIp6Addr
      return Ip6FilterValue( proto=proto,
                             source=srcIp,
                             destination=dstIp,
                             innerSource=anyIpAddr,
                             innerDest=anyIpAddr,
                             innerSource6=anyIp6Addr,
                             innerDest6=anyIp6Addr,
                             dport=portSpecs[ 0 ],
                             sport=portSpecs[ 1 ],
                             tracked=tracked,
                             ttl=ttlSpec )

def _addRule( acl, seqnum, action, proto, srcIp=None, dstIp=None,
              dport=None, dportOper='eq', sport=None, sportOper='eq',
              tracked=False, ttl=None,
              ipLen = None, ipLenOper='eq',aclType='ip' ):
   uniqueId = None
   if action == 'remark':
      # Currently there are no remarks in default config so this should never
      # hit. Adding this for reference if we decide to add default remarks later
      assert False, "Please add remark rule to default config"
      rule = RemarkConfigValue( remark='Some remark' )
      acl.remarkBySequence[ seqnum ] = rule
   else:
      ipFilter = _getFilter( proto, srcIp, dstIp, dport, dportOper, sport, sportOper,
                             tracked, ttl, ipLen, ipLenOper, aclType )
      uniqueId = genUniqueId( )
      if aclType == 'ip':
         rule = IpRuleValue( filter=ipFilter,
                             action=action,
                             log=False )
         acl.ipRuleById[ uniqueId ] = rule
      else:
         rule = Ip6RuleValue( filter=ipFilter,
                              action=action,
                              log=False )
         acl.ip6RuleById[ uniqueId ] = rule
      acl.ruleBySequence[ seqnum ] = uniqueId
   return uniqueId

def newServiceAclTypeVrfMap( config, serviceAclMap,
                             aclName, vrf=defaultVrf, aclType='ip',
                             proto=IPPROTO_TCP, port=646, defaultAction='deny',
                             sport=None, createAcl=False, standardAcl=False,
                             readonly=False, addToInputCol=False ):
   aclTypeConfig = config.config.newMember( aclType )
   aclConfig = aclTypeConfig.acl.get( aclName )
   if aclConfig is None and createAcl:
      if addToInputCol:
         config.config[ aclType ].acl.newMember( aclName, 
               config.config[ aclType ].type, standardAcl, readonly )
         aclConfig = config.config[ aclType ].acl[ aclName ]
      else:
         aclConfig = Tac.newInstance( "Acl::AclConfig", aclName, aclType,
                                      standardAcl, readonly )
         config.config[ aclType ].acl.addMember( aclConfig )
   # Create empty acl.currCfg
   setAclRules( aclConfig, [], aclType )
   key = Tac.Value( "Acl::AclTypeAndVrfName", aclType, vrf )
   serviceAclMap.aclName[ key ] = aclName
   return aclConfig

def newServiceAcl( aclConfig, aclCpConfig, serviceName, aclName, vrf=defaultVrf, 
                   aclType='ip', proto=IPPROTO_TCP, port=None, defaultAclName='',
                   defaultAction='deny', sport=None, createAcl=False,
                   standardAcl=False, readonly=False, tracked=False ):
   # proto and port are actually ignored
   cpConfig = aclCpConfig.cpConfig.newMember( aclType )
   config = aclConfig.config.newMember( aclType )
   serviceAcl = cpConfig.serviceAcl.newMember( vrf )
   service = serviceAcl.service.get( serviceName )
   if service is None:
      service = serviceAcl.service.newMember( serviceName, proto, vrf )
   if port:
      service.ports = ','.join( str( i ) for i in port )
   if sport:
      service.sports = ','.join( str( i ) for i in sport )
   service.tracked = tracked
   service.defaultAclName = defaultAclName
   service.defaultAction = defaultAction
   service.aclName = aclName
   acl = config.acl.get( aclName )
   if acl is None and createAcl:
      acl = config.newAcl( aclName, config.type, standardAcl, readonly )
   return acl

def setAclRules( acl, rules, aclType='ip', readonly=True, countersEnabled=True ):
   aclSubConfig = acl.newSubConfig( genUniqueId() )
   aclSubConfig.readonly = readonly
   aclSubConfig.countersEnabled = countersEnabled
   seqnum = 10
   ruleIds = []
   for rule in rules:
      ruleId = _addRule( aclSubConfig, seqnum, rule[ 'action' ], rule[ 'proto' ],
                  rule.get( 'srcIp' ), rule.get( 'dstIp' ),
                  rule.get( 'dport' ), rule.get( 'dportOper', 'eq' ),
                  rule.get( 'sport' ), rule.get( 'sportOper', 'eq' ),
                  False, rule.get( 'ttl' ),
                  rule.get( 'ipLen' ), rule.get( 'ipLenOper', 'eq' ),
                  aclType )
      seqnum += 10
      ruleIds.append( ruleId )
   acl.currCfg = aclSubConfig
   return ruleIds

def initializeDefaultCpAclName( entMan, cfg, paramCfg ):
   # pkgdeps: rpm Epoch-lib
   hwEpochStatus = entMan.lookup( "hwEpoch/status" )
   def _handleActive( active ):
      if not active or not hwEpochStatus.serviceAclEnabled:
         return
      # create and use open default cp acl only when service acl is used
      paramCfg.cpAclNameDefault = '' 
   entMan.registerActiveCallback( _handleActive )

def getAclTypeDisplayName( aclType ):
   return aclTypeDisplayNames.get( aclType ) 
