#!/usr/bin/env python
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Agent
import BothTrace
import Cell
import GenericReactor
import SharedMem
import Smash
import Tac
import os

bv = BothTrace.Var
bt5 = BothTrace.tracef5

class AclCvxClient( Agent.Agent ):
   # attribute-defined-outside-init
   # pylint: disable-msg=W0201
   def __init__( self, em ):
      Agent.Agent.__init__( self, em )

   def doInit( self, em ):
      bt5()
      mg = em.mountGroup()
      self.controllerClientConfig = mg.mount( "mgmt/controller/config",
                                                "ControllerClient::Config", "r" )
      self.status = mg.mount( "mgmt/controller/aclClient/status",
                              "AclCvxClient::Status", "w" )
      self.exportConfig = mg.mount( "mgmt/controller/policy/config",
                                 "AclCvxClient::ExportConfig", "r" )
      self.configDir = mg.mount( "acl/cvxacl/config/switch", "Tac::Dir", "ri" )
      self.applyConfigDir = mg.mount( "acl/cvxacl/applyConfig", "Tac::Dir", "ri" )
      self.controllerAclStatus = mg.mount( "acl/cvxacl/status",
                                             "ControllerAcl::AclStatusV1", "w" )
      self.aclConfig = mg.mount( "acl/config/input/cvxacl",
                                   "Acl::Input::Config", "wS" )
      self.aclIntfConfig = mg.mount( "acl/intf/config/input/cvxacl",
                                       "Acl::IntfConfig", "w" )
      self.aclParamConfig = mg.mount( "acl/paramconfig", "Acl::ParamConfig", "r" )
      self.aclHwConfig = mg.mount( "acl/hwconfig/cvxacl", "Acl::HwConfig", "w" )
      self.aclStatus = mg.mount( "acl/status/all", "Acl::Status", "r" )
      self.aclStatusDpDir = mg.mount( Cell.path( "acl/status/dp" ),
                                        "Tac::Dir", "ri" )
      self.tagConfig = mg.mount( "tag/config", "Tag::Config", "r" )
      self.bridgingConfig = mg.mount( "bridging/config", "Bridging::Config", "r" )
      self.ipStatus = mg.mount( "ip/status", "Ip::Status", "r" )
      self.mlagStatus = mg.mount( "mlag/status", "Mlag::Status", "r" )
      self.pcsExport = mg.mount( "pcs/switch/export", "Pcs::PcsSwitchExportV1", "w" )
      # mount arpSmash
      shmemEm = SharedMem.entityManager( sysdbEm=em )
      self.arpStatus = shmemEm.doMount( "arp/status", "Arp::Table::Status",
                                        Smash.mountInfo( 'shadow' ) )
      self.arpSmashVrfIdMap = shmemEm.doMount( 'vrf/vrfIdMapStatus',
                                               "Vrf::VrfIdMap::Status",
                                               Smash.mountInfo( 'shadow' ) )
      self.smashBridgingStatus = shmemEm.doMount( "bridging/status",
                                                  "Smash::Bridging::Status",
                                                  Smash.mountInfo( 'shadow' ) )
      self.bridgingStatus = Tac.newInstance( "Bridging::Status", "bridgingStatus" )
      self.bridgingShimSm = Tac.newInstance( "Bridging::ReaderFdbStatusSm",
                                             self.bridgingStatus,
                                             self.smashBridgingStatus )
      self.ethIntfStatusDir = self.createLocalEntity( "ethIntfStatusDir",
                                                      "Interface::EthIntfStatusDir",
                                                      "interface/status/eth/intf" )
      self.localEntitiesCreatedIs( True )

      def mountsComplete():
         self.root = self.agentRoot_.newEntity( "AclCvxClient::Root", "root" )
         # 11 is an arbitratay number chosen that's smaller than the other inputs
         # viz., EosSdk (50), PortMon (30) so that CVX's ACL is given higher priorty
         # but also less than CLI Acl input (10)
         # TODO : BUG359056
         self.aclIntfConfig.initialize( 11 )
         # Default config
         self.aclConfig.newConfig( 'ip' )
         self.aclConfig.newConfig( 'ipv6' )
         self.configReactor = GenericReactor.GenericReactor(
                                 self.controllerClientConfig, [ "enabled" ],
                                 self.handleEnabled, callBackNow=True )

      mg.close( mountsComplete )

   def handleEnabled( self, notifiee ):
      enabled = self.controllerClientConfig.enabled
      bt5( "enabled: ", bv( enabled ) )
      if enabled:
         aclMounts = Tac.newInstance( "AclCvxClient::AclMounts",
                        self.aclConfig, self.aclParamConfig, self.aclIntfConfig,
                        self.aclHwConfig, self.aclStatus, self.aclStatusDpDir,
                        self.controllerAclStatus )
         self.root.aclMediator = ( self.configDir, self.applyConfigDir, aclMounts,
                                   self.ethIntfStatusDir )
         self.root.exportSm = ( self.tagConfig,
                                self.exportConfig, self.bridgingConfig,
                                self.ipStatus, self.arpStatus, self.bridgingStatus,
                                self.mlagStatus, self.pcsExport,
                                self.arpSmashVrfIdMap )
      else:
         if self.root.aclMediator:
            self.root.aclMediator.doCleanup()
            self.root.aclMediator = None
         if self.root.exportSm:
            self.root.exportSm.doCleanup()
            self.root.exportSm = None

      self.flushEntityLog( self.handleFlushComplete )

   def handleFlushComplete( self, timedOut=False ):
      assert not timedOut, "EntityLog flush timed out"
      enabled = self.controllerClientConfig.enabled
      bt5( "handleFlushComplete: enabled=", bv( enabled ) )
      self.status.enabled = enabled

def main():
   qtfile = "AclCvxClient%s.qt" % ( "-%d" if "QUICKTRACEDIR" not in os.environ
                                       else "" )
   BothTrace.initialize( qtfile, "8,128,8,8,8,128,8,8,128,8", maxStringLen=80 )
   container = Agent.AgentContainer( [ AclCvxClient ], passiveMount=True )
   container.runAgents()

if __name__ == "__main__":
   main()
