#!/usr/bin/env python
# Copyright (c) 2009, 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac, AclLib, WaitForWarmup
import Ethernet
from eunuchs.in_h import IPPROTO_UDP, IPPROTO_TCP, IPPROTO_IP, IPPROTO_ICMP
from eunuchs.in_h import IPPROTO_ICMPV6
from AclLib import ( ICMP_ALL, ICMP6_ALL, defaultVrf, EcnValue, 
                     newServiceAcl, CopyToCpuDst )
import Tracing

th = Tracing.Handle( "AclCliLib" )
t0 = th.trace0

# Halo computations tend me lot more time consuming, and as a temporary
# stop-gap, we have a much higher than previous timeout of 60s
hwWaitTimeout = 180

# dscp list => name: ( value, description )
dscpAclNames = {
   'af11' : ( 10, 'AF11 dscp (001010)' ),
   'af12' : ( 12, 'AF12 dscp (001100)' ),
   'af13' : ( 14, 'AF13 dscp (001110)' ),
   'af21' : ( 18, 'AF21 dscp (010010)' ),
   'af22' : ( 20, 'AF22 dscp (010100)' ),
   'af23' : ( 22, 'AF23 dscp (010110)' ),
   'af31' : ( 26, 'AF31 dscp (011010)' ),
   'af32' : ( 28, 'AF32 dscp (011100)' ),
   'af33' : ( 30, 'AF33 dscp (011110)' ),
   'af41' : ( 34, 'AF41 dscp (100010)' ),
   'af42' : ( 36, 'AF42 dscp (100100)' ),
   'af43' : ( 38, 'AF43 dscp (100110)' ),
   'cs0' : ( 0, 'Default dscp (000000)' ),
   'cs1' : ( 8, 'CS1(precedence 1) dscp (001000)' ),
   'cs2' : ( 16, 'CS2(precedence 2) dscp (010000)' ),
   'cs3' : ( 24, 'CS3(precedence 3) dscp (011000)' ),
   'cs4' : ( 32, 'CS4(precedence 4) dscp (100000)' ),
   'cs5' : ( 40, 'CS5(precedence 5) dscp (101000)' ),
   'cs6' : ( 48, 'CS6(precedence 6) dscp (110000)' ),
   'cs7' : ( 56, 'CS7(precedence 7) dscp (111000)' ),
   'ef' : ( 46, 'EF dscp (101110)' ),
   }

# We lose the numeric value for enums in Python, so we just initialize
# the maps from this ordered list.
ecnAclNamesOrderedList = (
   ( 'dontCare', EcnValue.dontCare ),
   ( 'ect-ce', EcnValue.ectCe ),
   ( 'non-ect', EcnValue.nonEct ),
   ( 'ce', EcnValue.ce ),
   ( 'ect', EcnValue.ect ),
)

ecnAclNames = { x[ 0 ]: x[ 1 ] for x in ecnAclNamesOrderedList }

# the maps from copy to CPU destination to name
copyToCpuDestinations = {
   'none': CopyToCpuDst.cpuDstNone,
   'captive-portal': CopyToCpuDst.cpuDstCaptivePortal, 
}

# generic IP protocols (only support generic IP CLI)
genericIpProtocols = {
   'ospf': ( AclLib.IPPROTO_OSPF, 'OSPF routing protocol' ),
   'vrrp': ( AclLib.IPPROTO_VRRP, 'Virtual Router Redundancy Protocol (VRRP)' ),
   'pim': ( AclLib.IPPROTO_PIM, 'Protocol Independent Multicast (PIM)' ),
   'igmp': ( AclLib.IPPROTO_IGMP, 'Internet Group Management Protocol (IGMP)' ),
   'ahp':  ( AclLib.IPPROTO_AHP, 'Authentication Header Protocol' ),
   'rsvp': ( AclLib.IPPROTO_RSVP, 'Resource Reservation Protocol (RSVP)' ),
   }

# generic IPv6 protocols (only support generic IPv6 CLI)
genericIp6Protocols = {
   'ospf': ( AclLib.IPPROTO_OSPF, 'OSPF routing protocol' ),
   'vrrp': ( AclLib.IPPROTO_VRRP, 'Virtual Router Redundancy Protocol (VRRP)' ),
   'rsvp': ( AclLib.IPPROTO_RSVP, 'Resource Reservation Protocol (RSVP)' ),
   'pim': ( AclLib.IPPROTO_PIM, 'Protocol Independent Multicast (PIM)' ),
   }

# ICMP message { token: ( helpdesc, type, code ) }
# This is a long list.
icmpMessages = {
   "echo-reply": ( "Echo reply", 0, ICMP_ALL ),
   "unreachable": ( "All destination unreachables", 3, ICMP_ALL ),
   "net-unreachable": ( "Net unreachable", 3, 0 ),
   "host-unreachable": ( "Host unreachable", 3, 1 ),
   "protocol-unreachable": ( "Protocol unreachable", 3, 2 ),
   "port-unreachable": ( "Port unreachable", 3, 3 ),
   "packet-too-big": ( "Fragmentation needed but DF was set", 3, 4 ),
   "source-route-failed": ( "Source route failed", 3, 5 ),
   "network-unknown": ( "Network unknown", 3, 6 ),
   "host-unknown": ( "Host unknown", 3, 7 ),
   "host-isolated": ( "Source host isolated", 3, 8 ),
   "dod-net-prohibited": ( "Communication with network prohibited", 3, 9 ),
   "dod-host-prohibited": ( "Communication with host prohibited", 3, 10 ),
   "net-tos-unreachable": ( "Network unreachable for type of service",  3, 11 ),
   "host-tos-unreachable": ( "Host unreachable for type of service", 3, 12 ),
   "administratively-prohibited": ( "Communication administratively prohibited",
     3, 13 ),
   "host-precedence-unreachable": ( "Host precedence violation", 3, 14 ),
   "precedence-unreachable": ( "Precedence cutoff in effect", 3, 15 ),
   "source-quench": ( "Source quench", 4, ICMP_ALL ),
   "redirect": ( "All redirects", 5, ICMP_ALL ),
   "net-redirect": ( "Network redirect", 5, 0 ),
   "host-redirect": ( "Host redirect", 5, 1 ),
   "net-tos-redirect": ( "Network and type of service redirect", 5, 2 ),
   "host-tos-redirect": ( "Host and type of service redirect", 5, 3 ),
   "alternate-address": ( "Alternate host address", 6, ICMP_ALL ),
   "echo": ( "Echo", 8, ICMP_ALL ),
   "router-advertisement": ( "Router advertisement", 9, ICMP_ALL ),
   "router-solicitation": ( "Router solicitation", 10, ICMP_ALL ),
   "time-exceeded": ( "All time exceeded messages", 11, ICMP_ALL ),
   "ttl-exceeded": ( "Time to live exceeded in transit", 11, 0 ),
   "reassembly-timeout": ( "Fragment reassembly time exceeded", 11, 1 ),
   "parameter-problem": ( "All parameter problems", 12, ICMP_ALL ),
   "general-parameter-problem": ( "General parameter problem", 12, 0 ),
   "option-missing": ( "Missing a required option", 12, 1 ),
   "no-room-for-option": ( "Bad length for parameter", 12, 2 ),
   "timestamp-request": ( "Timestamp requests", 13, ICMP_ALL ),
   "timestamp-reply": ( "Timestamp replies", 14, ICMP_ALL ),
   "information-request": ( "Information requests", 15, ICMP_ALL ),
   "information-reply": ( "Information replies", 16, ICMP_ALL ),
   "mask-request": ( "Address mask request", 17, ICMP_ALL ),
   "mask-reply": ( "Address mask replies", 18, ICMP_ALL ),
   "traceroute": ( "Traceroute", 30, ICMP_ALL ),
   "conversion-error": ( "Datagram conversion error", 31, ICMP_ALL ),
   "mobile-host-redirect": ( "Mobile host redirect", 32, ICMP_ALL )
   }

# ICMPV6 message { token: ( helpdesc, type, code ) }
icmp6Messages = {
   "unreachable": ( "All destination unreachables", 1, ICMP6_ALL ),
   "no-route": ( "No route to destination", 1, 0 ),
   "no-admin": ( "Administration prohibited destination", 1, 1 ),
   "beyond-scope": ( "Beyond scope of source address", 1, 2 ),
   "address-unreachable": ( "Address unreachable", 1, 3 ),
   "port-unreachable": ( "Port unreachable", 1, 4 ),
   "source-address-failed": ( "Source address failed ingress/egress policy",
      1, 5 ),
   "reject-route": ( "Reject route to destination", 1, 6 ),
   "source-routing-error": ( "Error in source routing header", 1, 7 ),
   "packet-too-big": ( "Packet too big", 2, ICMP6_ALL ),
   "time-exceeded": ( "All time exceeded", 3, ICMP6_ALL ),
   "hop-limit-exceeded": ( "Hop limit exceeded in transit", 3, 0 ),
   "fragment-reassembly-exceeded" : ( "Fragment reassembly time exceeded", 3, 1 ),
   "parameter-problem": ( "Parameter problem", 4, ICMP6_ALL ),
   "erroneous-header": ( "All erroneous header field encountered", 4, 0 ),
   "unrecognized-next-header": ( "Unrecognized next header type encountered",
      4, 1 ),
   "unrecognized-ipv6-option": ( "Unrecognized IPv6 option encountered", 4, 2 ),
   "echo-request": ( "Echo request", 128, ICMP6_ALL ),
   "echo-reply": ( "Echo reply", 129, ICMP6_ALL ),
   #"mld-query": ( "Multicast listener query", 130, ICMP6_ALL ),
   #"mld-report": ( "Multicast listener report", 131, ICMP6_ALL ),
   #"mld-done": ( "Multicast listener done", 132, ICMP6_ALL ),
   "router-solicitation": ( "Router solicitation", 133, ICMP6_ALL ),
   "router-advertisement": ( "Router advertisement", 134, ICMP6_ALL ),
   "neighbor-solicitation": ( "Neighbor solicitation", 135, ICMP6_ALL ),
   "neighbor-advertisement": ( "Neighbor advertisement", 136, ICMP6_ALL ),
   "redirect-message": ( "Redirect message", 137, ICMP6_ALL ),
   #"renum": ( "All router renumbering", 138, ICMP6_ALL ),
   #"renum-command": ( "Router renumbering command", 138, 0 ),
   #"renum-result": ( "Router renumbering result", 138, 1 ),
   #"sequence-number-reset": ( "Sequence number reset", 138, 255 ),
   #"icmp-node-query": ( "All ICMP node information queries", 139, ICMP6_ALL ),
   #"icmp-node-subject-query": ( "The data field containts an IPv6 address which \
         #is the subject of this query", 139, 0 ),
   #"icmp-node-query-noop": ( "The data field contains a name which is the \
         #subject of this query, or is empty, as in the case of a NOOP",
      #139, 1 ),
   #"icmp-node-query-ipv4": ( "The data field contains an IPv4 address which \
         #is the subject of this query", 139, 2 ),
   #"icmp-node-response": ( "All ICMP node information responses", 140, ICMP6_ALL ),
   #"icmp-node-response-success": ( "A Successful reply", 140, 0 ),
   #"icmp-node-response-refuse": ( "The Responde refuses to supply the answer",
      #140, 1 ),
   #"icmp-node-response-unknown": ( "The Qtype of the Query is unknown to the \
         #Responder", 140, 1 ),
   #"inverse-nd-solicitation": ( "Inverse Neighbor Discovery Solicitation Message",
      #141, ICMP6_ALL ),
   #"inverse-nd-advertise": ( "Inverse Neighbor Discovery Advertisement Message",
         #142, ICMP6_ALL ),
   #"mldv2-reports": ( "Multicast Learner Discovery (MLDv2) reports", 143,
         #ICMP6_ALL ),
   #"ha-ad-request": ( "Home Agent Address Discovery Request Message", 144,
         #0 ),
   #"ha-ad-reply": ( "Home Agent Address Discovery Reply Message", 145,
         #0 ),
   #"mp-solicit": ( "Mobile Prefix Solicitation", 146, 0 ),
   #"mp-advertise": ( "Mobile Prefix Advertisement", 147, 0 ),
   #"cert-path-solicitation": ( "Certificate Path Solicitation", 148, ICMP6_ALL ),
   #"cert-path-advertise": ( "Certificate Path Advertisement", 149, ICMP6_ALL ),
   #"fmipv6-message": ( "FMIPv6 Messages", 150, ICMP6_ALL ),
   #"mcast-router-advertise": ( "Multicast Router Advertisement", 151, ICMP6_ALL ),
   #"mcast-router-solicit": ( "Multicast Router Solicitation", 152, ICMP6_ALL ),
   #"mcast-router-terminate": ( "Multicast Router Termination", 153, ICMP6_ALL ),
   #"rpl-control-message": ( "RPL Control Message", 155, ICMP6_ALL )
   }

noHwWarningMsg = "Hardware not present. ACL(s) not programmed in the hardware."

def findNameByValue( value, nameMap ):
   # Given a { name : value } map do a reverse lookup to find the name.
   # This isn't exactly fast, so use it in non-performance critical code
   # if the map is big (tests).
   for k, v in nameMap.iteritems( ):
      if isinstance( v, tuple ):
         if v[ 0 ] == value:
            return k
      elif v == value:
         return k
   return str( value )

def dscpValueFromCli( mode, dscpVal ):
   dscpResult = dscpAclNames.get( dscpVal, False )
   return ( dscpResult[ 0 ], True ) if dscpResult else ( dscpVal, False )

def ecnValueFromCli( mode, ecnVal ):
   return ecnAclNames[ ecnVal ]

def ecnNameFromValue( ecnValue ):
   return findNameByValue( ecnValue, ecnAclNames )

def dscpNameFromValue( dscpValue ):
   return findNameByValue( dscpValue, dscpAclNames )

def copyToCpuDstFromCli( mode, copyToCpuDst ):
   return copyToCpuDestinations.get( copyToCpuDst, CopyToCpuDst.cpuDstNone )

def copyToCpuDstFromValue( cpuDstValue ):
   return findNameByValue( cpuDstValue, copyToCpuDestinations )

def portNumberFromString( proto, port ):
   # from a string return the numeric port number
   serviceByName = AclLib.serviceMap.get( proto )
   if serviceByName and port in serviceByName:
      return serviceByName[ port ][ 0 ]
   return int( port )

# IP protocols
ipProtoByName = { 'ip': IPPROTO_IP,
                  'icmp': IPPROTO_ICMP,
                  'ospf': AclLib.IPPROTO_OSPF,
                  'vrrp': AclLib.IPPROTO_VRRP,
                  'pim': AclLib.IPPROTO_PIM,
                  'igmp': AclLib.IPPROTO_IGMP,
                  'ahp': AclLib.IPPROTO_AHP,
                  'udp': IPPROTO_UDP,
                  'tcp': IPPROTO_TCP,
                  'gre': AclLib.IPPROTO_GRE,
                  'mpls-over-gre': AclLib.IPPROTO_MPLSOVERGRE,
                  'rsvp': AclLib.IPPROTO_RSVP }

def ipProtoFromValue( proto ):
   # from proto value return a string
   return findNameByValue( proto, ipProtoByName )

def ipProtoFromString( proto ):
   # from proto string return a value
   v = ipProtoByName.get( proto )
   if v is not None:
      return v
   return int( proto )

# IPv6 protocols
ip6ProtoByName = { 'ipv6': IPPROTO_IP,
                   'icmpv6': IPPROTO_ICMPV6,
                   'ospf': AclLib.IPPROTO_OSPF,
                   'udp': IPPROTO_UDP,
                   'tcp': IPPROTO_TCP,
                   'sctp': AclLib.IPPROTO_SCTP,
                   'mpls-over-gre': AclLib.IPPROTO_MPLSOVERGRE,
                   'rsvp': AclLib.IPPROTO_RSVP,
                   'pim': AclLib.IPPROTO_PIM }

def ip6ProtoFromString( proto ):
   # from proto string return a value
   v = ip6ProtoByName.get( proto )
   if v is not None:
      return v
   return int( proto )

def payloadStringToValue( payloadString ):
   # returns list of (offset, pattern, mask, alias, patternOverride, maskOverride)
   # tuples from payload string
   if( payloadString == '' ):
      return []
   values = payloadString.split( AclLib.PayloadStringConstants.fieldDelim )
   payloadList = []
   i = 0
   while( i < len(values) ):
      offset, pattern, mask, alias = values[ i:i+4 ]
      aliasParts = alias.split( AclLib.PayloadStringConstants.overrideDelim )
      alias = aliasParts[ 0 ]
      patternOverride = False
      maskOverride = False
      for override in aliasParts[ 1: ]:
         if override == AclLib.PayloadStringConstants.patternOverride:
            patternOverride = True
         if override == AclLib.PayloadStringConstants.maskOverride:
            maskOverride = True
      payloadList.append( [ int( offset ), long( pattern ), long( mask ), alias,
                            patternOverride, maskOverride ] )
      i = i + 4
   return sorted( payloadList )

def ipLenFromString( inputString ):
   # return a Acl::IpLenSpec from a string
   if not inputString:
      return AclLib.anyIpLenValue
   ipLens = inputString.split( ' ' )
   return AclLib.IpLenValue( ipLens[ 1 ],
                             ' '.join( ipLens[ 2: ] ) )

def portFromString( proto, inputString ):
   # return a Acl::PortSpec from a string
   if not inputString:
      return AclLib.anyPortValue
   ports = inputString.split( ' ' )
   return AclLib.PortValue( ports[ 0 ],
                            ' '.join( str( portNumberFromString( proto, x ) )
                                      for x in ports[ 1: ] ) )

def remarkFromValue( remarkConfig ):
   # from a remark config, return the original CLI command string
   return "remark " + remarkConfig.remark

def ruleFromValue( rule, aclType, standard=False, convert=True ):
   # from a rule value, return the original CLI command string
   return rule.ruleStr( standard, convert )

# Error messages
def unsuppRulesWarning( name, aclType, errMsg="" ):
   if aclType == 'ipv6':
      return "Warning: ACL %s contains rules not supported in data-plane.\n" % \
            name + "Unsupported options will be ignored by hardware.\n" + errMsg
   else:
      return "Warning: ACL %s contains rules not supported in data-plane.\n" % \
         name + "Unsupported options will be ignored by hardware.\n" + errMsg

def notConfiguredError( aclType, aclName, intfName ):
   return "A different %s access-list is configured on %s%s." % (
      aclType, '' if 'control-plane' in intfName or 'service' in intfName \
         else 'interface ', intfName )

def dpiRulesWarning( aclName, aclType ):
   return "ACL %s contains deep inspection rules which " \
          "require deep inspection payload skip value to be 0.\n" % aclName + \
          "Unsupported options ( nvgre, gre, vxlan, gtp, mpls " \
          "matches ) will be ignored by hardware."

def extendedAclWarning():
   return 'Only source, destination and log are used in extended ACL'

def errWithReason( msg, reason="" ):
   if reason:
      return "%s (%s)" % ( msg, reason )
   else:
      return msg      

def aclSessionCommitError( error ):
   return errWithReason( 
      "Error in ACL commit, configuration may differ from hardware",
      error )

def aclTimeoutWarning():
   return "The ACL configuration is still being programmed into hardware. "\
       "The system might be busy for a while."

def aclCommitError( name, aclType, error="" ):
   return errWithReason( "Error: Cannot commit %s ACL %s" % (
      aclType, name ), error )

def aclCreateError( name, aclType, error="" ):
   return errWithReason( "Error: Cannot create %s ACL %s" % (
      aclType, name ), error )

def aclModifyError( name, aclType, error="" ):
   return errWithReason( "Error: Cannot modify %s ACL %s" % (
      aclType, name ), error )

def intfAclConfigError( name, aclType, intf, error="" ):
   return errWithReason( "Error: Cannot apply %s ACL %s to %s" % (
     aclType,  name, intf ), error )

def intfAclDeleteError( name, aclType, intf, error="" ):
   return errWithReason( "Error: Cannot remove %s ACL %s from %s" % (
      aclType, name, intf ), error )

def aclTypeError( name, aclType ):
   return "%s access list with this name already exists" % aclType

def aclTcamStatus( aclStatusDp, switchName ):
   for v in aclStatusDp.itervalues():
      acl = v.tcamStatus.get( switchName )
      if acl:
         return acl
   return None  
def dpAclVlanInterfaceOk( aclType, aclIntfTypeConfig, aclConfig, status, intf=None ):
   '''Determine whether the specified ACL can be supported on this
   interface, based on interface type and vlan filter.
   '''
   for ruleConfig in aclConfig.currCfg.ipRuleById.itervalues():
      if ruleConfig.filter.vlan or ruleConfig.filter.vlanMask:
         if not status.dpAclVlanSupported:
            return False
         if intf is not None:
            if intf.startswith( 'Vlan' ):
               return False
         else:
            for i in aclIntfTypeConfig.intf.itervalues():
               intfs = [ intf for intf, name in i.intf.iteritems()
                         if name == aclConfig.name ]
               for intf in intfs:
                  if intf.startswith( "Vlan" ):
                     return False
   return True

def dpAclOk( aclType, aclIntfTypeConfig, aclConfig, status ):
   '''Determine whether the specified ACL can be supported by the
   data plane, based on the status reported by the data plane.
   '''

   if aclType == 'ipv6':
      if not status.dpIp6AclSupported:
         return False
   else:
      if not status.dpAclSupported:
         return False

   for ruleConfig in aclConfig.currCfg.ipRuleById.itervalues():
      if ruleConfig.mirrorSession and not status.dpIpAclMirrorActionSupported:
         return False
      if ruleConfig.log and not status.dpAclLoggingSupported:
         return False
      if ( ruleConfig.copyToCpuDst != CopyToCpuDst.cpuDstNone and 
           not status.dpIngressCopyToCpuSupported ):
         return False
      if ruleConfig.filter.tracked:
         return False
      if not ( ruleConfig.filter.ttl.oper == 'any' or
               ruleConfig.filter.ttl.oper == 'eq' ):
         return False
      if ruleConfig.filter.innerVlanMask and not status.dpAclInnerVlanSupported:
         return False
      if ruleConfig.filter.greProtoMask and ruleConfig.filter.greProto == \
             AclLib.NVGRE_PROTO and not status.dpAclNvgreSupported:
         return False
      elif ruleConfig.filter.greProtoMask and \
             not status.dpAclGreSupported:
         return False
      if ruleConfig.filter.tniMask and not status.dpAclNvgreSupported:
         return False
      if ruleConfig.filter.vxlanValid and not status.dpAclVxlanSupported:
         return False
      if ruleConfig.filter.vniMask and not status.dpAclVxlanSupported:
         return False
      if ruleConfig.filter.gtpVersion and not status.dpAclGtpSupported:
         return False
      if ruleConfig.filter.teidMask and not status.dpAclGtpSupported:
         return False
      if ruleConfig.filter.userL4Mask and not status.dpAclUserL4Supported:
         return False
      if ruleConfig.filter.payloadOpt.offsetPattern != '' and \
               not status.dpAclPayloadSupported:
         return False
      if ( status.dpAclTcpFlagSupported.value !=
           ( ruleConfig.filter.tcpFlag.value |
             status.dpAclTcpFlagSupported.value ) ) :
         return False
      
      for i in aclIntfTypeConfig.intf.itervalues():
         if not status.dpRouterAclLoggingSupported and \
                ruleConfig.log and \
                aclConfig.name in i.intf.values():
            intfs = [ intf for intf, name in i.intf.iteritems()
                      if name == aclConfig.name ]
            for intf in intfs:
               if intf.startswith( "Vlan" ):
                  return False

      if aclType == 'ip':
         if ruleConfig.filter.matchDscp and not status.dpAclDscpSupported:
            return False
         if ( ruleConfig.filter.matchDscp and \
              not status.dpAclDscpMaskSupported and \
              ruleConfig.filter.dscpMask ^ 0x3F > 0 ):
            return False
      elif aclType == 'ipv6':
         if ruleConfig.filter.tcMask and not status.dpAclTcSupported:
            return False
         if ruleConfig.filter.tcMask and \
            not status.dpAclDscpMaskIpv6Supported and \
            ( ( ruleConfig.filter.tcMask >> 2 ) ^ 0x3F ) > 0:
            return False
 
      if not status.dpAclNonContiguousIpMaskSupported:
         maskSrc = ruleConfig.filter.source.mask
         maskDst = ruleConfig.filter.destination.mask
         # verify that masks are contiguous
         if ( ( maskSrc | ( maskSrc - 1 ) ) & 0xffffffff != ( -1 & 0xffffffff )  or \
              ( maskDst | ( maskDst - 1 ) ) & 0xffffffff != ( -1 & 0xffffffff ) ):
            return False

      if ruleConfig.filter.ecn != EcnValue.dontCare \
      and not status.dpAclEcnSupported:
         return False 
      if ruleConfig.filter.ipLen.oper != 'any' and \
             not status.dpAclIpLenMatchSupported:
         return False
   
   for ruleConfig in aclConfig.currCfg.ip6RuleById.itervalues():
      if ruleConfig.mirrorSession and not status.dpIpv6AclMirrorActionSupported:
         return False
      if ruleConfig.log and not status.dpAclLoggingSupported:
         return False
      if ( ruleConfig.copyToCpuDst != CopyToCpuDst.cpuDstNone and
           not status.dpIngressCopyToCpuSupported ):
         return False
      if ruleConfig.filter.innerVlanMask and not status.dpAclInnerVlanSupported:
         return False
      if ruleConfig.filter.greProtoMask and ruleConfig.filter.greProto == \
             AclLib.NVGRE_PROTO and not status.dpAclNvgreSupported:
         return False
      elif ruleConfig.filter.greProtoMask and \
             not status.dpAclGreSupported:
         return False
      if ruleConfig.filter.tniMask and not status.dpAclNvgreSupported:
         return False
      if ruleConfig.filter.vxlanValid and not status.dpAclVxlanSupported:
         return False
      if ruleConfig.filter.vniMask and not status.dpAclVxlanSupported:
         return False
      if ruleConfig.filter.gtpVersion and not status.dpAclGtpSupported:
         return False
      if ruleConfig.filter.teidMask and not status.dpAclGtpSupported:
         return False
      if ruleConfig.filter.userL4Mask and not status.dpAclUserL4Supported:
         return False
      if ruleConfig.filter.payloadOpt.offsetPattern != '' and \
               not status.dpAclPayloadSupported:
         return False
      if ( status.dpAclTcpFlagSupported.value !=
           ( ruleConfig.filter.tcpFlag.value |
             status.dpAclTcpFlagSupported.value ) ) :
         return False

   for ruleConfig in aclConfig.currCfg.macRuleById.itervalues():
      if ruleConfig.mirrorSession and not status.dpMacAclMirrorActionSupported:
         return False
      if ruleConfig.log and not status.dpAclLoggingSupported:
         return False
      if ruleConfig.filter.innerVlanMask and not status.dpAclInnerVlanSupported:
         return False
      if ruleConfig.filter.mplsLabelMask and not status.dpAclMplsTwoLabelSupported:
         return False
      if ruleConfig.filter.mplsInnerLabelMask and \
         not status.dpAclMplsTwoLabelSupported:
         return False
      if len( ruleConfig.filter.mplsFilter.label ) and not status.dpAclMplsSupported:
         return False
      if ruleConfig.filter.payloadOpt.offsetPattern != '' and \
               not status.dpAclPayloadSupported:
         return False

   return True

def getVrfNames( allVrfConfig ):
   vrfNames = allVrfConfig.vrf.members() + [ defaultVrf ]
   return vrfNames

def getAclNames( config, aclType ):
   return config.config[ aclType ].acl.keys()

def setServiceAclTypeVrfMap( mode, serviceAclConfig,
                             aclName, aclType='ip', vrfName=None,
                             force=False ):
   if not vrfName:
      vrfName = defaultVrf
   aclTypeAndVrfName = Tac.Value( 'Acl::AclTypeAndVrfName', aclType, vrfName )
   if not aclName and not force:
      del serviceAclConfig.aclName[ aclTypeAndVrfName ]
   else:
      serviceAclConfig.aclName[ aclTypeAndVrfName ] = aclName
   if mode:
      tryWaitForWarmup( mode )

def noServiceAclTypeVrfMap( mode, serviceAclConfig,
                            aclName, aclType='ip', vrfName=None ):
   if not vrfName:
      vrfName = defaultVrf
   aclTypeAndVrfName = Tac.Value( 'Acl::AclTypeAndVrfName', aclType, vrfName )
   name = serviceAclConfig.aclName.get( aclTypeAndVrfName, "" )
   if aclName is None or name == aclName:
      del serviceAclConfig.aclName[ aclTypeAndVrfName ]
      if mode:
         tryWaitForWarmup( mode )
   else:
      mode.addWarning( notConfiguredError( aclType, aclName,
                  'service %s(%s VRF)' % ( serviceAclConfig.name, vrfName ) ) )

def checkServiceAcl( mode, config, aclName, aclType='ip' ):
   if aclName:
      acl = config.config[ aclType ].acl.get( aclName )
      if acl and not acl.standard and mode:
         # mode can be None is called from CpAclTests
         mode.addWarning( extendedAclWarning() )

def setServiceAcl( mode, service, proto, config, cpConfig,
                   aclName, aclType='ip', vrfName=None,
                   port=None, sport=None, defaultAction='deny', tracked=False ):
   if not vrfName:
      vrfName = defaultVrf
   newServiceAcl( config, cpConfig, service, aclName, vrf=vrfName, aclType=aclType,
                  proto=proto, port=port, defaultAction=defaultAction, sport=sport,
                  tracked=tracked )
   if mode:
      tryWaitForWarmup( mode )

def noServiceAcl( mode, service, config, cpConfig, 
                  aclName, aclType='ip', vrfName=None ):
   if not vrfName:
      vrfName = defaultVrf
   serviceAclVrfConfig = cpConfig.cpConfig[ aclType ].serviceAcl.get( vrfName )
   if serviceAclVrfConfig:
      serviceConfig = serviceAclVrfConfig.service.get( service )
      if serviceConfig:
         if aclName is None or serviceConfig.aclName == aclName:
            if not serviceConfig.defaultAclName:
               # delete only when there is no default service Acl applied
               del serviceAclVrfConfig.service[ service ]
               if not serviceAclVrfConfig.service:
                  del cpConfig.cpConfig[ aclType ].serviceAcl[ vrfName ]
         elif mode:
            # mode can be None is called from CpAclTests
            mode.addError( notConfiguredError(
                  aclType, aclName, 'service %s(%s VRF) ' % ( service, vrfName ) ) )
      if mode:
         tryWaitForWarmup( mode )

def tryWaitForWarmup( mode ):
   # throttles Cli configuration so we do not change too fast
   # as for large ACLs it may take significant time for the agent to process
   try:
      WaitForWarmup.wait( mode.entityManager,
                          agentsToGrab=[ 'Acl' ],
                          timeout=300.0, sleep=True,
                          verbose=False )
   except Tac.Timeout:
      pass

def isRemarkConfig( remarkConfig ):
   return type( remarkConfig ) == Tac.Type( 'Acl::RemarkConfig' )

def numAclRules( aclSubConfig ):
   return len( aclSubConfig.ruleBySequence ) + len( aclSubConfig.remarkBySequence ) 

def sortedSequenceNumbers( aclSubConfig ):
   rules = aclSubConfig.ruleBySequence.keys() + aclSubConfig.remarkBySequence.keys()
   rules.sort()
   return rules

def getRuleById( subconfig, aclType ):
   return getattr( subconfig, aclType.lower().replace( "v", "" ) + 'RuleById' )

def getRuleValue( subConfig, seq, aclType, standard=False, convert=True ):
   return subConfig.ruleStr( seq, aclType, standard, convert )

def mergeAclTypeStatus( fromStatus, toStatus ):
   '''
   Merge Acl::AclTypeStatus object fromStatus into toStatus.
   First we merge AclStatus collection, then we merge AclIntf collection.
   '''
   for status in [ fromStatus, toStatus ]:
      assert status.__class__.__name__ == "Acl::AclTypeStatus"

   def mergeRuleStatus( ruleStatus1, ruleStatus2 ):
      return Tac.Value( "Acl::RuleStatus",
                        ruleStatus1.pkts + ruleStatus2.pkts,
                        max( ruleStatus1.lastChangedTime,
                             ruleStatus2.lastChangedTime ) )

   def copyRuleStatus( ruleStatus ):
      return Tac.Value( "Acl::RuleStatus", ruleStatus.pkts,
                        ruleStatus.lastChangedTime )
   
   for aclName, fromAclStatus in fromStatus.acl.iteritems():
      if not aclName in toStatus.acl:
         toStatus.acl.newMember( aclName )
      toAclStatus = toStatus.acl[ aclName ]

      for ruleId, ruleStatus in fromAclStatus.ruleStatus.iteritems():
         if ruleId in toAclStatus.ruleStatus:
            toAclStatus.ruleStatus[ ruleId ] = mergeRuleStatus(
               ruleStatus, toAclStatus.ruleStatus[ ruleId ] )
         else:
            toAclStatus.ruleStatus[ ruleId ] = copyRuleStatus( ruleStatus )

      for ruleId, ruleStatus in fromAclStatus.connRuleStatus.iteritems():
         if ruleId in toAclStatus.connRuleStatus:
            toAclStatus.connRuleStatus[ ruleId ] = mergeRuleStatus(
               ruleStatus, toAclStatus[ ruleId ] )
         else:
            toAclStatus.connRuleStatus[ ruleId ] = copyRuleStatus( ruleStatus )

      toAclStatus.counterUpdateTime = max(
         toAclStatus.counterUpdateTime, fromAclStatus.counterUpdateTime )
      toAclStatus.version = fromAclStatus.version
      toAclStatus.countersIncomplete = fromAclStatus.countersIncomplete
      if toAclStatus.noRuleMatches.pkts == 0:
         toAclStatus.noRuleMatches = copyRuleStatus(
            fromAclStatus.noRuleMatches )
      else:
         toAclStatus.noRuleMatches = mergeRuleStatus(
            toAclStatus.noRuleMatches, fromAclStatus.noRuleMatches )
      if toAclStatus.noConnRuleMatches.pkts == 0:
         toAclStatus.noConnRuleMatches = copyRuleStatus(
            fromAclStatus.noConnRuleMatches )
      else:
         toAclStatus.noConnRuleMatches = mergeRuleStatus(
            toAclStatus.noConnRuleMatches, fromAclStatus.noConnRuleMatches )

   for direction, fromAclIntf in fromStatus.intf.iteritems():
      if not direction in toStatus.intf:
         toAclStatus.intf.newMember( direction )
      toAclIntf = toStatus.intf[ direction ]
      for intfId, intfStr in fromAclIntf.intf.iteritems():
         toAclIntf.intf[ intfId ] = intfStr

def aggregateServiceAclStatus( serviceAclStatus, aggAclType=None ):
   ''' Create an aggregated serviceAcl status entity to pass it onto ACL API
       ServiceAclStatus is maintained per VRF to avoid Multi-writer problem
       with gated but Checkpoint is not maintained per VRF. ACL Cli APIs
       won't work with these different ways of maintaining information in
       ServiceAclStatus and Checkpoint. To Circumvent this problem
       ServiceAclStatus across various VRFs has been merged.
   '''
   if aggAclType is not None:
      assert aggAclType in [ "ip", "ipv6" ]
      
   aggServiceAclStatus = Tac.newInstance( 'Acl::StatusService', 'merge' )
   for vrf in serviceAclStatus:
      vrfAclStatus = serviceAclStatus[ vrf ]
      assert vrfAclStatus.__class__.__name__ == "Acl::StatusService"
      for aclType in vrfAclStatus.status:
         if aggAclType is not None and aclType != aggAclType:
            continue
         if aclType not in aggServiceAclStatus.status:
            aggServiceAclStatus.status.newMember( aclType )
         mergeAclTypeStatus( vrfAclStatus.status[ aclType ],
                             aggServiceAclStatus.status[ aclType ] )

      for aclTypeVrfName in vrfAclStatus.aclVersion:
         aggServiceAclStatus.aclVersion[ aclTypeVrfName ] = \
            vrfAclStatus.aclVersion[ aclTypeVrfName ]

      for aclTypeVrfName in vrfAclStatus.defaultAction:
         aggServiceAclStatus.action[ aclTypeVrfName ] = \
            vrfAclStatus.action[ aclTypeVrfName ]

   return aggServiceAclStatus

# initialize ancillary data structures in AclCliLib
def initialize():
   aclNameMaps = Tac.singleton( "Acl::AclNameMaps" )
   if aclNameMaps.initialized:
      return

   t0( "initialize() starts" )
   # IP Protocols
   for name, proto in ipProtoByName.iteritems():
      aclNameMaps.ipProto[ proto ] = name
   # IP6 Protocols
   for name, proto in ip6ProtoByName.iteritems():
      aclNameMaps.ip6Proto[ proto ] = name
   # L3 service names
   for l3proto, smap in AclLib.serviceMap.iteritems():
      srvMap = aclNameMaps.service.newMember( l3proto )
      for name, service in smap.iteritems():
         srvMap.map[ service[ 0 ] ] = name
   # ICMP
   for name, icmp in icmpMessages.iteritems():
      aclNameMaps.icmp[ ( icmp[ 1 ] << 16 ) | icmp[ 2 ] ] = name
   for name, icmp in icmp6Messages.iteritems():
      aclNameMaps.icmp6[ ( icmp[ 1 ] << 16 ) | icmp[ 2 ] ] = name

   # DSCP names
   for name, dscp in dscpAclNames.iteritems():
      aclNameMaps.dscp[ dscp[ 0 ] ] = name
   # ECN names
   for i, ecn in enumerate( ecnAclNamesOrderedList ):
      # i is the actual enum value
      aclNameMaps.ecn[ i ] = ecn[ 0 ]
   # GTP protocols
   for name, proto in AclLib.gtpProtoByName.iteritems():
      aclNameMaps.gtpProto[ proto[ 0 ] ] = name
   # TCP flags
   for name, value in AclLib.tcpFlagTokens.iteritems():
      aclNameMaps.tcpFlag[ Tac.Value( "Acl::TcpFlagIndex",
                                      value.bit_length() - 1 ) ] = name
   # MAC protocols
   for name, value in Ethernet.macProtoByName.iteritems():
      aclNameMaps.macProto[ value[ 0 ] ] = name

   aclNameMaps.initialized = True
   t0( "initialize() ends" )

