# Copyright (c) 2008, 2009, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import re
import weakref

from AaaDefs import AAA_PWENT_RESULT
import Arnet.NsLib
import Cell
import CliCommon
from IpLibConsts import DEFAULT_VRF
import PyClient
from Tracing import traceX
import Tac

# Valid keys of the attribute dictionary returned by a plugin's authorizeShell
# and authorizeShellCommand methods.  These are all optional -- a plugin is
# free to return an empty dictionary.
privilegeLevel = "privilegeLevel"
timeout = "timeout"
idleTime = "idletime"
autoCmd = "autocmd"
roles = "roles"
secureMonitor = CliCommon.SECURE_MONITOR_ATTR_NAME

# Recommended usage of QuickTrace levels:
#
# 0: Errors
# 1: Warnings
# 2: Authentication requests
# 3: Authorization requests
# 4: Accounting requests
# 5: Sessions
# 6: General information
# 9: Debug messages
#
# Note we don't use 8 or 9 and Aaa agent initializes those levels with a small value.
TR_ERROR = 0
TR_WARN = 1
TR_AUTHEN = 2
TR_AUTHZ = 3
TR_ACCT = 4
TR_SESSION = 5
TR_INFO = 6
TR_DEBUG = 9

class Authenticator( object ):
   """Handles a single authentication session."""
   def __init__( self, aaaConfig, method, type, service=None, remoteHost=None,
                 remoteUser=None, tty=None, user=None ):
      self.aaaConfig = aaaConfig
      self.method = method
      self.type = type
      self.service = service
      self.remoteHost = remoteHost
      self.remoteUser = remoteUser
      self.tty = tty
      self.user = user
      self.logFallback = True

   def authenticate( self, *responses ):
      """Called by Aaa repeatedly until it returns a result with a status
      other than 'inProgress'.  Returns a dict containing these elements:
        "status" : AaaApi::AuthenStatus
        "messages" : a sequence of AaaApi::AuthenMessage instances, maybe empty
        "user" : username string
        "authToken" : authentication token string (ie. password)
      where the AuthenMessage instances define messages/prompts for display to
      the user."""
      return { 'status' : 'fail', 'messages' : [] }

   """The promptUser/Password are internal styles that help Aaa to keep track of
   username and password in case of fallback. They will be converted to
   promptEchoOn/Off before writing to TACC or passing on to pam. Please use
   these functions when you ask for username or password in an authenticator."""
   def userPrompt( self ):
      return Tac.Value( "AaaApi::AuthenMessage",
                        style='promptUser', text=self.aaaConfig.userPrompt )

   def passwordPrompt( self ):
      return Tac.Value( "AaaApi::AuthenMessage",
                        style='promptPassword', text=self.aaaConfig.passwordPrompt )

   def unknownUserError( self ):
      return Tac.Value( "AaaApi::AuthenMessage",
                        style='error', text='Bad user' )

class BasicUserAuthenticator( Authenticator ):
   """This class implements a basic authenticator which authenticates with username
      and password"""
   # state machine values
   needUser = 1
   askedUser = 2
   needPassword = 3
   askedPassword = 4
   failed = 5
   readyToSucceed = 6 # only happens for no-password user provided as ctor arg
   unknownUser = 7    # only happens for unknown users provided as ctor arg
   succeeded = 8

   stateMap = { 1 : "needUser", 2 : "askedUser", 3 : "needPassword",
                4 : "askedPassword", 5 : "failed", 6 : "readyToSucceed",
                7 : "unknownUser", 8 : "succeeded" }

   def __init__( self, aaaConfig, method, type, service, remoteHost, remoteUser, 
                 tty, user, privLevel ):
      self.privLevel = privLevel
      self.password = ""
      self.sessionData = None
      Authenticator.__init__( self, aaaConfig, method, type, service,
                              remoteHost, remoteUser, tty, user )
      if self.user:
         traceX( TR_AUTHEN, "creating UserAuthenticator with pre-populated user",
                 self.user )
         if not self.checkUser( self.user ):
            traceX( TR_ERROR, "user", self.user, "is unknown" )
            self.state = self.unknownUser
         else:
            result = self.checkEmptyPassword( self.user )
            if result[ 'state' ] == self.succeeded:
               traceX( TR_AUTHEN, "user", self.user, "has empty password" )
               self.state = self.readyToSucceed
               self.sessionData = result.get( 'sessionData' )
            else:
               self.state = self.needPassword
      else:
         self.state = self.needUser

   def checkUser( self, user ):
      """Check if a user exists - if not, it can fail without asking
      for password. LocalUser uses this method to return False when the
      user is not present, which triggers returning 'unavailable' to
      Aaa and falling back to the next available method. This is the IOS
      behavior.

      The default is to return True and let checkPassword() handle the
      rest."""
      return True

   def checkEmptyPassword( self, user ):
      """This is used to speculatively authenticate a user without the
      password. If it fails, the authenticator will ask the user again
      for password. It returns a dictionary that can be fed to the
      transition() function. The default is to fail and let checkPassword()
      handle the rest.
      """
      return { 'state': self.failed, 'authenStatus': 'fail' }

   def checkPassword( self, user, password ):
      """The real authentication happens here; password could be None
      to attempt authenticattion without password.
      This function should return a dictionary that can be fed to the
      transition() function.
      """
      raise Exception( "no checkPassword function provided!" )

   def authenticate( self, *responses ):
      s = self.state
      if s == self.needUser:
         traceX( TR_AUTHEN, "authenticate: state=needUser, prompting for username " )
         return self.transition( self.askedUser, 'inProgress',
                                 [ self.userPrompt() ] )
      elif s == self.needPassword:
         traceX( TR_AUTHEN,
                 "authenticate: state=needPassword, prompting for password" )
         return self.transition( self.askedPassword, 'inProgress',
                                 [ self.passwordPrompt() ], self.user )
      elif s == self.askedUser:
         if len( responses ) > 0:
            self.user = responses[0]
            traceX( TR_AUTHEN, "authenticate: state=askedUser, response=",
                    self.user )
            if not self.user:
               # empty user, go back to asking for user
               return self.transition( self.askedUser, 'inProgress',
                                       [ self.userPrompt() ] )
            if self.checkUser( self.user ):
               result = self.checkEmptyPassword( self.user )
               if result[ 'state' ] == self.succeeded:
                  return self.transition( **result )
               return self.transition( self.askedPassword,
                                       'inProgress', [ self.passwordPrompt() ],
                                       self.user )
            else:
               # BUG13784: fallback to next method if local authentication
               # finds unknown user.
               return self.transition( self.failed, 'unavailable', 
                                       [ self.unknownUserError() ],
                                       self.user )
         else:
            traceX( TR_ERROR, "authenticate: state=askedUser, no response!" )
            # just fall through to the fail case... this shouldn't happen
      elif s == self.askedPassword:
         if len( responses ) > 0:
            self.password = responses[0]
            traceX( TR_AUTHEN, "authenticate: state=askedPassword" )
            result = self.checkPassword( self.user, self.password )
            return self.transition( **result )
         else:
            traceX( TR_ERROR, "authenticate: state=askedPassword, no response!" )
      elif s == self.readyToSucceed:
         traceX( TR_AUTHEN, "authenticate: state=readyToSucceed" )
         return self.transition( self.succeeded, 'success', [], self.user,
                                 self.password, self.sessionData )
      elif s == self.unknownUser:
         traceX( TR_AUTHEN, "authenticate: state=unknownUser" )
         return self.transition( self.failed, 'unavailable',
                                 [ self.unknownUserError() ],
                                 self.user )
      else:
         traceX( TR_ERROR, "authenticate: unexpected state:", self.stateMap[ s ] )
         # just fall through to the fail case... this shouldn't happen

      return self.transition( self.failed, 'fail' )

   def transition( self, state, authenStatus, messages=[], user="",
                   authToken="", sessionData=None ):
      sm = self.stateMap
      traceX( TR_AUTHEN, "UserAuthenticator transitioning from", sm[ self.state ],
              "to", sm[ state ] )
      self.state = state
      r = { "status" : authenStatus, "messages" : messages, "user" : user,
            "authToken" : authToken }
      if sessionData is not None:
         r[ 'sessionData' ] = sessionData
      return r

class Plugin( object ):
   """Base class for AaaPlugins."""
   def __init__( self, aaaConfig, name, authenMethods=None, allVrfStatusLocal=None ):
      """authenMethods can be a tuple of regular expressions that will match
      the authentication methods supported by this plugin.  The Aaa agent will
      first attempt an exact match on the plugin name, and if no plugin matches
      by name, Aaa will call the handlesAuthenMethod function on each loaded
      plugin until one matches."""
      self.aaaConfig = aaaConfig
      self.allVrfStatusLocal = allVrfStatusLocal
      self.name = name
      if authenMethods is not None:
         t = type( authenMethods )
         assert t == tuple or t == list
         authenMethods = [ re.compile( am ) for am in authenMethods ]
      self.authenMethods = authenMethods

   def getNsFromVrf( self, vrf ):
      if vrf == DEFAULT_VRF:
         return Arnet.NsLib.DEFAULT_NS

      vrf = self.allVrfStatusLocal.vrf.get( vrf )
      if not vrf:
         return None

      if vrf.state != 'active':
         return None

      return vrf.networkNamespace

   def ready( self ):
      """Returns True if the plugin is ready to handle requests, otherwise
      False.  A plugin that requires Sysdb configuration to be active
      (eg. a server address) would not be ready as long as the configuration
      is not present or invalid."""
      return False

   def logFallback( self ):
      """When the plugin returns 'unavailable', whether generate a syslog
      of AAA_AUTHN_FALLBACK or AAA_AUTHZ_FALLBACK. The default is yes, but
      for LocalUser it's no."""
      return True

   def handlesAuthenMethod( self, method ):
      """Returns True if the plugin supports the specified authentication
      method."""
      if method == self.name:
         return True
      if self.authenMethods:
         for am in self.authenMethods:
            match = re.match( am, method )
            if match is not None:
               return True
      return False

   def _handlesGroupAuthenMethod( self, method, typeName, groupType ):
      if method in ( typeName, "group " + typeName ):
         return True
      groupname = extractGroupFromMethod( method )
      if groupname:
         try:
            hg = self.aaaConfig.hostgroup[ groupname ]
            if hg.groupType != groupType or len( hg.member ) == 0:
               return False
            # ignore any configured servers for which the VRF is not valid
            return any( self.getNsFromVrf( m.vrf ) is not None \
                  for m in hg.member.itervalues() )
         except KeyError:
            return False
      return False

   def createAuthenticator( self, method, type, service, remoteHost, remoteUser,
                            tty, user=None, privLevel=0 ):
      """Returns a new Authenticator instance for the specified method.  This
      plugin will only be called with methods that handlesAuthenMethod claims
      that this plugin supports."""
      raise NotImplementedError()

   def openSession( self, authenticator ):
      """Called when authentication has succeeded and the application has
      decided to initiate a session.  The Authenticator that authenticated the
      user is passed as an argument.  Note that it is possible for
      authentication to succeed but no session to be started for a variety
      of reasons.  Optionally returns a token that will be passed to
      closeSession()."""
      pass

   def closeSession( self, token ):
      """Invoked when a session closes, with the token returned by openSession
      as an argument."""
      pass

   def authorizeShell( self, method, user, session ):
      """Returns a tuple where the first element is a boolean indicating if
      the user is authorized to start a shell, and the second is a message
      (which may be empty) to display to the user.  Arguments:
          method: the name of the authorization method
            user: the username
         session: the session id"""
      raise NotImplementedError()

   def authorizeShellCommand( self, method, user, session, mode, privlevel, tokens ):
      """Returns a tuple where the first element is a boolean indicating if
      the user is authorized to start a shell, and the second is a message
      (which may be empty) to display to the user.  Arguments are:
           method: the name of the authentication method
             user: the username
          session: the session id 
             mode: a tuple including the name of the mode, modeKey and logModeKey
        privlevel: the session's current privilege level
           tokens: a list of tokens that compose the command."""
      raise NotImplementedError()

   def sendCommandAcct( self, method, user, session, privlevel,
                              timestamp, tokens ):
      """Returns a tuple where the first element is a integer ( enum )
      indicating the result of accounting (see AcctStatus in AaaApi.tac),
      and the second is a message (which may be empty).
      Arguments are:
           method: the name of the authentication method
             user: the username
          session: the session id 
        privlevel: the session's current privilege level
        timestamp: the timestamp
           tokens: a list of tokens that compose the command."""
      raise NotImplementedError()

   def invalidateSessionPool( self, hostgroup ):
      """Some plugins (Tacacs/Radius) use a session pool for caching
      remote server credentials. This method is used to invalidate the
      pool when Aaa configuration changes."""
      pass

   def hasUserShell( self ):
      """The plugin may define users with custom shell."""
      return False

   def getUserShell( self, username ):
      """Get shell of the user. Plugins whose hasUserShell() return
      True should define this function. If this function returns None,
      Aaa will assign a default shell."""
      return None

   def getPwEnt( self, username ):
      """The Aaa agent calls this function to query plugins for its knowledge of a
      user that has never logged in. There are three different return values:

      1. If the user exists, return an "AaaApi::PasswdEntry" for this user.
         UID and GID can be left empty.
      2. If the user doesn't exist (e.g., non-existent user for local plugin),
         return AAA_PWENT_RESULT.INVALID.
      3. If the plugin is not sure if the user exists (e.g., Tacacs/Radius),
         return AAA_PWENT_RESULT.UNKNOWN."""
      return AAA_PWENT_RESULT.INVALD

class Session( object ):
   def __init__( self, hostgroup ):
      self.hostgroup_ = hostgroup
      self.time_ = Tac.now()

class HostGroupSessionPool( object ):
   # this holds all sessions for a certain host group
   # operation should be done under mutex protection
   def __init__( self, name, maxSize ):
      self.name_ = name
      self.maxSize_ = maxSize
      self.pool_ = []

   def get( self ):
      if self.pool_:
         traceX( TR_DEBUG, "returning session from pool", self.name_ )
         session = self.pool_.pop()
         # update the session timestamp so it's not discarded
         # when it's put back due to previous clear operations.
         session.time_ = Tac.now()
         return session
      else:
         traceX( TR_DEBUG, "pool", self.name_, "is empty" )
         return None

   def put( self, session ):
      assert self.name_.endswith( session.hostgroup_ )
      if len( self.pool_ ) < self.maxSize_:
         traceX( TR_DEBUG, "accepting session into pool", self.name_ )
         self.pool_.append( session )
      else:
         traceX( TR_ERROR, "pool", self.name_, "is full, rejecting session" )
         session.close()
         del session

   def clear( self ):
      traceX( TR_WARN, "clear", len( self.pool_ ), "sessions from pool", self.name_ )
      for s in self.pool_:
         s.close()
      self.pool_ = []

class SessionPool( object ):
   """A thread-safe pool of Session objects that can be reused.
   Retaining Session objects allows TCP connections to a server to be
   maintained, and for other protocol state to persist across uses.

   The session pool is indexed by hostgroup name, so we do not use the
   wrong session if multiple hostgroups are used. The pool is also 
   invalidated when hostgroup or server configuration changes.
   """
   def __init__( self, name, maxSize=5 ):
      import threading
      self.mutex_ = threading.Lock()
      self.name_ = name
      self.maxSize_ = maxSize
      self.pools_ = {}
      self.lastClearTime_ = 0

   def get( self, hostgroup ):
      """Retrieve a Session from the pool"""
      self.mutex_.acquire()
      try:
         pool = self.pools_.get( hostgroup )
         if pool:
            return pool.get()
      finally:
         self.mutex_.release()

   def put( self, session ):
      """Return a session to the pool"""
      self.mutex_.acquire()
      try:
         # we should reject the session if it's acquired before the last
         # clear time.
         if session.time_ < self.lastClearTime_:
            traceX( TR_AUTHEN, "discarding", self.name_, "session from pool",
                    session.hostgroup_, "as it's invalidated" )
            return
         if session.hostgroup_ not in self.pools_:
            self.pools_[ session.hostgroup_ ] = \
                HostGroupSessionPool( "%s::%s" % ( self.name_, 
                                                   session.hostgroup_ ),
                                      self.maxSize_ )
         pool = self.pools_[ session.hostgroup_ ]
         pool.put( session )
      finally:
         self.mutex_.release()

   def clear( self, hostgroup=None ):
      """Clears the pool.  This should be called whenever config such as 
      hostgroup changes so that sessions configured with the old configuration 
      will not be used in the future."""
      self.mutex_.acquire()
      try:
         # sessions in-fly (not in pool) will be invalidated by this
         self.lastClearTime_ = Tac.now()
         if hostgroup is not None:
            pool = self.pools_.get( hostgroup )
            if pool:
               pool.clear()
               del self.pools_[ hostgroup ]
         else:
            # clear everything
            for pool in self.pools_.itervalues():
               pool.clear()
            self.pools_.clear()
      finally:
         self.mutex_.release()

def extractGroupFromMethod( method ):
   match = re.match( "group (.+)", method )
   if match:
      groupname = match.group( 1 )
      return groupname
   return None

def serversForMethodName( method, aaaConfig, groupType ):
   servers = [ ]
   groupname = extractGroupFromMethod( method )
   if groupname:
      try:
         hg = aaaConfig.hostgroup[ groupname ]
         if ( hg.groupType == groupType ):
            for m in hg.member.itervalues():
               servers.append( m.spec )
      except KeyError:
         pass
   return ( groupname, servers )

HostSpec = Tac.Type( "Aaa::HostSpec" )

class IpIntfStatusReactor( Tac.Notifiee ):
   notifierTypeName = "Ip::IpIntfStatus"
   def __init__( self, notifier, configReactorRef ):
      Tac.Notifiee.__init__( self, notifier )
      self.configReactor_ = configReactorRef()
      self.invalidateSessionPool()

   def invalidateSessionPool( self ):
      self.configReactor_.invalidateSessionPool()

   @Tac.handler( 'activeAddrWithMask' )
   def handleActiveAddrWithMask( self ):
      traceX( TR_INFO, "Source interface activeAddrWithMask changed" )
      self.invalidateSessionPool()

   def close( self ):
      self.invalidateSessionPool()
      Tac.Notifiee.close( self )

class Ip6IntfStatusReactor( Tac.Notifiee ):
   notifierTypeName = "Ip6::IntfStatus"
   def __init__( self, notifier, configReactorRef ):
      Tac.Notifiee.__init__( self, notifier )
      self.configReactor_ = configReactorRef()
      self.invalidateSessionPool()

   def invalidateSessionPool( self ):
      self.configReactor_.invalidateSessionPool()

   @Tac.handler( 'addr' )
   def handleAddr( self, collKey ):
      traceX( TR_INFO, "Source interface addr changed" )
      self.invalidateSessionPool()

   def close( self ):
      self.invalidateSessionPool()
      Tac.Notifiee.close( self )

class ConfigReactor( Tac.Notifiee ):
   notifierTypeName = "AaaPlugin::Config"

   def __init__( self, notifier, status,
                 allVrfStatusLocal, ipStatus, ip6Status=None ):
      Tac.Notifiee.__init__( self, notifier )
      self.status_ = status
      self.ipStatus_ = ipStatus
      self.ip6Status_ = ip6Status
      self.ipStatusReactor_ = None
      self.ip6StatusReactor_ = None
      self.allVrfStatusLocal_ = allVrfStatusLocal
      self.vrfStatusReactor_ = Tac.collectionChangeReactor(
         self.allVrfStatusLocal_.vrf, VrfStatusReactor, reactorArgs=( self, ) )
      self.vrfStatusReactor_.inCollection = True
      self.checkpoint_ = hasattr( status, 'checkpoint' )
      self.hostReactor_ = Tac.collectionChangeReactor( notifier.host,
                                                       HostReactor,
                                                       reactorArgs=\
                                                          ( weakref.ref( self ), ) )
      self.handleHostEntryAll()

   def invalidateSessionPool( self ):
      raise NotImplementedError(
         "This method must be implemented by the derived class" )

   def handleHostEntryAll( self ):
      for h in self.notifier_.host:
         if not h in self.status_.counter:
            v = Tac.Value( self.counterTypeName )
            self.status_.counter[ h ] = v
            if self.checkpoint_:
               self.status_.checkpoint[ h ] = v
      if len( self.notifier_.host ) != len( self.status_.counter ):
         for h in self.status_.counter.keys():
            if not h in self.notifier_.host:
               del self.status_.counter[ h ]
               if self.checkpoint_:
                  del self.status_.checkpoint[ h ]

   def handleHostEntry( self, hostspec=None ):
      traceX( TR_INFO, self.notifierTypeName, "host changed" )
      self.invalidateSessionPool()
      # make sure counter entry is in sync
      if hostspec:
         if hostspec in self.notifier_.host:
            v = Tac.Value( self.counterTypeName )
            self.status_.counter[ hostspec ] = v
            if self.checkpoint_:
               self.status_.checkpoint[ hostspec ] = v
         else:
            del self.status_.counter[ hostspec ]
            if self.checkpoint_:
               del self.status_.checkpoint[ hostspec ]
      else:
         self.handleHostEntryAll()

   @Tac.handler( 'key' )
   def handleKey( self ):
      traceX( TR_INFO, self.notifierTypeName, "key changed" )
      self.invalidateSessionPool()

   @Tac.handler( 'timeout' )
   def handleTimeout( self ):
      traceX( TR_INFO, self.notifierTypeName, "timeout changed" )
      self.invalidateSessionPool()

   def closeIpStatusReactors( self ):
      if self.ipStatusReactor_:
         self.ipStatusReactor_.close()
         self.ipStatusReactor_ = None
      if self.ip6StatusReactor_:
         self.ip6StatusReactor_.close()
         self.ip6StatusReactor_ = None

   @Tac.handler( 'srcIntfName' )
   def handleSrcIntfName( self, vrfName=DEFAULT_VRF ):      
      traceX( TR_INFO, self.notifierTypeName, "source interface changed" )
      self.closeIpStatusReactors()
      if self.notifier_.srcIntfName:
         reactorFilter = lambda x: x in self.notifier_.srcIntfName.values()
         self.ipStatusReactor_ = Tac.collectionChangeReactor(
            self.ipStatus_.ipIntfStatus, IpIntfStatusReactor,
            reactorArgs=( weakref.ref( self ), ),
            reactorFilter=reactorFilter )
         self.ipStatusReactor_.inCollection = True
         if self.ip6Status_:
            self.ip6StatusReactor_ = Tac.collectionChangeReactor(
               self.ip6Status_.intf, Ip6IntfStatusReactor,
               reactorArgs=( weakref.ref( self ), ),
               reactorFilter=reactorFilter )
            self.ip6StatusReactor_.inCollection = True
      self.invalidateSessionPool()

   def handleVrfState( self, vrfName ):
      """If a plugin implements persistent connections and has VRF
      support, those connections may need to be re-evaluated when
      VRF-related state changes.  This method is used to notify the
      plugin about such changes.  Notifications can be ignored if the
      plugin does not support VRFs or persistent connections."""
      raise NotImplementedError(
         "This method must be implemented by the derived class" )

   def close( self ):
      self.closeIpStatusReactors()
      self.vrfStatusReactor_.close()
      self.vrfStatusReactor_ = None
      Tac.Notifiee.close( self )

class HostReactor( Tac.Notifiee ):
   notifierTypeName = "AaaPlugin::Host"

   def __init__( self, notifier, configReactorRef ):
      Tac.Notifiee.__init__( self, notifier, filtered=False )
      self.configReactor_ = configReactorRef()

   def onAttribute( self, attr, key ):
      self.configReactor_.invalidateSessionPool()

class CounterConfigReactor( Tac.Notifiee ):
   notifierTypeName = "AaaPlugin::CounterConfig"

   def __init__( self, notifier, status ):
      Tac.Notifiee.__init__( self, notifier )
      self.status_ = status

   def clearCounters( self ):
      for h in self.status_.counter:
         self.status_.counter[ h ] = Tac.Value( self.counterTypeName )
      self.status_.lastClearTime = Tac.now()

   @Tac.handler( 'clearCounterRequestTime' )
   def handleClearCounterRequestTime( self ):
      """Handle clear counter request from the Cli"""
      self.clearCounters( )

class VrfStatusReactor( Tac.Notifiee ):
   notifierTypeName = 'Ip::VrfStatusLocal'
   
   def __init__( self, vrfStatusLocal, master ):
      self.master_ = master
      self.vrfName = vrfStatusLocal.name
      self.vrfStatusLocal = vrfStatusLocal
      Tac.Notifiee.__init__( self, vrfStatusLocal )
      # we need to know about create/delete as well
      self.master_.handleVrfState( self.vrfName )

   @Tac.handler( 'state' )
   def handleState( self ):
      self.master_.handleVrfState( self.vrfName )

   def close( self ):
      assert self.vrfStatusLocal.state == Tac.Type( "Ip::VrfState" ).deleting
      self.master_.handleVrfState( self.vrfName )
      Tac.Notifiee.close( self )

def flushAcctQueue( sysname, waitTime, pyClient=None ):
   """Wait for the accounting queue to drain out with a timeout."""
   if pyClient == None:
      pyClient = PyClient.PyClient( sysname, "Aaa",  connectTimeout=waitTime )
   
   def _queueEmpty( ):
      pyClient.execute( "import Aaa" )
      output = pyClient.eval( "Aaa.acctQueue.empty()" )
      return output
   
   if waitTime == 0:
      return _queueEmpty()
   
   try:
      Tac.waitFor( _queueEmpty, warnAfter=0, timeout=waitTime, sleep=True )
      # Note, there is a bug: the queue being empty does not mean the message 
      # has been sent - it just means it's been dequeued. Ideally we should 
      # use acctQueue.join() here to wait for the task_done, but join() has no 
      # timeout. Hopefully the message would be sent out before reload
      # takes effect.
      return True
   except Tac.Timeout:
      return False
   
def sendShutDownSystemEvent( sysname, timeout ):
   deviceType = Cell.cellType()
   if deviceType in [ 'fixed', 'generic', 'supervisor' ]:
      pc = PyClient.PyClient( sysname , "Aaa", connectTimeout=timeout )
      pc.execute( "import Aaa; Aaa.agent.sendSystemAcct('shutdown','stop')" )
   else:
      pc = None
      
   return flushAcctQueue( sysname, timeout, pyClient=pc )
