# Copyright (c) 2008-2011, 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import AaaPluginLib
from AaaPluginLib import TR_ERROR, TR_WARN, TR_AUTHEN, TR_AUTHZ, TR_ACCT, TR_INFO
from BothTrace import traceX as bt
from BothTrace import Var as bv
import Cell
import Logging
import ReversibleSecretCli
import Tac
import Tacacs
from TacacsLib import CounterCallback
from Tracing import traceX

import libtacplus
from IpLibConsts import DEFAULT_VRF

# Trigger dependency on IpLib for 'Ira::IraIpStatusMounter' reference
# pkgdeps library IpLib

Logging.logD( id="AAA_INVALID_TACACS_SERVER",
              severity=Logging.logWarning,
              format="TACACS+ server '%s' port %d could not be used: %s",
              explanation="The configuration contains a TACACS+ server that "
                          "could not be used.  One cause for this type of "
                          "error is a hostname for which DNS resolution "
                          "fails.",
              recommendedAction="Correct the TACACS+ server configuration." )

Logging.logD( id="AAA_UNKNOWN_TACACS_SERVER",
              severity=Logging.logWarning,
              format="TACACS+ server group '%s' references unknown server '%s'",
              explanation="The TACACS+ server group contains a server that "
                          "was not configured using the 'tacacs-server host' "
                          "command.",
              recommendedAction="Configure the unknown server or remove it "
                                "from the server group." )

Logging.logD( id="AAA_NO_VALID_TACACS_SERVERS",
              severity=Logging.logError,
              format="No valid TACACS+ servers for method list '%s'",
              explanation="The configuration contains an authentication "
                          "method list that is not associated with any valid "
                          "TACACS+ servers. One common cause for this error is "
                          "a hostname for which DNS resolution fails.",
              recommendedAction="Correct the TACACS+ server configuration." )

Logging.logD( id="AAA_UNKNOWN_TACACS_MANDATORY_ATTRIBUTE",
              severity=Logging.logError,
              format="Received unknown mandatory attribute '%s' from Tacacs+ server",
              explanation="The authorization response from tacacs server includes"
                          "one or more mandatory attributes we donot support;"
                          "failing authorization.",
              recommendedAction="Change the TACACS+ server configuration.")

_knownAttributes = set( [ "priv-lvl", "priv_lvl", "autocmd", "timeout", 
                          "idletime", "cvp-roles", "roles", "secure-monitor" ] )

# ConfigReactor (below) listens for changes to Tacacs::Config and clears
# _sessionPool whenever the config changes to avoid the situation where a
# pooled session with outdated configuration data could be used.
_sessionPool = AaaPluginLib.SessionPool( 'tacacs' )

def _invalidateSessionPool( hostgroup=None ):
   bt( TR_WARN, 'invalidate TACACS+ session pool' )
   _sessionPool.clear( hostgroup )

class Authenticator( AaaPluginLib.Authenticator ):
   # state machine values
   sendAuthen = 1
   sendContinue = 2
   failed = 3
   succeeded = 4

   stateMap = { 1 : "sendAuthen", 2 : "sendContinue", 3 : "failed",
                4 : "succeeded" }

   def __init__( self, plugin, method, type, service,
                 remoteHost, remoteUser, tty, user, privLevel,
                 maxUsernameLength ):
      traceX( TR_AUTHEN, 'Tacacs Authenticator service', service,
              'remoteHost', remoteHost, 'remoteUser', remoteUser, 'tty', tty,
              'user', user, 'privLevel', privLevel )
      self.plugin = plugin
      AaaPluginLib.Authenticator.__init__( self, plugin.aaaConfig, method, 
                                           type, service,
                                           remoteHost, remoteUser, tty, user )
      self.state = self.sendAuthen
      self.lastStatus = None
      self.password = ""
      self.privLevel = privLevel
      self.authenReq = None
      self.maxUsernameLength = maxUsernameLength
      self.session = plugin.acquireSession( method )
      if not self.session:
         self.state = self.failed

   def sendAuthenReq( self ):
      if self.type == 'authnTypeEnable':
         service = libtacplus.TAC_AUTHEN_SVC_ENABLE
         privlevel = self.privLevel
      else:
         service = libtacplus.TAC_AUTHEN_SVC_LOGIN
         privlevel = 1
      s = self.session
      self.authenReq = s.createAuthenReq( service=service )
      rq = self.authenReq
      rq.privLevelIs( privlevel )
      if self.remoteHost:
         rq.remoteAddrIs( self.remoteHost )
      if self.tty:
         rq.portIs( self.tty )
      if self.user:
         rq.userIs( self.user )

      return self.processResponse( *s.sendAuthenReq( rq ) )

   def sendContinueAuthenReq( self, userInput ):
      self.authenReq.messageIs( userInput )
      return self.processResponse( *self.session.continueAuthenReq() )

   def processResponse( self, status, noecho, srvrMsg ):
      failText = "Error in authentication"
      self.lastStatus = status
      promptStyle = 'promptEchoOff' if noecho else 'promptEchoOn'
      if status == libtacplus.TAC_AUTHEN_STATUS_GETUSER:
         bt( TR_AUTHEN, "TACACS+ sent GETUSER:", bv( srvrMsg ) )
         nextState = self.sendContinue
         promptStyle = 'promptUser'
      elif status == libtacplus.TAC_AUTHEN_STATUS_GETPASS:
         bt( TR_AUTHEN, "TACACS+ sent GETPASS:", bv( srvrMsg ) )
         nextState = self.sendContinue
         promptStyle = 'promptPassword'
      elif status == libtacplus.TAC_AUTHEN_STATUS_GETDATA:
         bt( TR_AUTHEN, "TACACS+ sent GETDATA:", bv( srvrMsg ) )
         nextState = self.sendContinue
      elif status == libtacplus.TAC_AUTHEN_STATUS_RESTART:
         # RESTART is only supposed to be used for PPP
         bt( TR_ERROR, "TACACS+ sent RESTART:", bv( srvrMsg ) )
         failText = "TACACS+ server sent RESTART response"
         nextState = self.failed
      elif status == libtacplus.TAC_AUTHEN_STATUS_FOLLOW:
         # We don't support the crazy FOLLOW feature
         bt( TR_ERROR, "TACACS+ sent FOLLOW:", bv( srvrMsg ) )
         failText = "TACACS+ server sent unsupported FOLLOW response"
         nextState = self.failed
      elif status == libtacplus.TAC_AUTHEN_STATUS_PASS:
         bt( TR_AUTHEN, "TACACS+ sent PASS:", bv( srvrMsg ) )
         nextState = self.succeeded
      elif status == libtacplus.TAC_AUTHEN_STATUS_FAIL:
         bt( TR_ERROR, "TACACS+ sent FAIL:", bv( srvrMsg ) )
         nextState = self.failed
      else:
         bt( TR_ERROR, "TACACS+ sent unknown status", bv( status ),
             ":", bv( srvrMsg ) )
         nextState = self.failed

      if nextState == self.sendContinue:
         prompt = Tac.Value( "AaaApi::AuthenMessage",
                             style=promptStyle, text=srvrMsg )
         return self.transition( nextState, 'inProgress', [ prompt ],
                                 self.user, self.password )
      elif nextState == self.succeeded:
         return self.transition( nextState, 'success', [], self.user,
                                 self.password )
      
      assert nextState == self.failed
      msg = Tac.Value( "AaaApi::AuthenMessage", style='error', text=failText )
      return self.transition( nextState, 'fail', [ msg ], 
                              self.user, self.password )

   def authenticate( self, *responses ):
      s = self.state
      traceX( TR_AUTHEN, "authenticate: state=", self.stateMap[ s ] )
      failType = 'unavailable'
      try:
         if s == self.sendAuthen:
            return self.sendAuthenReq()
         elif s == self.sendContinue:
            if len( responses ) > 0:
               r = responses[ 0 ]
               if self.lastStatus == libtacplus.TAC_AUTHEN_STATUS_GETUSER:
                  traceX( TR_AUTHEN, "GETUSER:", r )
                  r = r[ :self.maxUsernameLength ]
                  self.user = r
               elif self.lastStatus == libtacplus.TAC_AUTHEN_STATUS_GETPASS:
                  self.password = r
               return self.sendContinueAuthenReq( r )
            else:
               bt( TR_ERROR, "User input required, but no input received!" )
               return self.transition( self.failed, 'fail', [],
                                       self.user )
         else:
            bt( TR_ERROR, "authenticate: unexpected state:",
                bv( self.stateMap[ s ] ) )
            # if no valid session, it's probably an DNS issue, otherwise
            # it should not happen.
            if self.session: 
               failType = 'fail'
      except Tacacs.AuthenticationError, e:
         bt( TR_ERROR, "TACACS+ AuthenticationError:", bv( str( e ) ) )
         # fall through to the fail case

      return self.transition( self.failed, failType )

   def transition( self, state, authenStatus, messages=[], user="",
                   authToken="" ):
      sm = self.stateMap
      traceX( TR_INFO, "Authenticator transitioning from", sm[ self.state ], "to",
              sm[ state ] )
      self.state = state
      r = { "status" : authenStatus, "messages" : messages, "user" : user,
            "authToken" : authToken }
      if state == self.succeeded or state == self.failed:
         self.plugin.releaseSession( self.session, state == self.succeeded )
         self.session = None
         self.authenReq = None
      return r

class TacacsPlugin( AaaPluginLib.Plugin ):
   def __init__( self, config, status, aaaConfig, ipStatus, ip6Status,
                 allVrfStatusLocal ):
      AaaPluginLib.Plugin.__init__( self, aaaConfig, "tacacs+",
                                    allVrfStatusLocal=allVrfStatusLocal )
      self.config = config
      self.status = status
      self.aaaConfig = aaaConfig
      self.ipStatus = ipStatus
      self.ip6Status = ip6Status

   def ready( self ):
      return len( self.config.host )

   def handlesAuthenMethod( self, method ):
      return self._handlesGroupAuthenMethod( method, "tacacs+", "tacacs" )

   def createAuthenticator( self, method, type, service, remoteHost, remoteUser,
                            tty, user=None, privLevel=0 ):
      return Authenticator( self, method, type, service,
                            remoteHost, remoteUser,
                            tty, user, privLevel,
                            self.config.maxUsernameLength )

   def openSession( self, authenticator ):
      return authenticator

   def closeSession( self, token ):
      pass

   def _doAuthz( self, method, configReqFunc ):
      session = self.acquireSession( method )
      if not session:
         # It's probably a DNS issue, otherwise it should not happen.
         return ( 'authzUnavailable',
                  'Could not acquire a session - is DNS available?', {} )
      req = session.createAuthzReq() 
      configReqFunc( req )
      try:
         status, msg, m_av, o_av = session.sendAuthzReq( req )
      except Tacacs.AuthorizationError, e:
         bt( TR_ERROR, "TACACS authorization error:", bv( str( e ) ) )
         self.releaseSession( session, False )
         return ( 'authzUnavailable', str( e ), {} )

      if ( status != libtacplus.TAC_AUTHOR_STATUS_PASS_ADD and
           status != libtacplus.TAC_AUTHOR_STATUS_PASS_REPL ):
         bt( TR_ERROR, "TACACS authz: status", bv( status ), "message", bv( msg ) )
         self.releaseSession( session, True )
         return ( 'denied', msg, {} )

      traceX( TR_AUTHZ, "authorizeShellCommand: status", status,
              "m_av", m_av, "o_av", o_av )
      if not self.config.ignoreUnknownMandatoryAttr:
         for av in m_av:
            if av not in _knownAttributes:
               # pylint: disable=undefined-variable
               Logging.log( AAA_UNKNOWN_TACACS_MANDATORY_ATTRIBUTE, av )
               # pylint: enable=undefined-variable
               self.releaseSession( session, True )
               return ( 'denied', msg, {} )

      def _translateAVPair( attrs, tacacsAttr, aaaAttr, valueType=int,
                            verifyFunc=None ):
         # If the same attribute is in mandatory and optional attributes list 
         # prefer mandatory attribute.
         v = m_av.get( tacacsAttr, o_av.get( tacacsAttr ) )
         if v is None:
            return
         try:
            v = valueType( v )
            if verifyFunc is None or verifyFunc( v ):
               attrs[ aaaAttr ] = [ v ] if tacacsAttr == 'roles' else v
         except ( ValueError, TypeError ):
            bt( TR_ERROR, "invalid TACACS+ attr", bv( tacacsAttr ),
                "value:", bv( v ) )
         return

      # Translate from TACACS+ attrs to Aaa attrs
      attrs = {}
      for a in ( "priv-lvl", "priv_lvl" ):
         _translateAVPair( attrs, a, AaaPluginLib.privilegeLevel,
                           verifyFunc=lambda x: x >=0 or x <= 15 )

      _translateAVPair( attrs, "roles", AaaPluginLib.roles, str,
                        verifyFunc=lambda x: len(x) > 0 )
      _translateAVPair( attrs, "timeout", AaaPluginLib.timeout )
      _translateAVPair( attrs, "idletime", AaaPluginLib.idleTime )
      _translateAVPair( attrs, "autocmd", AaaPluginLib.autoCmd, str )
      _translateAVPair( attrs, "secure-monitor", AaaPluginLib.secureMonitor, bool )

      self.releaseSession( session, True )
      return ( 'allowed', msg, attrs )

   def _configureReqFromSession( self, req, session ):
      if session is not None:
         # authenMethod defaults to TACACS+, so I only need to do something
         # here for other methods.
         if session.authenMethod == "local":
            req.authenMethodIs( libtacplus.TAC_AUTHEN_METH_LOCAL )
         elif session.authenMethod == "none":
            req.authenMethodIs( libtacplus.TAC_AUTHEN_METH_NONE )
         if session.remoteHost:
            req.remoteAddrIs( session.remoteHost )
         if session.tty:
            req.portIs( session.tty )

   def authorizeShell( self, method, user, session ):
      traceX( TR_AUTHZ, "authorizeShell for method", method, "user", user )
      def _configShellAuthzReq( req ):
         self._configureReqFromSession( req, session )
         req.userIs( user )
         req.privLevelIs( 1 ) # FIXME: is this correct?
         req.serviceIs( "shell" )
         req.commandIs( [] )

      return self._doAuthz( method, _configShellAuthzReq )
   
   def authorizeShellCommand( self, method, user, session, mode, privlevel, tokens ):
      traceX( TR_AUTHZ, "authorizeShellCommand for method", method, "user", user,
              "mode", mode, "privlevel", privlevel, "tokens", tokens )
      def _configShellCommandAuthzReq( req ):
         self._configureReqFromSession( req, session )
         req.userIs( user )
         req.privLevelIs( privlevel )
         req.serviceIs( "shell" )
         req.commandIs( tokens )
      return self._doAuthz( method, _configShellCommandAuthzReq )

   def _doAcct( self, method, action, createReqFunc ):
      """ Helper function for sendShellAcct and sendCommandAcct. """
      
      r = 'acctError'
      tacacsSession = self.acquireSession( method )
      if not tacacsSession:
         # It's probably a DNS issue, otherwise it should not happen.
         # Returning acctError will cause "dispatch" method in Aaa to
         # fallback to next accounting method.
         return ( r, 'Could not acquire a session - is DNS available?' )
      
      req = createReqFunc( tacacsSession )
      try:
         status, msg = tacacsSession.sendAcctReq( req )
         if status == libtacplus.TAC_ACCT_STATUS_SUCCESS:
            r = 'acctSuccess'
         elif status == libtacplus.TAC_ACCT_STATUS_FOLLOW:
            bt( TR_ERROR, "TACACS+ sent FOLLOW:", bv( msg ) )
            msg = "TACACS+ server sent unsupported FOLLOW response"
      except Tacacs.AccountingError, e:
         bt( TR_ERROR, "TACACS+ AccountingError:", bv( str( e ) ) )
         msg = str( e )
      self.releaseSession( tacacsSession, r == 'acctSuccess' )
      return ( r, msg )
   
   def sendShellAcct( self, method, user, session, action, startTime,
                      elapsedTime=None ):
      traceX( TR_ACCT, "acctShell for method", method, "user", user, "action",
              "session", session, "action", action, "startTime", startTime,
              "elapsedTime", elapsedTime )
      def _createShellAcctReq( tacacsSession ):
         acctAction = libtacplus.TAC_ACCT_START
         taskId = None
         if action == 'stop':
            acctAction = libtacplus.TAC_ACCT_STOP
            assert session
            if session.execAcctTaskId:
               taskId = session.execAcctTaskId
         req = tacacsSession.createAcctReq( acctAction, taskId=taskId )
         self._configureReqFromSession( req, session )
         req.userIs( user )
         # start_time is number of seconds from epoch
         req.startTimeIs( startTime )
         req.timeZoneIs()
         # store the exec acct taskid needed for the "stop" action
         if action == 'start':
            assert session
            session.execAcctTaskId = req.taskId()
         if action == 'stop':
            assert elapsedTime
            req.elapsedTimeIs( elapsedTime )
         return req

      acctAction = libtacplus.TAC_ACCT_STOP if action == 'stop' else \
          libtacplus.TAC_ACCT_START
      return self._doAcct( method, action, _createShellAcctReq )
   
   def sendCommandAcct( self, method, user, session, privlevel, timestamp,
                        tokens ):
      traceX( TR_ACCT, "acctShellCommand for method", method, "user", user,
              "privlevel", privlevel, "timestamp", timestamp, "tokens", tokens )
      def _createCommandAcctReq( tacacsSession ):
         req = tacacsSession.createAcctReq( libtacplus.TAC_ACCT_STOP )
         self._configureReqFromSession( req, session )
         req.userIs( user )
         req.privLevelIs( privlevel )
         req.startTimeIs( timestamp )
         req.timeZoneIs()
         req.commandIs( ' '.join( tokens ) )
         return req
      return self._doAcct( method, libtacplus.TAC_ACCT_STOP, _createCommandAcctReq )

   def sendSystemAcct( self, method, event, startTime, reason, action ):
      traceX( TR_ACCT, "sendSystemAcct for method", method, "action", action,
          "startTime", startTime, "reason", reason )
      def _createSystemAcctReq( tacacsSession ):
         acctAction = libtacplus.TAC_ACCT_START
         taskId = None
         if action == 'stop':
            acctAction = libtacplus.TAC_ACCT_STOP
         req = tacacsSession.createAcctReq( acctAction, acctService='system' )
         self._configureReqFromSession( req, None )
         req.userIs( 'unknown' )
         # start_time is number of seconds from epoch
         req.startTimeIs( startTime )
         req.timeZoneIs()
         req.setAttr( 'event', event )
         req.setAttr( 'reason', reason )
         return req

      return self._doAcct( method, action, _createSystemAcctReq )

   def invalidateSessionPool( self, hostgroup ):
      _invalidateSessionPool( hostgroup )

   def acquireSession( self, methodName ):
      hg, serverSpecs = AaaPluginLib.serversForMethodName( methodName,
                                                           self.aaaConfig, 
                                                           'tacacs' )
      if hg is not None:
         session = _sessionPool.get( hg )
         if session is not None:
            # We are reusing an existing session. If we use single-connection,
            # it's possible that the server has closed the connection if it's
            # been idle for too long, and if we still want to use it, we'll
            # see an error. So we make sure the connection is still good,
            # or we'll force a reconnection.
            session.revalidateConnection()
            session.maxUsernameLengthIs( self.config.maxUsernameLength )
            return session

      session = Tacacs.Session( hg )
      addedServers = 0
      tacacsConfig = self.config
      # If no aaa group servers are found, then follow the order of tacacs
      # server configuration. Otherwise, honor the group server configuration
      if  len( serverSpecs ) == 0:
         # If the server dict is empty I consider all configured servers to be
         # fair game.  Either methodName indicates that all servers should be
         # included, or it's a server group with no members, but I expect the
         # former because I shouldn't get to this point without any servers
         # since handlesAuthenMethod should have screened the request out.
         hostList = sorted( tacacsConfig.host.values(),
                            key=lambda host: host.index )
      else:
         hostList = []
         for spec in serverSpecs:
            try:
               host = tacacsConfig.host[ spec ]
               hostList.append( host )
            except KeyError:
               # Server group contains members that were not in the 
               # host collection
               groupname = AaaPluginLib.extractGroupFromMethod( methodName )
               serverStr = "%s:%d" % ( spec.hostname, spec.port )
               if spec.vrf and spec.vrf != DEFAULT_VRF:
                  serverStr += " (vrf %s)" % spec.vrf
               # pylint: disable=undefined-variable
               Logging.log( AAA_UNKNOWN_TACACS_SERVER, groupname, serverStr )
               # pylint: enable=undefined-variable

      for h in hostList:
         key = h.useKey and h.key or tacacsConfig.key
         #  the key in sysdb is encoded, now decode it
         key = ReversibleSecretCli.decodeKey( key )
         timeout = h.useTimeout and h.timeout or tacacsConfig.timeout

         ns = self.getNsFromVrf( h.vrf )
         if not ns:
            bt( TR_ERROR, "cannot get namespace for vrf", bv( h.vrf ) )
            continue

         hs = AaaPluginLib.HostSpec( h.hostname, h.port, 0, h.vrf )
         singleConn = ( h.connType == 'singleConnection' )
         cb = CounterCallback( self.status, hs )
         try:
            srcIpAddr = ""
            srcIp6Addr = ""
            assert h.vrf != ''
            if session and tacacsConfig.srcIntfName.get( h.vrf ):
               # There is a single primary IPv4 address, so for IPv4 there is no
               # ambiguity in setting the source IPv4 address.
               # But there can be multiple IPv6 addresses, so we set the first
               # IPv6 address in the collection that has a global scope (NOT
               # link local) as the source IPv6 address. This IPv6 address
               # selected is also first in the list displayed via the "show ipv6
               #  interface" and "show running-config" commands.
               # It is a fair assumption that the interface being used as
               # source-interface (such as Loopback0) will typically have a single
               # IPV6 address assigned
               srcIpIntfStatus = self.ipStatus.ipIntfStatus.get(
                  tacacsConfig.srcIntfName.get( h.vrf ) )
               if srcIpIntfStatus and \
                  srcIpIntfStatus.activeAddrWithMask.address != "0.0.0.0":
                  srcIpAddr = srcIpIntfStatus.activeAddrWithMask.address
               srcIp6IntfStatus = self.ip6Status.intf.get(
                  tacacsConfig.srcIntfName.get( h.vrf ) )
               if srcIp6IntfStatus:
                  for ip6addr in srcIp6IntfStatus.addr:
                     if ip6addr.address.isLinkLocal:
                        continue
                     else:
                        srcIp6Addr = ip6addr.address.stringValue
                        break

            bt( TR_AUTHEN, 'add server', bv( h.stringValue() ),
                "srcIpAddr", bv( srcIpAddr or "" ),
                "srcIp6Addr", bv( srcIp6Addr or "" ) )
            session.addServer( h.hostname, h.port, h.vrf, ns, key,
                               int( timeout ),
                               srcIp=srcIpAddr, srcIp6=srcIp6Addr,
                               singleConn=singleConn, statusCallback=cb )
            addedServers += 1
         except Tacacs.BadServerException, e:
            bt( TR_ERROR, "failed to add server", bv( h.stringValue() ),
                ":", bv( str( e ) ) )
            # pylint: disable=undefined-variable
            Logging.log( AAA_INVALID_TACACS_SERVER, h.hostname, h.port,
                         str( e ) )
            # pylint: enable=undefined-variable

      if addedServers == 0:
         # No servers indicates is a configuration error
         # pylint: disable=undefined-variable
         Logging.log( AAA_NO_VALID_TACACS_SERVERS, methodName )
         # pylint: enable=undefined-variable
         session.close()
         session = None

      if session:
         session.maxUsernameLengthIs( self.config.maxUsernameLength )
      return session

   def releaseSession( self, session, keepConnection ):
      bt( TR_AUTHEN, 'releaseSession keepConnection=', bv( keepConnection ) )
      # Close Tacacs connection if status is fail.
      # Always keep the session to prevent the tacacs lib from
      # stucking on the bad server if it can only find out 
      # the server is unavailable during continueAuthen
      if session is None: return
      if not keepConnection:
         session.closeConnection()
      _sessionPool.put( session )

   def getPwEnt( self, _ ):
      if self.ready():
         return AaaPluginLib.AAA_PWENT_RESULT.UNKNOWN
      return AaaPluginLib.AAA_PWENT_RESULT.INVALID

class ConfigReactor( AaaPluginLib.ConfigReactor ):
   notifierTypeName = "Tacacs::Config"
   counterTypeName = "Tacacs::Counters"

   def __init__( self, notifier, status, allVrfStatusLocal, ipStatus, ip6Status ):
      AaaPluginLib.ConfigReactor.__init__( self, notifier, 
                                           status, allVrfStatusLocal, 
                                           ipStatus, ip6Status )

   def invalidateSessionPool( self ):
      _invalidateSessionPool()

   @Tac.handler( 'host' )
   def handleHost( self, hostspec=None ):
      self.handleHostEntry( hostspec )

   def handleVrfState( self, vrfName ):
      # Theoretically we could use the vrfName to figure out if we
      # actually care about this change or not, but for simplicity we
      # just hit it with a big hammer and throw out any persistent
      # singleConn connections any time any VRF changes.  Since this
      # is all driven by config change events rather than protocol or
      # packet events, this isn't really a big deal
      _invalidateSessionPool()

class CounterConfigReactor( AaaPluginLib.CounterConfigReactor ):
   counterTypeName = "Tacacs::Counters"

_reactors = {}

def Plugin( ctx ):
   mountGroup = ctx.entityManager.mountGroup()
   config = mountGroup.mount( 'security/aaa/tacacs/config', 'Tacacs::Config',
                              'r' )
   counterConfig = mountGroup.mount( 'security/aaa/tacacs/counterConfig', 
                                     'AaaPlugin::CounterConfig', 'r' )
   status = mountGroup.mount( Cell.path( 'security/aaa/tacacs/status' ), 
                              'Tacacs::Status', 'wf' )
   aaaConfig = ctx.aaaAgent.config
   Tac.Type( "Ira::IraIpStatusMounter" ).doMountEntities( mountGroup.cMg_, True,
                                                          True )
   ipStatus = mountGroup.mount( 'ip/status', 'Ip::Status', 'r' )
   ip6Status = mountGroup.mount( 'ip6/status', 'Ip6::Status', 'r' )
   allVrfStatusLocal = mountGroup.mount( Cell.path( 'ip/vrf/status/local' ),
                                         'Ip::AllVrfStatusLocal', 'r' )

   def _finish():
      _reactors[ "TacacsConfigReactor" ] = ConfigReactor( config, status,
                                                          allVrfStatusLocal,
                                                          ipStatus,
                                                          ip6Status )
      _reactors[ "TacacsCounterConfigReactor" ] = \
          CounterConfigReactor( counterConfig, status )
   mountGroup.close( _finish )
   return TacacsPlugin( config, status, aaaConfig, ipStatus, ip6Status,
                        allVrfStatusLocal )
