# Copyright (c) 2008-2010, 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""
Provides a mechanism for other packages to extend the list of authentication,
authorization and accounting methods that can be used in the Aaa CLI.
For example, the Tacacs package can provide the "group tacacs+" method by
calling registerGroup with appropriate parameters.
"""

import BasicCli
import CliCommand
import CliGlobal
import CliMatcher
import CliParser
import ConfigMount
import HostnameCli
import LazyMount
import collections
from CliMode.Aaa import ServerGroupMode
from CliPlugin.VrfCli import VrfExprFactory, DEFAULT_VRF
import Tac
import Tracing

traceHandle_ = Tracing.Handle( "AaaCliLib" )
t0 = Tracing.trace0

def getHostIndex( hosts ):
   """ An "index" member is added to both Tacacs and Radius host entry to
   record the order in which servers are configured. This method gives the
   the next index to be assigned to a new server entry.
   """
   if hosts is None:
      indx = 1
   else: 
      sortedHost = sorted( hosts.values(), key=lambda host: host.index )
      indx = sortedHost[ -1 ].index + 1
   return indx

class ActionType( object ):
   AUTHN_LOGIN   =  1
   AUTHN_ENABLE  =  2
   AUTHZ_EXEC    =  4
   AUTHZ_COMMAND =  8
   ACCT_EXEC     = 16
   ACCT_COMMAND  = 32
   ACCT_SYSTEM   = 64
   AUTHN_DOT1X   = 128
   ACCT_DOT1X    = 256
   # some compound bits
   AUTHN = AUTHN_LOGIN|AUTHN_ENABLE
   AUTHZ = AUTHZ_EXEC|AUTHZ_COMMAND
   ACCT = ACCT_EXEC|ACCT_COMMAND|ACCT_SYSTEM

authnMethodListNonGroup = dict()
authnMethodListGroup = dict()
authnDot1xMethodListNonGroup = dict()
authnDot1xMethodListGroup = dict()
authzExecMethodListNonGroup = dict()
authzExecMethodListGroup = dict()
authzCommandMethodListNonGroup = dict()
authzCommandMethodListGroup = dict()
acctExecMethodListNonGroup = dict()
acctExecMethodListGroup = dict()
acctCommandMethodListNonGroup = dict()
acctCommandMethodListGroup = dict()
acctSystemMethodListNonGroup = dict()
acctSystemMethodListGroup = dict()
acctDot1xMethodListNonGroup = dict()
acctDot1xMethodListGroup = dict()

authnMethodListNonGroupMatcher = CliMatcher.DynamicKeywordMatcher(
   lambda mode: authnMethodListNonGroup )
authnMethodListGroupMatcher = CliMatcher.DynamicKeywordMatcher(
   lambda mode: authnMethodListGroup )
authnDot1xMethodListNonGroupMatcher = CliMatcher.DynamicKeywordMatcher(
   lambda mode: authnDot1xMethodListNonGroup )
authnDot1xMethodListGroupMatcher = CliMatcher.DynamicKeywordMatcher(
   lambda mode: authnDot1xMethodListGroup )
authzExecMethodListNonGroupMatcher = CliMatcher.DynamicKeywordMatcher(
   lambda mode: authzExecMethodListNonGroup )
authzExecMethodListGroupMatcher = CliMatcher.DynamicKeywordMatcher(
   lambda mode: authzExecMethodListGroup )
authzCommandMethodListNonGroupMatcher = CliMatcher.DynamicKeywordMatcher(
   lambda mode: authzCommandMethodListNonGroup )
authzCommandMethodListGroupMatcher = CliMatcher.DynamicKeywordMatcher(
   lambda mode: authzCommandMethodListGroup )
acctExecMethodListNonGroupMatcher = CliMatcher.DynamicKeywordMatcher(
   lambda mode: acctExecMethodListNonGroup )
acctExecMethodListGroupMatcher = CliMatcher.DynamicKeywordMatcher(
   lambda mode: acctExecMethodListGroup )
acctCommandMethodListNonGroupMatcher = CliMatcher.DynamicKeywordMatcher(
   lambda mode: acctCommandMethodListNonGroup )
acctCommandMethodListGroupMatcher = CliMatcher.DynamicKeywordMatcher(
   lambda mode: acctCommandMethodListGroup )
acctSystemMethodListNonGroupMatcher = CliMatcher.DynamicKeywordMatcher(
   lambda mode: acctSystemMethodListNonGroup )
acctSystemMethodListGroupMatcher = CliMatcher.DynamicKeywordMatcher(
   lambda mode: acctSystemMethodListGroup )
acctDot1xMethodListNonGroupMatcher = CliMatcher.DynamicKeywordMatcher(
   lambda mode: acctDot1xMethodListNonGroup )
acctDot1xMethodListGroupMatcher = CliMatcher.DynamicKeywordMatcher(
   lambda mode: acctDot1xMethodListGroup )

def addAuthnMethod( name, desc ):
   authnMethodListNonGroup[ name ] = desc

def addAuthnDot1xMethod( name, desc ):
   authnDot1xMethodListNonGroup[ name ] = desc

def addAuthzExecMethod( name, desc ):
   authzExecMethodListNonGroup[ name ] = desc

def addAuthzCommandMethod( name, desc ):
   authzCommandMethodListNonGroup[ name ] = desc

def addAcctExecMethod( name, desc ):
   acctExecMethodListNonGroup[ name ] = desc

def addAcctCommandMethod( name, desc ):
   acctCommandMethodListNonGroup[ name ] = desc

def addAcctSystemMethod( name, desc ):
   acctSystemMethodListNonGroup[ name ] = desc

def addAcctDot1xMethod( name, desc ):
   acctDot1xMethodListNonGroup[ name ] = desc

def addMethod( name, descTemplate,
               suppType=ActionType.AUTHN|ActionType.AUTHZ|ActionType.ACCT|\
                        ActionType.AUTHN_DOT1X|ActionType.ACCT_DOT1X ):
   if suppType & ActionType.AUTHN:
      addAuthnMethod( name, descTemplate % "authentication" )
   if suppType & ActionType.AUTHN_DOT1X:
      addAuthnDot1xMethod( name, descTemplate % "authentication" )
   if suppType & ActionType.AUTHZ_EXEC:
      addAuthzExecMethod( name, descTemplate % "authorization" )
   if suppType & ActionType.AUTHZ_COMMAND:
      addAuthzCommandMethod( name, descTemplate % "authorization" )
   if suppType & ActionType.ACCT_EXEC:
      addAcctExecMethod( name, descTemplate % "accounting" )
   if suppType & ActionType.ACCT_COMMAND:
      addAcctCommandMethod( name, descTemplate % "accounting" )
   if suppType & ActionType.ACCT_SYSTEM:
      addAcctSystemMethod( name, descTemplate % "accounting" )
   if suppType & ActionType.ACCT_DOT1X:
      addAcctDot1xMethod( name, descTemplate % "accounting" )

addMethod( 'local', "Use local database for %s",
           suppType=ActionType.AUTHN | ActionType.AUTHZ )
addMethod( 'none', "No %s (always succeeds)",
           suppType=ActionType.AUTHN | ActionType.AUTHZ )
addMethod( 'logging', "Use syslog for %s",
           suppType=ActionType.ACCT | ActionType.ACCT_DOT1X )

# The map from groupType to various group-specific values
# populated by registerGroup()
_serverGroupDB = { }

#-------------------------------------------------------------------------------
#     aaa group server [tacacs+|radius|...] <server-group-name>
#-------------------------------------------------------------------------------

class ServerGroupConfigMode( ServerGroupMode, BasicCli.ConfigModeBase ):
   #----------------------------------------------------------------------------
   # This is meant to be a base class only.
   # Attributes required for mode class should be defined by the subclass
   #----------------------------------------------------------------------------
   def __init__( self, parent, session, group, groupName ):
      self.groupName = groupName
      groupType = _serverGroupDB[ group.groupType ].cliToken
      self.defaultPort = _serverGroupDB[ group.groupType ].authport.defaultPort
      if _serverGroupDB[ group.groupType ].acctport:
         self.defaultAcctPort = \
               _serverGroupDB[ group.groupType ].acctport.defaultPort
      else:
         self.defaultAcctPort = 0
      ServerGroupMode.__init__( self, groupType, groupName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def setServer( self, hostname, port=None, vrf=DEFAULT_VRF, acctPort=None ):
      if port is None:
         port = self.defaultPort
      assert vrf != ''
      if vrf is None:
         vrf = DEFAULT_VRF
      if acctPort is None:
         acctPort = self.defaultAcctPort
      spec = Tac.Value( "Aaa::HostSpec", hostname=hostname, port=port,
                        acctPort=acctPort, vrf=vrf )
      for m in configAaa( self ).hostgroup[ self.groupName ].member.itervalues():
         if m.spec == spec:
            break
      else:
         member = Tac.Value( "Aaa::HostGroupMember", spec )
         configAaa( self ).hostgroup[ self.groupName ].member.enq( member )

   def noServer( self, hostname, port=None, vrf=DEFAULT_VRF, acctPort=None ):
      if port is None:
         port = self.defaultPort
      assert vrf != ''
      if vrf is None:
         vrf = DEFAULT_VRF
      if acctPort is None:
         acctPort = self.defaultAcctPort
      spec = Tac.Value( "Aaa::HostSpec", hostname=hostname, port=port,
                        acctPort=acctPort, vrf=vrf )
      for i, m in configAaa( self ).hostgroup[ self.groupName ].member.iteritems():
         if m.spec == spec:
            del configAaa( self ).hostgroup[ self.groupName ].member[ i ]
            break

configMode = BasicCli.GlobalConfigMode

aaaKwMatcher = CliMatcher.KeywordMatcher(
   'aaa',
   helpdesc="Authentication, Authorization and Accounting" )

gv = CliGlobal.CliGlobal( dict( config=None,
                                counterConfig=None ) )

def counterConfigAaaIs( entityManager ):
   gv.counterConfig = LazyMount.mount( entityManager,
                                       "security/aaa/counterConfig",
                                       "Aaa::CounterConfig", "w" )

def counterConfigAaa( mode ):
   return gv.counterConfig

def configAaaIs( entityManager ):
   gv.config = ConfigMount.mount( entityManager,
                                  "security/aaa/config",
                                  "Aaa::Config", "w" )

def configAaa( mode ):
   return gv.config

def initLibrary( entityManager ):
   configAaaIs( entityManager )
   counterConfigAaaIs( entityManager )

def getGroupTypeFromToken( groupToken ):
   for k in _serverGroupDB:
      if _serverGroupDB[ k ].cliToken == groupToken:
         return k
   return 'unknown'

def getCliDisplayFromGroup( groupType ):
   return _serverGroupDB[ groupType ].cliDisplay

# This is for the new parser
serverGroupNameMatcher = CliMatcher.DynamicNameMatcher(
   lambda mode: configAaa( mode ).hostgroup,
   helpdesc='Server-group name',
   priority=CliParser.PRIO_LOW )

# return a dynamic config mode based on the token entered
def _createServerGroupConfigModeType( groupToken ):
   className = '%sServerGroupConfigMode' % ( groupToken )
   return type( className, ( ServerGroupConfigMode, ),
                dict ( name='Server-group %s' % ( groupToken ),
                       modeParseTree=CliParser.ModeParseTree() ) )

def _gotoServerGroupConfigMode( mode, groupToken, groupName ):
   # do not allow 'groupToken' to be used as groupName
   if groupToken == groupName:
      mode.addError( "group \'%s\' is reserved" % groupToken )
      return

   groups = configAaa( mode ).hostgroup
   # find the group type from groupToken
   groupType = getGroupTypeFromToken( groupToken )
   assert groupType != 'unknown'

   if groupName in groups:
      group = groups[ groupName ]
      # if the group exists but is not an intended group, ignore the command
      if group.groupType != groupType:
         gtype = _serverGroupDB[ group.groupType ].cliToken
         mode.addError( "Group \'%s\' already exists for %s" % ( groupName, gtype ) )
         return
   else:
      group = groups.newMember( groupName )
      group.groupType = groupType

   childMode = mode.childMode( _serverGroupDB[ groupType ].configModeType,
                               group=group, groupName=groupName )
   mode.session_.gotoChildMode( childMode )

def _noServerGroup( mode, groupToken, groupName ):
   groups = configAaa( mode ).hostgroup
   groupType = getGroupTypeFromToken( groupToken )
   assert groupType != 'unknown'

   if ( groupName in groups and
        groups[ groupName ].groupType == groupType ):
      del groups[ groupName ]

# simple type for port registration, see [Tacacs|Radius]Group.py
AaaPortInfo = collections.namedtuple( 'AaaPortInfo',
                                      'portToken portHelp defaultPort' )

ServerGroupInfo = collections.namedtuple(
   'ServerGroupInfo',
   'cliToken cliDisplay authport acctport configModeType suppType vrfSupport' )

def registerGroup( groupType, cliToken, cliDisplay, authport,
                   acctport, suppType, vrfSupport ):
   """
   Register a Cli plugin with its method list and various Aaa group CLI commands.

   groupType:   The HostGroupType enum defined in Aaa.tac to uniquely identify
                this protocol
   cliToken:    The token used in the CLI commands, such as 'tacacs+'
   cliDisplay:  How to display the protocol in CLI, such as 'TACACS+'
   authport/acctport: Authentication and accouting port information
   suppType:    Which types of authn/authz/acct are supported
   vrfSupport:  Is VRF supported, True/False
   """
   assert not groupType in _serverGroupDB
   configModeType = _createServerGroupConfigModeType( cliToken )
   _serverGroupDB[ groupType ] = ServerGroupInfo( cliToken,
                                                  cliDisplay,
                                                  authport,
                                                  acctport,
                                                  configModeType,
                                                  suppType,
                                                  vrfSupport )
   # register for method list
   helpdesc = "Use list of all defined %s hosts" % cliDisplay

   authnMethodListGroup[ cliToken ] = helpdesc

   if suppType & ActionType.AUTHN_DOT1X:
      authnDot1xMethodListGroup[ cliToken ] = helpdesc
   if suppType & ActionType.AUTHZ_EXEC:
      authzExecMethodListGroup[ cliToken ] = helpdesc
   if suppType & ActionType.AUTHZ_COMMAND:
      authzCommandMethodListGroup[ cliToken ] = helpdesc
   if suppType & ActionType.ACCT_EXEC:
      acctExecMethodListGroup[ cliToken ] = helpdesc
   if suppType & ActionType.ACCT_COMMAND:
      acctCommandMethodListGroup[ cliToken ] = helpdesc
   if suppType & ActionType.ACCT_SYSTEM:
      acctSystemMethodListGroup[ cliToken ] = helpdesc
   if suppType & ActionType.ACCT_DOT1X:
      acctDot1xMethodListGroup[ cliToken ] = helpdesc

   # We cannot use serverGroupNameMatcher as it auto-completes all server groups.
   # We need to only auto-complete on server groups specific to our type.
   myServerGroupNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: [ k for ( k, v ) in configAaa( mode ).hostgroup.iteritems( )
                     if v.groupType == groupType ],
      helpdesc='Server-group name' )

   # the server-group config mode
   class ServerGroupCmd( CliCommand.CliCommandClass ):
      syntax = "aaa group server %s NAME" % cliToken
      noOrDefaultSyntax = syntax
      data = {
         'aaa' : aaaKwMatcher,
         'group' : 'Group definitions',
         'server' : 'AAA server-group definitions',
         cliToken : '%s server-group definition' % cliDisplay,
         'NAME' : myServerGroupNameMatcher }

      @staticmethod
      def handler( mode, args ):
         _gotoServerGroupConfigMode( mode, cliToken, args[ 'NAME' ] )

      @staticmethod
      def noOrDefaultHandler( mode, args ):
         _noServerGroup( mode, cliToken, args[ 'NAME' ] )

   BasicCli.GlobalConfigMode.addCommandClass( ServerGroupCmd )

   #-------------------------------------------------------------------------------
   # In tacacs+ server group config mode:
   #
   #    [no] server <ip-addr-or-hostname> [VRF] [port <0-65535>]
   #
   # In radius server group config mode:
   #
   #    [no] server <ip-addr-or-hostname> [acct-port <0-65535>] [auth-port <0-65535>]
   #-------------------------------------------------------------------------------
   class ServerCmd( CliCommand.CliCommandClass ):
      syntax = "server HOSTNAME"
      data = {
         'server' : 'Add a % s server to the server - group' % cliDisplay,
         'HOSTNAME' : HostnameCli.IpAddrOrHostnameMatcher(
            helpname='WORD',
            helpdesc='Hostname or IP address of %s server' % cliDisplay,
            ipv6=True )
      }
      if vrfSupport:
         syntax += ' [ VRF ]'
         data[ 'VRF' ] = VrfExprFactory( helpdesc='VRF for this server' )

      _portNumber = CliMatcher.IntegerMatcher( 1, 65535,
                                               helpdesc="Number of the port to use" )
      syntax += " [ %s AUTHPORT ]" % authport.portToken
      data[ authport.portToken ] = authport.portHelp
      data[ 'AUTHPORT' ] = _portNumber

      if acctport is not None:
         syntax += " [ %s ACCTPORT ]" % acctport.portToken
         data[ acctport.portToken ] = acctport.portHelp
         data[ 'ACCTPORT' ] = _portNumber

      noOrDefaultSyntax = syntax

      @staticmethod
      def handler( mode, args ):
         mode.setServer( args[ 'HOSTNAME' ],
                         port=args.get( 'AUTHPORT' ),
                         vrf=args.get( 'VRF', DEFAULT_VRF ),
                         acctPort=args.get( 'ACCTPORT' ) )

      @staticmethod
      def noOrDefaultHandler( mode, args ):
         mode.noServer( args[ 'HOSTNAME' ],
                        port=args.get( 'AUTHPORT' ),
                        vrf=args.get( 'VRF', DEFAULT_VRF ),
                        acctPort=args.get( 'ACCTPORT' ) )

   configModeType.addCommandClass( ServerCmd )

def groupTypeSupported( groupType ):
   """Returns a bitmap of ActionType supported by the groupType"""
   return _serverGroupDB[ groupType ].suppType

#######################################################################
# The following reactors let Cli skip doing PyClient to Aaa unless 
# it's necessary, thus improving the performance of running CLI 
# commands. It also caches the results in Python to avoid GenericIf 
# accesses which are very slow.
#######################################################################

class AuthzMethodReactor( Tac.Notifiee ):
   notifierTypeName = "Aaa::AuthzMethodList"

   def __init__( self, notifier, configReactor ):
      self.configReactor_ = configReactor
      Tac.Notifiee.__init__( self, notifier )

   @Tac.handler( 'method' )
   def handleMethod( self, methodIndex ):
      self.configReactor_.updateAuthz()

class AcctMethodReactor( Tac.Notifiee ):
   notifierTypeName = "Aaa::AcctMethodList"

   def __init__( self, notifier, configReactor ):
      self.configReactor_ = configReactor
      Tac.Notifiee.__init__( self, notifier )

   @Tac.handler( 'defaultAction' )
   def handleDefaultAction( self ):
      self.configReactor_.updateAcct()

   @Tac.handler( 'consoleAction' )
   def handleConsoleAction( self ):
      self.configReactor_.updateAcct()

   @Tac.handler( 'consoleUseOwnMethod' )
   def handleConsoleUseOwnMethod( self ):
      self.configReactor_.updateAcct()

class AaaConfigReactor( Tac.Notifiee ):
   notifierTypeName = "Aaa::Config"

   def __init__( self, notifier ):
      Tac.Notifiee.__init__( self, notifier )
      self.suppressCfgCmdAuthz_ = False
      self.skipCmdAuthz_ = False
      self.skipCmdAcct_ = False
      self.skipConsoleCmdAuthz_ = False
      self.skipConsoleCmdAcct_ = False
      self.privLevel_ = None
      self.authzMethodReactor_ = None
      self.acctMethodReactor_ = None

   def skipCmdAuthz( self, privLevel, isConfigCmd, isConsole ):
      self.privLevelIs( privLevel )
      if self.suppressCfgCmdAuthz_ and isConfigCmd:
         return True
      return self.skipConsoleCmdAuthz_ if isConsole else self.skipCmdAuthz_

   def skipCmdAcct( self, privLevel, isConsole ):
      self.privLevelIs( privLevel )
      return self.skipConsoleCmdAcct_ if isConsole else self.skipCmdAcct_

   def privLevelIs( self, privLevel ):
      """This is required to initialize the reactor."""
      if self.privLevel_ != privLevel:
         t0( "privLevel is", privLevel )
         self.privLevel_ = privLevel
         self.update()
         self._createMethodReactors()

   def _cmdName( self ):
      return "command%02d" % self.privLevel_

   def _createMethodReactors( self ):
      self.authzMethodReactor_ = None
      name = self._cmdName()
      ml = self.notifier_.authzMethod.get( name )
      if ml:
         t0( "creating AuthzMethodReactor for", name )
         self.authzMethodReactor_ = AuthzMethodReactor( ml, self )

      self.acctMethodReactor_ = None
      ml = self.notifier_.acctMethod.get( name )
      if ml:
         t0( "creating AcctMethodReactor for", name )
         self.acctMethodReactor_ = AcctMethodReactor( ml, self )

   def updateAuthz( self ):
      if self.privLevel_ is None:
         return

      self.suppressCfgCmdAuthz_ = self.notifier_.suppressConfigCommandAuthz
      t0( 'suppressCfgCmdAuthz is', self.suppressCfgCmdAuthz_ )

      name = self._cmdName()
      ml = self.notifier_.authzMethod.get( name )
      self.skipCmdAuthz_ = ( ml is not None and
                             ml.method.values() == [ 'none' ] )
      if not self.notifier_.consoleAuthz:
         self.skipConsoleCmdAuthz_ = True
      else:
         self.skipConsoleCmdAuthz_ = self.skipCmdAuthz_
      t0( 'skipCmdAuthz is', self.skipCmdAuthz_ )
      t0( 'skipConsoleCmdAuthz is', self.skipConsoleCmdAuthz_ )

   def updateAcct( self ):
      if self.privLevel_ is None:
         return

      name = self._cmdName()
      ml = self.notifier_.acctMethod.get( name )
      if ml is None:
         self.skipCmdAcct_ = False
         self.skipConsoleCmdAcct_ = False
      elif not ml.consoleUseOwnMethod:
         self.skipCmdAcct_ = ( ml.defaultAction == 'none' )
         self.skipConsoleCmdAcct_ = self.skipCmdAcct_
      else:
         # We don't know if we are console or not, so both have to be "none"
         self.skipCmdAcct_ = ( ml.defaultAction == 'none' and
                               ml.consoleAction == 'none' )
         self.skipConsoleCmdAcct_ = ( ml.consoleAction == 'none' )
      t0( 'skipCmdAcct is', self.skipCmdAcct_ )
      t0( 'skipConsoleCmdAcct is', self.skipConsoleCmdAcct_ )

   def update( self ):
      self.updateAuthz()
      self.updateAcct()

   @Tac.handler( 'suppressConfigCommandAuthz' )
   def handleSuppressConfigCommandAuthz( self ):
      self.updateAuthz()

   @Tac.handler( 'consoleAuthz' )
   def handleConsoleAuthz( self ):
      self.updateAuthz()
      
