# Copyright (c) 2008, 2009, 2010, 2011  Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import Tac, Tracing, Aaa
import Logging

Logging.logD( id="AAA_EXEC_AUTHZ_FAILED",
              severity=Logging.logWarning,
              format="User %s failed authorization to start a shell on %s",
              explanation="AAA server denied authorization to start a shell.",
              recommendedAction="Confirm that the user settings on authorization "
                                "server(s) are correct." )

Logging.logD( id="AAA_CMD_AUTHZ_FAILED",
              severity=Logging.logWarning,
              format="User %s failed authorization to execute command '%s'",
              explanation="AAA denied authorization for the specified command.",
              recommendedAction="Confirm that the user settings on authorization "
                                "server(s) are correct." )

t0 = Tracing.trace0

def _waitForAaa( ):
   '''Wait for Aaa to initialize before proceeding (especially important if
   Aaa restarts). 30 seconds should be plenty, and if we timeout the client 
   will get an exception.

   Use sleep=True as the client should use execModeThreadPerConnection to
   connect to Aaa.
   '''
   Tac.waitFor( lambda: Aaa.agent and Aaa.agent.initialized,
                description="Aaa to initialized", warnAfter=None,
                sleep=True, timeout=30 )

def startLoginAuthenticate( service, tty=None, remoteHost=None,
                            remoteUser=None, user="", method=None ):
   """The authentication conversation for a login starts here, with an optional
   username.  The type of authentication is assumed to be 'authnTypeLogin'.  The
   'service' parameter specifies the service to which the user is
   authenticating, eg. "sshd". All the other parameters are optional, but should
   be provided if they are available to the caller.  Returns an
   AaaApi::AuthenState instance that includes the id used to continue the
   conversation."""
   _waitForAaa()
   state = Aaa.agent.startAuthenticate( 'authnTypeLogin', service, tty,
                                        remoteHost,
                                        remoteUser,
                                        user, method=method )
   Aaa.agent.incAuthnCounter( state.status )
   return Tac.valueToStrep( state )

def startEnableAuthenticate( service, uid, privLevel, sessionId=None ):
   """The authentication conversation for the CLI enable command starts here.
   The uid is required, and the effective uid of the calling process should be
   passed.  The type of authentication is assumed to be 'authnTypeEnable'.
   Returns an AaaApi::AuthenState instance that includes the authenStateId used
   to continue the conversation."""
   _waitForAaa()
   state = Aaa.agent.startAuthenticate( 'authnTypeEnable', service,
                                        tty=None, remoteHost=None,
                                        remoteUser=None, user="", uid=uid,
                                        privLevel=privLevel,
                                        sessionId=sessionId )
   Aaa.agent.incAuthnCounter( state.status )
   return Tac.valueToStrep( state )

def continueAuthenticate( authenStateId, *responses, **keywords ):
   """After starting an authentication conversation, the client continues
   calling this function as long as the returned AuthenState instance has a
   status of inProgress.  After each call the messages in the message list
   should be displayed to the user, and responses received for any prompts in
   the list.  Then the responses should be passed in order as arguments to the
   next continueAuthenticate invokation.  Returns an AaaApi::AuthenState
   instance."""
   _waitForAaa()
   state = Aaa.agent.continueAuthenticate( authenStateId, *responses, 
                                           **keywords )
   Aaa.agent.incAuthnCounter( state.status )
   return Tac.valueToStrep( state )

def abortAuthenticate( authenStateId ):
   """Ends the authentication conversation. Causes the AAA agent to forget
   about this AuthenState id.  Returns nothing."""
   _waitForAaa()
   Aaa.agent.abortAuthenticate( authenStateId )
   Aaa.agent.incAuthnCounter( 'fail' )
   return

def createSession( user, service, tty=None, remoteHost=None, remoteUser=None,
                   authenMethod="" ):
   """Creates a session for a user in case the authentication was not handled
   by AAA."""
   _waitForAaa()
   session = Aaa.agent.createSession( user, service, tty, remoteHost, remoteUser,
                                      authenMethod=authenMethod )
   return Tac.valueToStrep( session )

def openSession( authenStateId ):
   """Opens a session for a previously authenticated user. The caller passes in
   the AuthenState id for the conversation in which the user authenticated.
   Returns an AaaApi::Session instance."""
   _waitForAaa()
   session = Aaa.agent.openSession( authenStateId )
   return Tac.valueToStrep( session )

def closeSession( sessionId ):
   """Closes a previously opened session.  Returns string.
   As AaaPamHelper tries to parse the return value we have to return something."""
   _waitForAaa()
   result = Aaa.agent.closeSession( sessionId )
   return Tac.valueToStrep( result )

def getPwEnt( name=None, uid=None, force=True ):
   "Returns an AaaApi::PasswdResult instance for the name or uid specified."
   _waitForAaa()
   result = Tac.Value( "AaaApi::PasswdResult" )
   entry = Aaa.agent.getPwEnt( name, uid, force )
   if entry is not None:
      result.found = True
      result.passwordEntry = entry
   else:
      result.found = False
   # remove the extra back slashes added by "valueToStrep" using "decode"
   return Tac.valueToStrep( result ).decode( 'string_escape' )

def getSessionRoles( sessionId ):
   "Returns the roles for the current session."
   _waitForAaa()
   roles = Aaa.agent.getSessionRoles( sessionId )
   return roles

def getGrEnt( name=None, gid=None ):
   "Returns an AaaApi::GroupResult instance for the name or gid specified."
   _waitForAaa()
   result = Tac.Value( "AaaApi::GroupResult" )
   entry = Aaa.agent.getGrEnt( name, gid )
   if entry is not None:
      result.found = True
      result.groupEntry = entry
   else:
      result.found = False
   # remove the extra back slashes added by "valueToStrep" using "decode"
   return Tac.valueToStrep( result ).decode( 'string_escape' )

def _authorizeShell( uid, sessionId=None, sessionPid=None, tty=None ):
   """Determines whether the user specified is authorized to start a new shell.
   For this to succeed the Aaa agent must have an active login session for this
   uid.  Returns an AaaApi::AuthzResult."""
   _waitForAaa()
   r = Aaa.agent.authorizeShell( uid, sessionId, sessionPid, tty )
   Aaa.agent.incAuthzCounter( r.status )
   if r.status == 'denied':
      ent = Aaa.agent.getPwEnt( uid=uid )
      if ent :
         user = ent.userName
      else:
         user = "UID:%d" % uid
      Logging.log( AAA_EXEC_AUTHZ_FAILED, user, tty or "unknown tty" )
   return r

def authorizeShell( uid, sessionId=None, sessionPid=None, tty=None ):
   """Returns AaaApi::AuthzResult string representation."""
   r = _authorizeShell( uid, sessionId=sessionId, sessionPid=sessionPid,
                        tty=tty )
   return Tac.valueToStrep( r )

def authorizeShellWithOutput( uid, sessionId=None, sessionPid=None,
                              tty=None ):
   """Same as authorizeShell(), but return the result in a string
   instead of Tac.Value() by printing it. This function is supposed 
   to be called by execute() instead of eval(). It allows the caller 
   to not have to import Tac.py for better performance."""
   r = _authorizeShell( uid, sessionId=sessionId, sessionPid=sessionPid,
                        tty=tty )
   # result is ( status, privLevel, autocmd, message )
   result = [ str( r.status ), '1', '', r.message ]
   for v in r.av.itervalues():
      if v.name == "privilegeLevel":
         result[ 1 ] = v.val
      elif v.name == "autocmd":
         # value is repr() of the autocmd string (see _doAuthz() in Aaa.py),
         # so strip that
         result[ 2 ] = v.val[ 1:-1 ]
   # Use '\x01' as a delimiter.
   print "\x01".join( result )

def authorizeShellCommand( uid, mode, privLevel, tokens, 
                           sessionId=None ):
   """Determines whether the user is authorized to run a certain command in
   a certain mode.  token is a list of tokens that compose the command.
   Returns an AaaApi::AuthzResult."""
   _waitForAaa()
   r = Aaa.agent.authorizeShellCommand( uid, mode, privLevel, tokens, sessionId )
   Aaa.agent.incAuthzCounter( r.status )
   if r.status == 'denied':
      ent = Aaa.agent.getPwEnt( uid=uid )
      if ent :
         user = ent.userName
      else:
         user = "UID:%d" % uid
      Logging.log( AAA_CMD_AUTHZ_FAILED, user,
                   " ".join( tokens ) )

   return Tac.valueToStrep( r )

def sendCommandAcct( uid, privLevel, tokens, sessionId=None ):
   """Sends command accounting to the accounting module. Tokens is a list
   of tokens that compose the command. Returns an AaaApi::AcctResult."""
   _waitForAaa()
   r = Aaa.agent.sendCommandAcct( uid, privLevel, tokens, sessionId )
   return Tac.valueToStrep( r )

def authenticateAndAuthorizeLocalCommand( user, mode, tokens,
                                          privLevel=1 ):
   """Authenticate and Authorize a local shell command in the same call to avoid
   multiple PyClient RPCs.
   Returns an AaaApi::AuthzResultSimple that includes authorization result."""
   _waitForAaa()

   result = Tac.Value( 'AaaApi::AuthzResultSimple', status='denied', message="" )

   state = Aaa.agent.startAuthenticate( 'authnTypeLogin', 'shell', None,
                                        None, None, user, None )
   Aaa.agent.incAuthnCounter( state.status )

   if state.status == 'fail':
      result.message = "Unauthorized client"
      return Tac.valueToStrep( result )
   elif state.status == 'unavailable':
      result.message = "Authentication service is currently unavailable"
      return Tac.valueToStrep( result )
   elif state.status == 'unknown':
      result.message = "Unknown client identity"
      return Tac.valueToStrep( result )

   entry = Aaa.agent.getPwEnt( name=user )

   if entry is None:
      result.message = "Unauthorized client"
      return Tac.valueToStrep( result )

   ar = Aaa.agent.authorizeShellCommand( entry.userId,
                                        mode,
                                        privLevel,
                                        tokens,
                                        state.id )
   Aaa.agent.incAuthzCounter( ar.status )

   if ar.status == 'denied':
      Logging.log( AAA_CMD_AUTHZ_FAILED, user, " ".join( tokens ) )

   result.status = ar.status
   result.message = ar.message
   return Tac.valueToStrep( result )
