# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import gdb  # pylint: disable=import-error

def simplifyType( t ):
   return t.strip_typedefs().unqualified()

def getTag( t ):
   return simplifyType( t ).tag

def getPtr( t, val ):
   raw = False
   if getTag( t ).startswith( "TacStash::CompressedStashRawPtr<" ):
      raw = True
   targetType = str( t.strip_typedefs().unqualified().template_argument( 0 ) )
   alignBits = int( str( t.strip_typedefs().unqualified().template_argument( 1 ) ) )
   base = int( str( gdb.parse_and_eval( 'TacStash::bottomOfStash' ) ) )
   offset = None
   if raw:
      offset = int( val[ 'compressedRawPtr_' ] )
   else:
      offset = int( val[ 'rawPtr_' ][ 'compressedRawPtr_' ] )
   ptr = ( ( offset << alignBits ) + base )
   return ( targetType, ptr )

class CompressedRawPtr( object ):
   "Print a CompressedRawPtr"
   def __init__( self, val ):
      self.val = val

   def to_string( self ):
      targetType, ptr = getPtr( simplifyType( self.val.type ), self.val )
      output = '(( %s * )0x%x)' % ( targetType, ptr )
      return output

class CompressedSmartPtr( object ):
   "Print a CompressedSmartPtr"
   def __init__( self, val ):
      self.val = val

   def to_string( self ):
      targetType, ptr = getPtr( simplifyType( self.val.type ), self.val )
      output = '(( %s * )0x%x)' % ( targetType, ptr )
      return output

def compressedPtrLookup( val ):
   tag = getTag( val.type )
   if tag and tag.startswith( "TacStash::CompressedStashSmartPtr<" ):
      return CompressedSmartPtr( val )
   if tag and tag.startswith( "TacStash::CompressedStashRawPtr<" ):
      return CompressedRawPtr( val )
   return None

gdb.pretty_printers.append( compressedPtrLookup )
