#!/usr/bin/env python
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
from __future__ import absolute_import, division, print_function

# Avoid directly importing gdb, to avoid pkgdeps (and pylint) noticing that
# gdb is injecting it's module straight into the interpreter. See aid/10
import importlib
gdb = importlib.import_module( 'gdb' )

class PrintAssertionFailure(object):
   ''' This provides a callable "hook" for gdb's prompt_hook.
   Before outputting the prompt, we check if the process id of the selected target
   has changed, and if so, scan it's stack to see if there's a failed assertion,
   and print out what details we can glean.
   '''

   def __init__( self ):
      self.inferior_pid = -1
      # If there's already another prompt_hook, be prepared to chain-call it.
      oldHook = getattr( gdb, 'prompt_hook', None )
      if oldHook:
         self.prev_hook = oldHook
      else:
         self.prev_hook = lambda _ : None

      # register our hook
      gdb.prompt_hook = self.hook


   def show_assert_fail( self ):
      # glibc puts the assert message into __abort_msg, which is of type:
      #
      # > struct abort_msg_s
      # > {
      # >   unsigned int size;
      # >   char msg[0];
      # > };
      # > extern struct abort_msg_s *__abort_msg;
      #
      # Once we have a non-null pointer to __abort_msg, as a pointer-to-int
      # we can add one, and get the address of the "msg" field above.
      #
      # A previous version of this test extracted the arguments to
      # __assert_fail from the stack. There's no reliable way to do that on
      # x86_64, so depending on the glibc internals seems less likely to break
      # than disassembling __assert_fail every time we upgrade our compiler.
      int_p_p = gdb.lookup_type( 'int' ).pointer().pointer()
      abort_msg_pp = gdb.parse_and_eval( '&__abort_msg' ).cast( int_p_p )
      abort_msg_p = abort_msg_pp.dereference()
      if abort_msg_p:
         char_p = gdb.lookup_type( 'char' ).pointer()
         txt = ( abort_msg_p + 1 ).cast( char_p ).string()
         print( "PROGRAM TERMINATED WITH ASSERTION FAILURE: %s" % txt )

   def maybe_show_assert_fail( self ):
      ''' check if the current inferior's pid has changed, and if so, show
      details of any assertion failure on the stack '''
      current_inferior = gdb.selected_inferior()
      if current_inferior is None or current_inferior.pid == self.inferior_pid:
         return
      self.inferior_pid = current_inferior.pid
      self.show_assert_fail()


   def hook( self, arg ):
      ''' Actual gdb hook - see if we need to print the assertion details, then
      chain-call any original hook. '''
      try:
         self.maybe_show_assert_fail()
      except Exception: # pylint: disable=broad-except
         pass
      finally:
         self.prev_hook( arg )

# Create our PrintAssertionFailure class. Make sure to only do this once if our
# script is evaluated multiple times.
if "assert_fail_printer" not in globals():
   assert_fail_printer = PrintAssertionFailure()
