#!/bin/bash

err_notify() {
   # List of existing variables and their value.
   # 'set' lists the functions as well therefore we need to put the sub-shell in
   # posix mode with 'set -o posix' in order to get 'set' to print only variables.
   local variables="$(set -o posix; set)"
   local exitcode="$1" # Failed command's exit code.
   local lineno="$2"   # Failed command's line number.
   local filepath="$(realpath "$BASH_SOURCE")" # Absolute path of current script.
   local linecontent="$(sed -n "$lineno"p "$filepath")" # Failed command's line content.
   local detailedlogpath="/var/log/sysinit-error.log"
   local log="The following command terminated with a non zero exit status: \
${filepath}:${lineno}: $linecontent - exit code: $exitcode \
- Detailed shell environment at that time can be found in $detailedlogpath"

   logger -p local4.info -t sysinit "$log"

   # logger doesn't handle the large number of lines this early in the boot
   # process. Therefore we create a separate file with the shell environment
   # at the time of the error.
   cat >> "$detailedlogpath" << EOF
The following command terminated with a non zero exit status:
   ${filepath}:${lineno}: $linecontent
   exit code: $exitcode
Shell environment at that time:
$variables


EOF
}

# Use trap ERR to log any command that ends with a non 0 exit status. See BUG190356.
# There will be a log of the failure in syslog. The shell environment at the time
# of the error will be recorded in /var/log/sysinit-error.log.
trap 'err_notify $? $LINENO' ERR

# If the kernel cmdline option TEST_SYSINIT_ERR_TRAP=1 is present, execute an invalid
# command to trigger the ERR trap. This is used for testing by EosInit/SysinitErrTrapTest.
if grep -q 'TEST_SYSINIT_ERR_TRAP=1' /proc/cmdline; then
   # Variable used to check that the shell env is recorded.
   testVar="testValue"
   ls --invalidarg
fi

# Stuff from rc.sysinit we need to put somewhere

# Replace various configuration files with our own
for file in `find /etc/ -name \*.Eos -type f`; do
    cp -pf $file ${file%.Eos}
done

# Only reinstall the ACL if it's missing from the root directory of the flash
if ! getfacl -d /mnt/flash 2>/dev/null | grep -qs 'group:eosadmin:rwx' || \
   ! getfacl /mnt/flash 2>/dev/null | grep -qs '^group:eosadmin:rwx' || \
   ! getfacl /mnt/flash 2>/dev/null | grep -qs '^mask::rwx' || \
   ! getfacl -d /mnt/flash 2>/dev/null | grep -qs '^other::r-x'; then
   # Install an ACL on the flash to guarantee that there is no difference for
   # the user between ext4 and vfat. It is necessary to do it in EOS because
   # even if it's persistent, Aboot doesn't have ACL support and can't
   # reinstall the ACL after a fullrecover has been done
   setfacl -Rm mask:rwx /mnt/flash > /dev/null 2>&1 || :
   setfacl -Rm g:eosadmin:rwx /mnt/flash > /dev/null 2>&1 || :
   setfacl -Rdm g:eosadmin:rwx /mnt/flash > /dev/null 2>&1 || :
   setfacl -Rdm o:rx /mnt/flash > /dev/null 2>&1 || :
fi

function set_timezone() {
    # Without actually parsing startup-config there are pathological cases
    # of multi-line commands that can fool the grep below.  In pratice
    # those pathological cases are not expected to occur and if they do
    # the SuperServer Timezone plugin will set the timezone to what is
    # configured in startup-config, resulting in a small cosmetic blip
    # at startup.
    tz_path=/usr/share/zoneinfo/posix
    startup_timezone=$(egrep 'clock +timezone +' /mnt/flash/startup-config \
    		   2> /dev/null | head -1 | cut -d' ' -f 3)
    # Ensure we have a valid timezone value
    if ! test -f $tz_path/$startup_timezone ; then
        startup_timezone=UTC
    fi
    # Set the system timezone
    rm -f /etc/localtime
    ln -s $tz_path/$startup_timezone /etc/localtime
}

# First approximation of the proper timezone
set_timezone

# Replace /etc/cron.daily/logrotate with ours
rm /etc/cron.daily/logrotate
cp /etc/logrotate.eos /etc/cron.daily/logrotate
