#!/bin/sh

# Get a file's modification date in vfat compatible format
function get_vfat_date()
{
   timestamp=$(stat -c %y $1)
   words=(`echo $timestamp | tr '.' ' '`)

   date=${words[0]}'-'${words[1]}

   vfatdate=${date//:/-}

   echo "$vfatdate"
}

# New products are using a flash formatted in ext4. The layout of the flash
# is a little bit different than the one on vfat. We now have 2 partitions:
# - /mnt/flash
# - /mnt/crash
# This crash partition is a tiny partition that holds kernel crash logs. The
# goal behind this is to keep it R/O most of the time so that it's very hard
# to corrupt it. It's going to be remounted R/W on 2 occasions. The first one
# is after a kernel crash, kdump will remount it to write the crash. The second
# one is here. If we have a kernel crash, we copy the crash to the flash and
# remount R/O the partition
CRASH_MNT=/mnt/crash
FLASH_MNT=/mnt/flash
debugdir=${FLASH_MNT}/debug
kernelcrashfile=${debugdir}/kernelcrash

# Create the debug directory if it doesn't exist and make sure the permissions
# are correct
mkdir -p ${debugdir}
chmod 770 ${debugdir}

if grep -qs ${CRASH_MNT} /proc/mounts; then
   if [ -f ${CRASH_MNT}/kernelcrash ]; then
      cp ${CRASH_MNT}/kernelcrash $kernelcrashfile
   fi
   if [ -f ${CRASH_MNT}/local_reload_cause ]; then
      cp ${CRASH_MNT}/local_reload_cause* ${debugdir}
   fi
   sync

   if mount -o remount,rw ${CRASH_MNT}; then
      rm -f ${CRASH_MNT}/kernelcrash
      rm -f ${CRASH_MNT}/local_reload_cause*
      if ! mount -o remount,ro ${CRASH_MNT}; then
         echo "Failed to remount crash partition R/O"
      fi
   else
      echo "Failed to remount crash partition R/W"
   fi
fi

# Setup link to kernel core crash file from /var/core with date/time named
# kernelcrash.last file that was kernelcrash prior to Ucd agent rename of file
# We only run this check when we are not in the crash kernel so adding the check
# to ensure the inkernelcrash not present, which is only found in crash kernel
kernelcrashlastfile=${FLASH_MNT}/debug/kernelcrash.last
inkernelcrash=/proc/vmcore
if [[ ! -e $inkernelcrash ]] && [[ -f $kernelcrashfile ]];then
   # Rotate the last X kernel crash files
   crashfilestokeep=3

   if [ -f $kernelcrashlastfile ]; then
       date=$(get_vfat_date $kernelcrashlastfile)

       mv $kernelcrashlastfile ${FLASH_MNT}/debug/kernelcrash.last.$date
   fi

   # Remove old logs
   ls -t ${FLASH_MNT}/debug/kernelcrash.last.* 2> /dev/null | tail -n +$(($crashfilestokeep + 1)) | xargs rm -f

   # Append date in the log file
   lastcrashdate=$(get_vfat_date $kernelcrashfile)
   echo $lastcrashdate >> $kernelcrashfile
   sync

   # Generate the symlink to /var/core/kernelcrash.last.date
   corefilename='/var/core/kernelcrash.last.'$lastcrashdate
   ln -s $kernelcrashlastfile $corefilename
fi
