#!/usr/bin/env python
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import argparse
import functools
import os.path
import sys

def main():
   # Calling int with base=0 causes the int function to handle hex, oct,
   # binary, decimal by checking the prefix
   intType = functools.partial( int, base=0 )

   parser = argparse.ArgumentParser( description="Write a PSoC's application image" )
   parser.add_argument( "--image1", required=True )
   parser.add_argument( "--image2" )
   parser.add_argument( "--bootloader", type=intType, required=True )
   parser.add_argument( "--mfrDate" )
   parser.add_argument( "--mfrSerial" )
   parser.add_argument( "--psocType", default="single", help="'single' or 'dual'" )
   parser.add_argument( "--chunk", type=intType )

   args = parser.parse_args()

   if not ( args.mfrDate or args.mfrSerial ):
      print "mfrDate or mfrSerial must be specified"
      return 1

   fileBase, extension = os.path.splitext( args.image1 )

   if extension != ".cyacd":
      print "Psoc firmware file must end with .cyacd"
      return 1

   # set the chunk to its default if it wasn't passed in.
   if args.chunk:
      chunk = args.chunk
   else:
      if args.psocType == "single":
         chunk = 100
      elif args.psocType == "dual":
         chunk = 57
      else:
         print "Unknown PsocType %s" % args.psocType
         return 1

   newFilePath = fileBase + ".acyacd"

   headers = [ ( "bootloader", hex( args.bootloader ) ),
               ( "psoc_type", args.psocType ),
               ( "chunk", str( chunk ) ) ]
   if args.mfrDate:
      headers.append( ( "mfr_date", args.mfrDate ) )
   if args.mfrSerial:
      headers.append( ( "mfr_serial", args.mfrSerial ) )

   with open( newFilePath, "w" ) as dstFile:
      for key, value in headers:
         dstFile.write( key + ": " + value + "\n" )

      def addPsocImage( fileName, dstFile, index ):
         dstFile.write( "#IMAGE_START_%d\n" % index )
         with open( fileName ) as srcFile:
            for line in srcFile:
               # Replacing all newlines with Unix newlines
               dstFile.write( line.strip() + "\n" )
         dstFile.write( "#IMAGE_END_%d\n" % index )

      addPsocImage( args.image1, dstFile, 0 )
      if ( args.image2 ):
         addPsocImage( args.image2, dstFile, 1 )

   return 0

if __name__ == "__main__":
   sys.exit( main() )
