#!/usr/bin/env python
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
#
# Recursively synchronize (archive) the specified source directory by
# copying files to the specified archive directory. If the copy is successful
# optionally delete the source file to free up space in the source directory.
#
# Since the source directories are in memory file systems, keeping space to a
# minimum ensures there is room for agents to write log and quicktrace entries
# to log files and quicktrace files respectively, as well as to create core
# files as needed. This is very desirable so that agents do not receive I/O errors
# trying to write their log and quicktrace data as that can cause them to stop
# writing these records.
#
# Example command usage:
# ---------------------
#
#      python archivetree.py --src /var/dir --archive /archive
# or
#      python archivetree.py -s /var/dir -a /archive
#
# Specifies to copy files recursively from source directory /var/dir
# to target archive directory /archive. The -a or --archive and the -s or --src
# parameters are required parameters.
#
# As shown abbreviations may be used if desired. The default action is not to
# delete successfully copied source files.
#
# Note that for ASU2 the -s --src parameter can be a file not a directory so the
# code was enhanced to support the src parameter as specifying either a file or
# a directory. ASU2 need this to archive files in a specific order.
#
# Additional command options:
# --------------------------
#
# In addition to the -s --src and -a --archive required parameters:
#
# There are several options, a: -e --exemptdirs option that allows you to specify the
# path for one or more directories with the src tree being archive that you would
# like omitted from archival. A comma separated list can be supplied as input.
#
# Here are common invocations of this scripts:
# --------------------------------------------
#
# To archive (and remove fully written) core files:
#
#      python archivetree.py -s /var/core -a /archive
#
# To archive (and remove) log files:
#
#      python archivetree.py -s /var/log -a /archive -e /var/log/qt
#
# To archive quicktrace files:
#
#      python archivetree.py -s /var/log/qt -a /archive
#
# another option is -f --flush will do a file flush after each file copy so that
# the archived file(s) will be safely copied to their archive destination before
# the script returns. ASU2 needed this to ensure each file was flushed after archive.
#
# Here is an example invocation using the flush option:
# ----------------------------------------------------
#
# To sync a copy of the messages file to the archive and ensure it has been copied
# before returning:
#
#      python archivetree.py -s /var/log/messages -a /archive -f
#
# or the equivalent request using the longer command format:
#
#      python archivetree.py --src /var/log/messages --archive /archive --flush
#
#
# Some information about how each of these invocations are processed.
# ------------------------------------------------------------------
#
#      python archivetree.py -s /var/core -a /archive
#
# This is used to archive and remove core files that haven't been written to
# (modified) for at least the minimum specified duration. Core files that are
# eligible for archival are copied to the same path in the archive
# (i.e. /archive/var/core) and once successfully copied to the archive, a file
# is removed from the source location (i.e. /var/core) to minimize filesystem
# space usage. The deleted core file is replaced by a symlink to it's new
# location in the /archive filesystem.
#
#      python archivetree.py -s /var/log -a /archive -e /var/log/qt
#
# This is used to archive log files that have been rotated and so are not the
# active log files as well as any open files to check point active files.
# Rotated log files are named and managed by logrotate.
#
# The default name format for rotated files is of the form foo.# for uncompressed
# files and foo.#.gz for compressed files. Rotated files are renamed when rotated
# so that the newest rotated file # is set to 1, the second oldest # is set to 2
# and so on. Using this scheme logrotate can set a maximum number for the oldest
# file it wants to keep. Every time a rotate occurs, logrotate must rename the
# rotated files in the order from oldest to newest, so if there are currently
# 4 versions of a logfile (including the active file), the oldest is moved from
# # 3 to #4, the second oldest from #2 to #3 and the third oldest from #1 to #2.
# The active logfile is copied to #1 and the active file is truncated (another
# more sophisticated mechanism is available to alow renaming here too but our agents
# do not support this mechanism). Our logrotate code has been using this naming
# scheme.
#
# The problem with this naming scheme is that really complicates archiving log
# this because our archival process runs as an independent process from logrotate.
# We do this so the archival process does not interfere with the normal operation
# of logrotate. Renaming and copying files makes it difficult to ensure we have
# copied the correct rotated files in the correct order since there is no
# serialization between logrotate and the archival process. So a better scheme
# is to rename the files to a unique but meaningful name when they created so there
# is no race condition. We discovered a way to do this with logrotate and so
# we are using that approach. This ensures much simpler serialization and also
# prevents unnecessary copies.
#
# We do check that rotated log files are not open when we copy them to the archive.
# Active log files are copied while open to ensure we have the latest version.
# Any successfully copied rotated log files are deleted to keep space available
# in the source memory filesystem (i.e. /var/log).
#
#      python archivetree.py -s /var/log/qt -a /archive
#
# This is used to archive the latest version of the agent's quicktrace files.
# The quicktrace files are highspeed memory mapped files that are
# persisted into /tmpfs that are used for highspeed logging by the agents. The
# quicktrace files wrap after they reach a maximum fixed size to keep their memory
# footprint modest.
#
# We use copyfile not rsync to copy quicktace files as rsync tries to recopy
# a file if it changes while copying and this can corrupt the
# resulting copied file. Since quicktrace files are memory mapped and fixed size
# they are never deleted form the source memory filesystem (i.e. /var/log/qt).
#
# For file copies in general we use python copy2 instead of rsync because
# rsync is more expensive requirng a process to be created and deleted with
# each invocation.
#
import argparse
import ctypes
import errno
import grp
import hashlib
import math
import os
import re
import subprocess
import sys
import syslog
import time
import timeit  # pylint: disable=W0611
import Tac     # pylint: disable=W0611
from pwd import getpwnam

PROGRAM_NAME = ""

# If SSD device requires archive user id and group id set so entries in /etc/passwd
# for archive and entry in /etc/group for eosadmin. Archive files will be created
# as archive:eosadmin because /mnt/drive is defined as root:eosadmin rwx:rwx:---
# so that is the only way we can write to /mnt/drive
ARCHIVE_AS_ROOT = False
ARCHIVE_ID_ACTIVE = False
ARCHIVE_ID_USER_NAME = "archive"
ARCHIVE_ID_GROUP_NAME = "eosadmin"
ARCHIVE_GROUP_ID = 0
ARCHIVE_USER_ID = 0
ROOT_GROUP_ID = 0
ROOT_USER_ID = 0
FLUSH_FILE = False

class Archive( object ):
   """ Archive class that holds data about the SSD file archive and methods to
       operate on SSD file archive data.
   """
   __DEFAULT_MIN_MODTIME = 70      # Files should be unchanged for 70 seconds

   minModTime = __DEFAULT_MIN_MODTIME

   def __init__( self, srcDir, archiveDir, exemptDirs ):
      """ __init__ method for Archive Class """
      self.srcDir = srcDir
      self.archiveDir = archiveDir
      # The exempt directory is filled in here if the user specified one or more
      # directories to exempt from checking, specified as a comma separated list.
      if exemptDirs is not None:
         self.exemptDirs = exemptDirs.split( ',' )
      else:
         self.exemptDirs = None

   def exemptDirCheck( self, dirPath ):
      """
      exemptDirCheck():
      ----------------
      Check to see if this is a user specified directory that we should skip over
      The user can specify one more directories to skip over see -e parameter
      """
      if self.exemptDirs is None:
         return False

      return True if dirPath in self.exemptDirs else False

   def rotatedLogFileCheck( self, fileName ):
      """
      rotatedLogFileCheck():
      ---------------------
      Determine if the fileName indicates that this file is a logrotate
      log file.
      """
      if re.match( r'.*\.([0-9]+)(.gz)?$', fileName ):
         return True
      if re.match( r'.*\.(\d+\-\d+\-\d+\_\d+)(.gz)?$', fileName ):
         datevalue = fileName.split( '.' )[ -2 ]
         try:
            time.strptime( datevalue, '%Y-%m-%d_%s' )
            return True
         except ValueError:
            pass

      return False

   def qtFileCheck( self, fileName ):
      """
      qtFileCheck():
      -------------
      Determine if the fileName indicates that this file is a quicktrace
      log file.
      """
      if re.match( r'.*\.qt$', fileName ):
         return True

      return False

   def coreFileCheck( self, fileName ):
      """
      coreFileCheck():
      ---------------
      Determine if the fileName indicates that this file is a core file.
      """
      if re.match( r'core\..*(.gz)?$', fileName ):
         return True

      return False

#
# End of class Archive
#

class DirectoryArchive( object ):
   """ Directory class that holds data about a specific directory that we are
       we are archving and the methods to operate on that data
   """
   def __init__( self, archive ):
      """ __init__ method for DirectoryArchive Class """
      self.archive = archive
      self.renamePrefix = ".__TemporaryRenameFile__." # Prefix used for renamed files

   def rootDir( self ):
      """
      Return the src or tmpfs root directory (the DirectoryArchive is acting on).
      """
      return self.archive.srcDir

   def archiveRoot( self ):
      """
      Return archive root directory.
      """
      return self.archive.archiveDir

   def minModTime( self ):
      """
      Return the minimum modification time.
      """
      return self.archive.minModTime

   def exemptDirCheck( self, dirPath ):
      """
      Exempt directory (if any) returned
      """
      return self.archive.exemptDirCheck( dirPath )

   def coreFileCheck( self, fileName ):
      """
      Check if fileName is a core file.
      """
      return self.archive.coreFileCheck( fileName )

   def rotatedLogFileCheck( self, fileName ):
      """
      Check if fileName ia a rotated log file.
      """
      return self.archive.rotatedLogFileCheck( fileName )

   def qtFileCheck( self, fileName ):
      """
      Check if the file is a quicktrace file.
      """
      return self.archive.qtFileCheck( fileName )

   def archiveDirPath( self, dirPath ):
      """
      Return the archive directory path.
      """
      relDirPath = dirPath.strip( '/' )
      destDirPath = os.path.join( self.archiveRoot(), relDirPath )
      return destDirPath

   def archiveVerifyDir( self, dirPath ):
      """
      Verifies corresponding achive dir exists in case it hasn't been created
      yet and we need to creat it.
      """
      destDirPath = self.archiveDirPath( dirPath )
      if not os.path.exists( destDirPath ):
         try:
            if ARCHIVE_ID_ACTIVE and ARCHIVE_AS_ROOT is False:
               os.setegid( ARCHIVE_GROUP_ID )
               os.seteuid( ARCHIVE_USER_ID )

            os.makedirs( destDirPath )

            if ARCHIVE_ID_ACTIVE and ARCHIVE_AS_ROOT is False:
               os.seteuid( ROOT_USER_ID )
               os.setegid( ROOT_GROUP_ID )

         except ( IOError, OSError ) as e:
            if e.errno == errno.ENOSPC or e.errno == errno.EDQUOT:
               sys.exit( 0 )

            syslog.syslog( '%%ERR-LOG: Error create directory dst: %s err: %s'
               % ( destDirPath, os.strerror( e.errno ) ) )
            sys.exit( 0 )

      return destDirPath

   def processDir( self, dirPath ):
      """
      Given the dirPath determine if this a directory we want to exempt from
      archiving as we walk the tree. A list of none or more exempted
      directories may be supplied to archivetree as an in put parameter.
      """
      # If the directory was exempted don't process it
      if self.exemptDirCheck( dirPath ):
         return False
      # Make sure we have a corresponding directory in the archive
      self.archiveVerifyDir( dirPath )
      return True

   def archiveFilePath( self, fileName, filePath ):
      """
      Returns the archive destination path for the given fileName and filePath.
      Should be used to convert a tmpfs src file path to a archive dst file
      path.
      """
      dirPath = os.path.dirname( filePath )
      destDirPath = self.archiveVerifyDir( dirPath )
      destFilePath = os.path.join( destDirPath, fileName )
      return destFilePath

   def compressedFilePath( self, fileName, filePath ):
      """
      Create archive destination with added compression .gz suffix in name
      """
      dirPath = os.path.dirname( filePath )
      destDirPath = self.archiveVerifyDir( dirPath )
      compressedFileName = fileName + ".gz"
      compressedFilePath = os.path.join( destDirPath, compressedFileName )
      return compressedFilePath

   def filesCheckModtimesEqual( self, modtime1, modtime2 ):
      """
      Check modification times for src and archive file to see if the src
      hasn't changed and still matches the archive copy of the file.
      """
      if math.floor( modtime1 ) == math.floor( modtime2 ):
         return True
      return False

   def verifyArchiveFileSize( self, fileName, filePath ):
      """ Verify that src and dst file are same size and same modtimes """
      if not os.path.isfile( filePath ):
         return False
      dstFilePath = self.archiveFilePath( fileName, filePath )
      if not os.path.isfile( dstFilePath ):
         return False
      fstats = os.stat( filePath )
      dstats = os.stat( dstFilePath )
      if fstats is None or dstats is None:
         return False
      if fstats.st_size == dstats.st_size:
         # If src file is link same size is sufficient, link mtime can be different
         if os.path.islink( filePath ):
            return True
         # Regular files must have same mtime or src may have changed with same size
         if self.filesCheckModtimesEqual( fstats.st_mtime, dstats.st_mtime ):
            return True
      return False

   def fileRemove( self, filePath ):
      """
      Remove the file designated by filePath and return whether the file
      was removed in case we want to track how many files were removed.
      """
      try:
         # We're here if the file was not in use, so we can remove it
         os.remove( filePath )
      except ( IOError, OSError ) as e:
         syslog.syslog( '%%ERR-LOG: Failed removing file: %s err: %s' %
               ( filePath, os.strerror( e.errno ) ) )
         return False
      # The file was successfully removed
      return True

   def fileCopy( self, fileName, filePath ):
      """ If file doesn't need to be copied or can't be copied returns False """
      # Check if file or a compressed version for large files is in the archive
      destFilePath = self.archiveFilePath( fileName, filePath )
      statPath = filePath

      # Check that the file still exists at the given location
      if not os.path.isfile( filePath ):
         return False

      # Special handling of link files
      if os.path.islink( filePath ):
         linkPath = os.readlink( filePath )
         # Skip archive link check, by defintion that is up to date
         archiveDir = self.archiveRoot()
         if os.path.commonprefix( [ linkPath, archiveDir ] ) == archiveDir:
            return False
         # Not an archive link, get stats for the link destination
         statPath = linkPath
         filePath = linkPath

      try:
         stats = os.stat( statPath )
         dStats = None
         try:
            dStats = os.stat( destFilePath )
         except OSError as err:
            if err.errno == errno.ENOENT:
               pass

         # If src path starts with /proc can't check file modify time so use md5
         # to determine if the file has changed and we need to copy it.
         if filePath.startswith( "/proc" ):
            if os.path.isfile( destFilePath ):
               HASH = hashlib.new( 'md5' )
               HASH.update( open( filePath, 'rb' ).read() )
               md5src = HASH.hexdigest()
               HASH.update( open( destFilePath, 'rb' ).read() )
               md5dst = HASH.hexdigest()
               if md5src == md5dst:
                  # No copy needed
                  return False
         else:
            if stats.st_size == 0 and dStats != None and dStats.st_size == 0:
               return False
            if self.verifyArchiveFileSize( fileName, filePath ):
               return False

         cmd = [ 'cp', '-a', '-u', filePath, destFilePath ]
         if filePath.startswith( "/proc" ):
            # Why skip saving attributes? /proc doesn't have attributes
            # so for any files like that we don't want the copy to fail
            # just because of a limitation of the filesystem implementation
            cmd = [ 'cp', '-u', filePath, destFilePath ]
         subprocess.call( cmd )

         if os.path.isfile( destFilePath ):
            if ARCHIVE_ID_ACTIVE and ARCHIVE_AS_ROOT is False:
               os.chown( destFilePath, ARCHIVE_USER_ID, ARCHIVE_GROUP_ID )
               os.setegid( ARCHIVE_GROUP_ID )
               os.seteuid( ARCHIVE_USER_ID )

            if FLUSH_FILE:
               with open( destFilePath, "ab" ) as filePtr:
                  filePtr.flush()
                  os.fsync( filePtr.fileno() )
                  filePtr.close()

      except ( IOError, OSError ) as err:
         if err.errno == errno.ENOSPC or err.errno == errno.EDQUOT:
            try:
               os.remove( destFilePath )
            except ( IOError, OSError ):
               pass
         elif err.errno == errno.EACCES:
            syslog.syslog( '%%ERR-LOG: Error access copy src: %s dst: %s err: %s'
                  % ( filePath, destFilePath, os.strerror( err.errno ) ) )
            return False
         elif err.errno != errno.ENOENT:
            syslog.syslog( '%%ERR-LOG: Error copy src: %s dst: %s err: %s'
                  % ( filePath, destFilePath, os.strerror( err.errno ) ) )
            sys.exit( 0 )

      finally:
         if os.path.isfile( destFilePath ):
            if ARCHIVE_ID_ACTIVE and ARCHIVE_AS_ROOT is False:
               os.seteuid( ROOT_USER_ID )
               os.setegid( ROOT_GROUP_ID )

      return True

   def fileModifiedCheck( self, filePath ):
      """
      Check if file has remained unmodified for the minimum time.
      # SymLinks are handled differently depending on whether they point to
      # the /archive filesystem
      """
      if os.path.islink( filePath ):
         linkPath = os.readlink( filePath )
         # Link to an archive file means we're up to date
         if os.path.commonprefix( [ linkPath, self.archiveRoot() ] ) == \
            self.archiveRoot():
            return True
         # Link to an internal file should process the link
         filePath = linkPath
      curtime = time.time()
      try:
         stats = os.stat( filePath )
      except ( IOError, OSError ) as err:
         if err.errno != errno.ENOENT:
            syslog.syslog( '%%ERR-LOG: Error check modtime file %s err: %s' %
                  ( filePath, os.strerror( err.errno ) ) )
         return True
      if not stats:
         return True
      difftime = curtime - stats.st_mtime
      if difftime < self.minModTime():
         return True
      # The file has not been modified for at least the minimum modtime
      return False

   def fileCopyIfModified( self, fileName, filePath ):
      """
      Copy file to the archive if it has been modified
      Common check for nmodified file for required time before copy
      """
      if self.fileModifiedCheck( filePath ):
         return False
      # Safe to copy and if copy successful, remove the file
      if self.fileCopy( fileName, filePath ):
         return True
      return False

   def fileCreateSymlink( self, fileName, filePath, tmpfsFilePath ):
      """
      Given a destination fileName and filePath create a symlink from
      the given tmpfsFilePath to the destination archive file.
      Always flush the symlink to make sure we aren't failing creating
      it, which could happen if we run out of space or the filesystem
      has an I/O error and goes R/O. The caller will flush the dest
      file to make sure it is already successfully copied.
      """
      destFilePath = self.archiveFilePath( fileName, filePath )
      absDestFilePath = os.path.abspath( destFilePath )
      try:
         if ARCHIVE_ID_ACTIVE and ARCHIVE_AS_ROOT is False:
            os.setegid( ARCHIVE_GROUP_ID )
            os.seteuid( ARCHIVE_USER_ID )

         os.symlink( absDestFilePath, tmpfsFilePath )

         if ARCHIVE_ID_ACTIVE and ARCHIVE_AS_ROOT is False:
            os.seteuid( ROOT_USER_ID )
            os.setegid( ROOT_GROUP_ID )

      except ( IOError, OSError ) as e:
         if e.errno == errno.ENOSPC or e.errno == errno.EDQUOT:
            sys.exit( 0 )

         syslog.syslog( '%%ERR-LOG: Error symlink, dst: %s from: %s err: %s'
            % ( absDestFilePath, tmpfsFilePath, os.strerror( e.errno ) ) )
         return False

      return True

   def ensureSrcFileIsSymlink( self, fileName, filePath, tmpFilePath ):
      """ Ensure the src file is converted to a symlink """
      if os.path.islink( filePath ):
         return

      if self.fileCreateSymlink( fileName, filePath, tmpFilePath ):
         try:
            # For file archival must do move as root because core files are
            # written as root and the move will fail if run as archive id.
            # Replace the actual file with the symlink, only here if file
            # fileCopy completed. Then replace file with symlink to file.
            # Use os.rename since both files live in the same directory.
            os.rename( tmpFilePath, filePath )

         except ( IOError, OSError ) as e:
            if e.errno == errno.ENOSPC or e.errno == errno.EDQUOT:
               sys.exit( 0 )

            syslog.syslog( '%%ERR-LOG: Error move, src: %s dst: %s err: %s'
               % ( tmpFilePath, filePath, os.strerror( e.errno ) ) )
            sys.exit( 0 )

      return

   def processCoreFile( self, fileName, filePath ):
      """
      Process agent core file
      Only copy core files that are not open and have not been modified for
      the minimum time and once successfully copied they should be deleted
      and replaced a symlink to their new archive location. We do a careful
      replacement of the deleted file with a symlink. Always force flush on
      core file copy to archive so that if the copy fails we know and avoid
      clobbering core file that did not successfully copy file to archive with
      symlink.
      """
      global FLUSH_FILE
      if self.fileModifiedCheck( filePath ):
         return

      tmpPath = os.path.dirname( filePath )
      tmpName = ".__tmp__" + fileName
      tmpFilePath = os.path.join( tmpPath, tmpName )

      # Only copy if file is not already copied to archive
      if self.verifyArchiveFileSize( fileName, filePath ):
         self.ensureSrcFileIsSymlink( fileName, filePath, tmpFilePath )
         return

      saveFlushFile = FLUSH_FILE
      FLUSH_FILE = True
      try:
         if ARCHIVE_ID_ACTIVE and ARCHIVE_AS_ROOT is False:
            os.seteuid( ROOT_USER_ID )
            os.setegid( ROOT_GROUP_ID )
         self.fileCopy( fileName, filePath )
         destFilePath = self.archiveFilePath( fileName, filePath )
         os.chown( destFilePath, ARCHIVE_USER_ID, ARCHIVE_GROUP_ID )
      except Exception as err:   # pylint: disable-msg=W0703
         print 'Error: %s' % err.message
      finally:
         if ARCHIVE_ID_ACTIVE and ARCHIVE_AS_ROOT is False:
            os.setegid( ARCHIVE_GROUP_ID )
            os.seteuid( ARCHIVE_USER_ID )

      FLUSH_FILE = saveFlushFile

      # Only create symlink if file is fully copied to archive
      if not self.verifyArchiveFileSize( fileName, filePath ):
         return

      self.ensureSrcFileIsSymlink( fileName, filePath, tmpFilePath )

      return

   def processRotatedLogFile( self, fileName, filePath ):
      """
      Process logrotate generated rotated log file
      Only copy rotated log files if they have not been modified for a min time
      We let logrotate remove rotated log files with it's file management
      """
      self.fileCopyIfModified( fileName, filePath )
      return

   def processQuicktraceFile( self, fileName, filePath ):
      """
      Process agent quicktrace in memory file. Quicktrace files use copyfile
      instead of rsync to avoid file corruption. The file is never deleted.
      """
      # We don't need to check if the copy was succcesful, this is best effort
      self.fileCopy( fileName, filePath )
      return

   def processRegularFile( self, fileName, filePath ):
      """
      Process a file not recognized as a special type of file based on naming
      A regular file may be copied to the Archive but is not deleted
      """
      self.fileCopy( fileName, filePath )
      return

   def processFile( self, filePath ):
      """
      Process a file that may need to be archived based on attributes, the
      type and or location file.
      """
      dirPath = os.path.dirname( filePath )
      fileName = os.path.basename( filePath )
      # If the directory for this file was exempted don't process it
      if self.exemptDirCheck( dirPath ):
         return

      if self.qtFileCheck( fileName ):
         self.processQuicktraceFile( fileName, filePath )
      elif self.coreFileCheck( fileName ):
         self.processCoreFile( fileName, filePath )
      elif self.rotatedLogFileCheck( fileName ):
         self.processRotatedLogFile( fileName, filePath )
      else:
         self.processRegularFile( fileName, filePath )
      return

   def directoryWalk( self ):
      """
      Recursively descend the directory tree.
      We process each file looking to see if we need to copy it to it's
      corresponding archive directory. We do our copy check as we
      encounter each file.
      We create corresponding archive directories as needed along the way.
      We omit any exempted directories as we descend the tree.
      We also skip operating on any recently modified file when requested.
      """
      if not self.processDir( self.rootDir() ):
         return

      # Skip processing any exempted directories
      for root, dirs, filenames in os.walk( self.rootDir(), topdown=True ):
         for dirPath in dirs:
            self.processDir( os.path.join( root, dirPath ) )
         for filename in filenames:
            self.processFile( os.path.join( root, filename ) )
      return

#
# End of class DirectoryArchive
#

def main( srcDir, archiveDir, exemptDirs, flush, rootid ):
   """
   The main function that processes input to the program and drives the
   actions that the program performs.
   """
   global PROGRAM_NAME
   global FLUSH_FILE
   global ARCHIVE_GROUP_ID
   global ARCHIVE_USER_ID
   global ARCHIVE_ID_ACTIVE
   global ARCHIVE_AS_ROOT

   PROGRAM_NAME = sys.argv[ 0 ]

   if flush:
      FLUSH_FILE = True

   if rootid:
      ARCHIVE_AS_ROOT = True

   try:
      ARCHIVE_USER_ID = getpwnam( ARCHIVE_ID_USER_NAME ).pw_uid
      ARCHIVE_GROUP_ID = grp.getgrnam( ARCHIVE_ID_GROUP_NAME ).gr_gid
      ARCHIVE_ID_ACTIVE = True
   except ( IOError, OSError ) as e:
      # OK If archive user id or group do not exist, that indicates no SSD present
      syslog.syslog( '%%ERR-LOG: No archive user id %s group id: %s found: %s'
                      % ARCHIVE_ID_USER_NAME, ARCHIVE_ID_GROUP_NAME,
                      os.strerror( e.errno ) )

      ARCHIVE_ID_ACTIVE = False

   # Add check if archiveDir is invalid or R/O we can't archive anything
   if not os.path.exists( archiveDir ) or not os.path.isdir( archiveDir ):
      print "Error: invalid archive directory path specified:", archiveDir
      syslog.syslog( '%%ERR-LOG: invalid archive directory path specified: %s'
                     % archiveDir )
      return False
   if not os.access( archiveDir, os.W_OK ):
      print "Error: R/O archive directory: %s - cannot archive files" % archiveDir
      syslog.syslog( '%%ERR-LOG: R/O archive directory: %s - cannot archive files'
                     % archiveDir )
      return False

   # Normal case is that srcDir is the src directory we archive from but for ASU2
   # srcDir can be a file, they archive one file at a time, so handle that case
   if os.path.isdir( srcDir ):
      archive = Archive( srcDir, archiveDir, exemptDirs )
   elif os.path.isfile( srcDir ):
      srcDirPath = os.path.dirname( srcDir )
      archive = Archive( srcDirPath, archiveDir, exemptDirs )
   else:
      syslog.syslog( '%%ERR-LOG: Invalid src directory of file path specified: %s'
                     % srcDir )
      print "Error: invalid src directory or file path specified:", srcDir
      return False

   dirArch = DirectoryArchive( archive )

   syslog.openlog( 'archivetree', 0, syslog.LOG_USER )
   if os.path.isdir( srcDir ):
      dirArch.directoryWalk()
   elif os.path.isfile( srcDir ):
      dirArch.processFile( srcDir )
   syslog.closelog()

   if flush:
      libc = ctypes.CDLL( "libc.so.6" )
      libc.sync()

   return True

if __name__ == '__main__':
   parser = argparse.ArgumentParser( prog=sys.argv[ 0 ] )

   parser.add_argument( '-a', '--archive',
                        help='target archive directory (i.e., /archive)',
                        required=True )
   # Don't operate on the directories specified in the comma separated list
   parser.add_argument( '-e', '--exemptdirs',
                        help='directory(s) to exempt from copy',
                        default=None )
   parser.add_argument( '-f', '--flush',
                        help='flush each archive file before returning',
                        action="store_true" )
   parser.add_argument( '-r', '--root',
                        help='archive file(s) as root id not archive id',
                        action="store_true" )
   parser.add_argument( '-s', '--src',
                        help='source dirpath or filepath (i.e. /var/log)',
                        required=True )

   args = parser.parse_args()

   main( args.src, args.archive, args.exemptdirs, args.flush, args.root )
