#!/usr/bin/env python
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import datetime as dt
import argparse
import os
import re
import sys
import subprocess

import Tac

defaultLogFileDir = "/mnt/flash/"
defaultLogFileName = "autobw-Rsvp.log"
defaultQtDir = "/var/log/qt/"
defaultQtFileName = "autobw-Rsvp.qt"
defaultQtLevels = "0"
defaultLogFileRotateSize = 5000000
defaultNumOfLogFiles = 4
defaultPattern = r"^\d\d\d\d-\d\d-\d\d \d\d:\d\d:\d\d\.\d+ \d+ \d+\.\d+, "\
                 r"\+{0,1}\d+ \"Tunnel: "

def parseArgs( *args ):
   parser = argparse.ArgumentParser(
         description="append quicktrace file to a log file" )
   parser.add_argument( 'qtFile', nargs='?', help="full path of quicktrace file",
         default=os.path.join( defaultQtDir, defaultQtFileName ),
         metavar='<quicktrace file>' )
   parser.add_argument( 'logFile', nargs='?', help="full path of log file",
         metavar='<log file>' )
   parser.add_argument( '-n', '--log-file-num',
                     dest='numOfLogFiles', type=int,
                     help='number of kept log files',
                     metavar='NUM',
                     default=defaultNumOfLogFiles )
   parser.add_argument( '-s', '--log-file-size',
                     dest='logFileRotateSize', type=int,
                     help='log file size cap',
                     metavar='BYTES',
                     default=defaultLogFileRotateSize )
   parser.add_argument( '-d', '--log-file-dir',
                     dest='logFileDir',
                     help='log file directory (not used when <log file> is set)',
                     metavar='DIRECTORY',
                     default=defaultLogFileDir )
   parser.add_argument( '-f', '--log-file-name',
                     dest='logFileName',
                     help='log file name (not used when <log file> is set)',
                     metavar='FILENAME',
                     default=defaultLogFileName )
   parser.add_argument( '-l', '--levels',
                     dest='levels',
                     help='quicktrace levels',
                     default=defaultQtLevels )
   parser.add_argument( '-p', '--pattern',
                     dest='pattern',
                     help='quicktrace filter regexp pattern',
                     default=defaultPattern )
   parsedArgs = parser.parse_args( *args )

   # determine log file
   if parsedArgs.logFile:
      logFile = parsedArgs.logFile
      logFileDir = os.path.dirname( logFile )
      logFileName = os.path.basename( logFile )
      if not logFileName:
         logFileName = defaultLogFileName
   else:
      logFileDir = parsedArgs.logFileDir
      logFileName = parsedArgs.logFileName

   # determine qt levels
   if parsedArgs.levels == "all":
      qtLevels = None
   else:
      qtLevels = parsedArgs.levels

   return ( parsedArgs.qtFile, logFileDir, logFileName,
            parsedArgs.logFileRotateSize, parsedArgs.numOfLogFiles, qtLevels,
            parsedArgs.pattern )

def main():
   args = parseArgs()
   err = appendQuickTrace( *args )
   if err is not None:
      sys.stderr.write( err )

# just in case python does not do this on its own
# let's call fsync on the file when we close it
class FsyncFile( file ):
   def __exit__( self, *args ):
      self.flush()
      os.fsync( self.fileno() )
      return super( FsyncFile, self ).__exit__( *args )

def appendQuickTrace( qtFile, logFileDir, logFileName,
      logFileRotateSize=defaultLogFileRotateSize,
      numOfLogFiles=defaultNumOfLogFiles,
      qtLevels=defaultQtLevels,
      pattern=defaultPattern ):
   if not os.path.exists( qtFile ):
      return qtFile + " does not exist\n"
   if not os.path.exists( logFileDir ):
      return logFileDir + " does not exist\n"
   logFile = os.path.join( logFileDir, logFileName )
   firstTs = None
   lastTs = 0.0
   lastLine = None
   if os.path.exists( logFile ):
      # get last timestamp in the logfile
      lastLine = Tac.run( [ "tail", "-n", "-1", logFile ],
                          stdout=Tac.CAPTURE ).strip()
      try:
         date, time = lastLine.split()[ : 2 ]
         lastTs = float(
            dt.datetime.strptime( date + " " + time, "%Y-%m-%d %H:%M:%S.%f" ).
            strftime( '%s.%f' ) )
      except ValueError:
         lastTs = os.stat( logFile ).st_mtime

      # rotate logfile, if needed
      size = os.path.getsize( logFile )
      if size >= logFileRotateSize:
         # rename file
         rotatedLogFile = os.path.join( logFileDir,
            logFileName + "." + dt.datetime.now().strftime( "%Y-%m-%d_%s" ) )
         os.rename( logFile, rotatedLogFile )
         # compress file
         Tac.run( [ "gzip", rotatedLogFile ], ignoreReturnCode=True,
                  stdout=Tac.DISCARD )
      # keep only most recent logs
      cmd = "ls -1t %s*gz 2>/dev/null | tail -n +%d | xargs rm -f" %\
               ( logFile, numOfLogFiles + 1 )
      Tac.run( [ "bash", "-c", cmd ], ignoreReturnCode=True, stdout=Tac.DISCARD )

   # read qt, append new lines to logfile
   cmd = [ "qtcat" ]
   if qtLevels is not None:
      cmd += [ "-l", qtLevels ]
   cmd.append( qtFile )
   qtrace = subprocess.Popen( cmd, bufsize=-1, stdin=subprocess.PIPE,
         stdout=subprocess.PIPE, stderr=subprocess.PIPE, close_fds=True )
   qtrace.stdin.close() # in case qtcat gets into pdb
   rePattern = re.compile( pattern )
   while True:
      qtLine = qtrace.stdout.readline()
      if not qtLine:
         break
      if not rePattern.match( qtLine ):
         continue
      line = qtLine.split()
      if firstTs is None:
         date, time = line[ : 2 ]
         ts = float(
            dt.datetime.strptime( date + " " + time, "%Y-%m-%d %H:%M:%S.%f" ).
            strftime( '%s.%f' ) )
         firstTs = ts
      else:
         delta = line[ 3 ].rstrip( ',' )
         ts = firstTs + float( delta )
      if ts > lastTs:
         # double check that we don't duplicate last line,
         # in case we had a rounding error in timestamp calculation
         if lastLine != " ".join( line[ : 2 ] + line[ 5 : ] ):
            break
   with FsyncFile( logFile, 'a' ) as f:
      while qtLine:
         if rePattern.match( qtLine ):
            line = qtLine.split()
            # Quicktrace entries 2, 3 and 4 are: level, time in seconds since start
            # of the trace, time in ticks since previous message.
            # We don't need them in the log
            f.write( " ".join( line[ : 2 ] + line[ 5 : ] ) + "\n" )
         qtLine = qtrace.stdout.readline()
   err = qtrace.stderr.readlines()
   qtrace.wait()
   if qtrace.returncode:
      return "qtcat terminated with error: " + err[ 0 ]

if __name__ == "__main__":
   main()
