#!/usr/bin/env python
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Tracing
import os

t0 = Tracing.trace0
os.environ[ "TRACEFILE" ] = "/tmp/policer.log"
Tracing.traceSettingIs( "InterfacePolicer*" )

excludeRuleData = {}

tcCmd = "/usr/sbin/tc "

qdiscShowCommand = tcCmd + "qdisc show dev %s"
qdiscAddCommand = tcCmd + "qdisc add dev %s handle ffff: ingress"
qdiscDelCommand = tcCmd + "qdisc del dev %s handle ffff: ingress"
qdiscAttachedString = "qdisc ingress ffff: parent ffff:fff1"

filterShowCommand = tcCmd + "filter show dev %s root"
filterAddCommand = tcCmd + "filter add dev %s parent " \
                   "ffff: prio 2 u32 match u32 0 0 " \
                   "police rate 10mbps burst 100k mtu 10000b drop"
filterDelCommand = tcCmd + "filter del dev %s parent ffff: prio 2"
filterAttachedString = "rate 80Mbit burst 100Kb mtu 10000b action"

excludeFilterPortAddCommand = tcCmd + \
      "filter add dev %s parent ffff: prio 1 u32 match ip %s %d 0xffff flowid" \
      " ffff:%d"
excludeFilterDelCommand = tcCmd + "filter del dev %s parent ffff: prio 1"

def runCmd( cmd ):
   t0( "Running command ", cmd )
   output = ""
   error = 0
   try:
      output = Tac.run( cmd.split(), stdout=Tac.CAPTURE, asRoot=True )
   except Tac.SystemCommandError as e:
      error = e.error
   return ( output, error )

def attachQdiscOnIntf( intf ):
   qdiscOutput, err = runCmd( qdiscShowCommand % intf )
   if err:
      return False
   if qdiscAttachedString not in qdiscOutput:
      _, err = runCmd( qdiscAddCommand % intf )
      return err == 0
   return True

def attachExcludeFilters( intf ):
   flowId = 1
   for port in excludeRuleData[ 'sport' ]:
      runCmd( excludeFilterPortAddCommand % ( intf, 'sport', port, flowId ) )
      flowId += 1

   for port in excludeRuleData[ 'dport' ]:
      runCmd( excludeFilterPortAddCommand % ( intf, 'dport', port, flowId ) )
      flowId += 1

def attachFilterOnIntf( intf, force=False ):
   filterOutput, err = runCmd( filterShowCommand % intf )
   if err:
      return False

   if force:
      if filterOutput:
         removeFilterFromIntf( intf )
      _, err = runCmd( filterAddCommand % intf )
      attachExcludeFilters( intf )
      return err == 0

   if filterAttachedString not in filterOutput:
      _, err = runCmd( filterAddCommand % intf )
      attachExcludeFilters( intf )
      return err == 0

   return True

def removeQdiscFromIntf( intf, force=False ):
   runCmd( qdiscShowCommand % intf )
   runCmd( qdiscDelCommand % intf )

def removeExcludeFilters( intf ):
   runCmd( excludeFilterDelCommand % intf )

def removeFilterFromIntf( intf ):
   runCmd( filterShowCommand % intf )
   runCmd( filterDelCommand % intf )
   removeExcludeFilters( intf )

def attachPolicerOnIntf( intf ):
   return attachQdiscOnIntf( intf ) and attachFilterOnIntf( intf, force=True )

def removePolicerFromIntf( intf ):
   removeFilterFromIntf( intf )
   removeQdiscFromIntf( intf )
   return True

def attachPolicer( intfList ):
   result = True
   for intf in intfList:
      result = result and attachPolicerOnIntf( intf )
   return result

def removePolicer( intfList ):
   result = True
   for intf in intfList:
      result = result and removePolicerFromIntf( intf )
   return result

def setupExcludeRuleData():
   global excludeRuleData
   aptp = Tac.Type( "Arnet::PrivateTcpPorts" )
   portsToExclude = [ aptp.cvxOobPort, aptp.cvxClusterPort, 
                      aptp.controllerdbDefaultPort ]
   excludeRuleData[ 'sport' ] = portsToExclude[ : ]
   excludeRuleData[ 'dport' ] = portsToExclude[ : ]

def main():
   import argparse
   parser = argparse.ArgumentParser( "Control ingress policing for an interface" )
   group = parser.add_mutually_exclusive_group( required=True )
   group.add_argument( "-a", "--attach", action="store_true", 
                       help="Attach policer to devices" )
   group.add_argument( "-d", "--delete", action="store_true", 
                       help="Remove policer from devices" )
   parser.add_argument( "-i", "--intf", dest="interface", default=[],
                        action="append", help="Device name" )
   args = parser.parse_args()
   
   setupExcludeRuleData()
   if args.attach:
      return attachPolicer( intfList=args.interface )
   if args.delete:
      return removePolicer( intfList=args.interface )

if __name__ == '__main__':
   res = main()
   t0( "Result:", res )
   if not res:
      exit( 1 )
