#!/usr/bin/env python

# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
#
# owner: echron@arista.com
#
# file: EosSsdSpaceMgmtTests.py
#
# synopsis:
# Archive processing tests that generate src and archive files and directories
# that we can use to test our SSD Archive space management code to ensure it is
# working properly.
#
import argparse
import datetime
import errno
import fnmatch
import grp
import inspect
import os
import re
import shutil
import signal
import sys
import time
import traceback
import Tracing
import Tac
import subprocess
from subprocess import (
   PIPE,
   Popen,
)
from pwd import getpwnam

traceHandle = Tracing.Handle("EosSsdSpaceMgmtTests")
t0 = traceHandle.trace0
t1 = traceHandle.trace1

# Important constant values and default paths

scriptName = os.path.basename(__file__)

archivedArchiveFileName = "var_archive."
defaultArchiveDevMountDir = None
defaultArchiveDir = "/archive"
defaultSrcLogDir = "/var/log"
defaultSrcCoreDir = "/var/core"
defaultSrcAgentsDir = defaultSrcLogDir + "/agents"
defaultSrcTestFileDir = defaultSrcLogDir + "/testdir"
defaultTmpfsArchiveDir = "/archive"
defaultArchiveCoreDir = defaultArchiveDir + defaultSrcCoreDir
defaultArchiveLogDir = defaultArchiveDir + defaultSrcLogDir
defaultArchiveAgentsDir = defaultArchiveDir + defaultSrcAgentsDir
defaultArchiveTestFileDir = defaultArchiveDir + defaultSrcTestFileDir

defaultCreateAgentFiles = 128
defaultFileSize = 1048576
defaultNumFiles = 128
defaultNetNsCoreSize = 65536
startingAgentFileProcNum = 100000 # Default can be overriden
oneGB  = 1000000000
oneMB  = 1000000
oneKB  = 1000
oneKiB = 1024

runNum = 1

ARCHIVE_ID_USER_NAME = "archive"
ARCHIVE_ID_GROUP_NAME = "eosadmin"
ARCHIVE_GROUP_ID = 0
ARCHIVE_USER_ID = 0
ROOT_GROUP_ID = 0
ROOT_USER_ID = 0


class ReRaiseException(Exception):
   pass


def fileSystemInfo( fileSystem=None ):
   """
   Return the fileSystem information (typically for the default SSD).
   Values returned can be used to determine what actions to take based on the
   device size and utilization.

   Optional Input: fileSystem mountpoint.
   Returns: device name, device size, amount of space used, amount of unused
            space, percent of total space used, mount point. Note that size, used
            and unused are all in units of KB. The percent used is returned
            directly from Linux df. All values are returned as string values.

   Invoked with the input parameter: --filesysteminfo
   """
   if fileSystem is None:
      fileSystem = defaultArchiveDevMountDir

   if os.path.exists( fileSystem ):
      dfOut = subprocess.Popen( [ "df", "-P", fileSystem ], shell=False,
                                stdout=subprocess.PIPE )
      outPut = dfOut.communicate()[ 0 ]
      dev, size, used, unused, percent, mountPt = outPut.split( "\n" )[ 1 ].split()
      size = str( ( int( size ) * oneKiB ) / oneKB )
      used = str( ( int( used ) * oneKiB ) / oneKB )
      unused = str( ( int( unused ) * oneKiB ) / oneKB )
   else:
      dev = mountPt = ""
      size = used = unused = "0"
      percent = "0%"

   return dev, size, used, unused, percent, mountPt


def calcSizeFromPct( objSize, pctOfObj ):
   """
   Get the size of an object based on the percentage of the total size it uses.
   """
   objSizeFloat = float( objSize )
   return int( float( objSizeFloat ) * ( float( pctOfObj ) / 100 ) )


def createFile( path, fileName, fileSize, srcDest=False ):
   """
   Add a new file with the specified Name and size to the targeted directory
   location. This is needed for testing to speed up our testing process.
   If the srcDest is tmpfs then the file should be created by root else it
   is a file being created in the archive and creator should be archive.
   """
   filePath = os.path.join( path, fileName )

   numMB = fileSize / oneMB
   remBytes = fileSize % oneMB

   # Generate buffer to use for writes
   buff = " " * oneMB

   try:
      if srcDest:
         os.seteuid( ROOT_USER_ID )
         os.setegid( ROOT_GROUP_ID )
      else:
         os.setegid( ARCHIVE_GROUP_ID )
         os.seteuid( ARCHIVE_USER_ID )

      with open( filePath, 'w' ) as newFile:
         for _ in xrange( 0, numMB ):
            newFile.write( buff )
         if remBytes > 0:
            newFile.write( buff[ :remBytes ] )
         os.fsync( newFile )
         t0( "createFile: %s" % filePath )

   except (IOError, OSError) as err:
      if err.errno == errno.ENOSPC or err.errno == errno.EDQUOT:
         # There is nothing we can do if we're out of space here
         return

      print ( 'Writing file %s file failed srcDest: %r with error: %s'
              % ( filePath, srcDest, os.strerror( err.errno ) ) )
      raise Exception( os.strerror( err.errno ) )

   finally:
      if not srcDest:
         os.seteuid( ROOT_USER_ID )
         os.setegid( ROOT_GROUP_ID )

   return


def createSymLink( symLinkPath, filePath, fileName ):
   """
   Add a symbolic link in symLinkPath, point at the fileName located in filePath
   """
   srcFilePath = os.path.join( symLinkPath, fileName )
   destFilePath = os.path.join( filePath, fileName )
   absDestFilePath = os.path.abspath( destFilePath )

   try:
      # Symlinks are created in the tmpfs not the archive so root writes
      os.seteuid( ROOT_USER_ID )
      os.setegid( ROOT_GROUP_ID )
      os.symlink( absDestFilePath, srcFilePath )

   except (IOError, OSError) as err:
      if err.errno == errno.ENOSPC or err.errno == errno.EDQUOT:
         # There is nothing we can do if we're out of space here
         return

      print ( "Error creating symlink, src: %s dst: %s error: %s"
              % ( srcFilePath, absDestFilePath, os.strerror( err.errno ) ) )
      raise Exception( os.strerror( err.errno ) )

   return


def removeFileObject( path ):
   """
   Remove file / directory / symlink recursively if ncessary. Cleanup our test mess.
   """
   try:
      if os.path.islink( path ):
         t0( "Removing link:", path )
         os.unlink( path )
      elif os.path.isdir( path ):
         t0( "Removing dir:", path )
         shutil.rmtree( path )
      else:
         t0( "Removing file:", path )
         os.remove( path )

   except (IOError, OSError) as err:
      if err.errno != errno.ENOENT:
         print ( 'Failed to remove file %s with error: %s'
                 % ( path, os.strerror( err.errno ) ) )
         raise Exception( os.strerror( err.errno ) )

   return


def removeAnySsdTempFiles():
   """
   Remove any residual SSD files that start with "temp" that other product tests
   don't bother cleaning up.
   """
   path = defaultArchiveDevMountDir
   targetFiles = r"^temp*"

   for fileName in os.listdir( path ):
      matchFile = re.search( targetFiles, fileName )
      if matchFile:
         filePath = os.path.join( path, fileName )
         removeFileObject( filePath )

   return


def sendQuitSignal( pidNums ):
   """
   Send a SIGQUIT signal to each agent specified by their process id number.
   """
   for pidNum in pidNums:
      print "Pid:", pidNum, "Signal:", signal.SIGQUIT
      try:
         os.kill( int(pidNum), signal.SIGQUIT )
      except OSError:
         # Signals are delivered on a best effort basis
         continue

   return


def sortDirByMtime( dirPath, reverseOrder=False ):
   """
   Sort the specified directory by mtime and get either the oldest or newest
   created files in order.
   """
   dirFiles = os.listdir( dirPath )
   pathFiles = []

   for fileName in dirFiles:
      pathFiles.append( os.path.join( dirPath, fileName ) )

   fileStats = []
   for obj in pathFiles:
      fileStats.append( ( os.stat( obj ).st_mtime, os.path.basename ( obj ) ) )

   sortedFiles = []
   for _, fileName in sorted( fileStats, reverse=reverseOrder ):
      sortedFiles.append( fileName )

   return sortedFiles


def validateFileType( fileType ):
   """
   Check that the specified fileType is a supported file system.
   """
   validFileTypes = [ 'ssdMount',
                      'archivedArchives',
                      'agents',
                      'archivedAgents',
                      'cores',
                      'archivedCores',
                      'logs',
                      'archivedLogs',
                      'testfiles',
                      'archivedTestfiles',
                    ]

   if any( fileType in validType for validType in validFileTypes ):
      return True

   return False


def directoryForFileType( fileType ):
   """
   Return the default Directory for the given fileType so we can execute
   the requested action.
   """
   return {
      'ssdMount': defaultArchiveDevMountDir,
      'archivedArchives' : defaultArchiveDevMountDir,
      'agents' : defaultSrcAgentsDir,
      'archivedAgents' : defaultArchiveAgentsDir,
      'cores' : defaultSrcCoreDir,
      'archivedCores' : defaultArchiveCoreDir,
      'logs' : defaultSrcLogDir,
      'archivedLogs' : defaultArchiveLogDir,
      'testfiles' : defaultSrcTestFileDir,
      'archivedTestfiles' : defaultArchiveTestFileDir,
   }.get( fileType, None )


def returnNewestFiles( fileType ):
   """
   Return the file names in newest ctime order for the given file type.
   Invoked with the input parameter: --returnnewest 
   where the argument supplied with the parameter is the fileType
   """

   # If fileType was specified validate it so if it is not recognized we can
   # record that a bad value was supplied and could not be processed
   if validateFileType( fileType ) is False:
      print "%s Error in input, unrecognized filetype supplied: %s", \
            scriptName, fileType
      return None

   dirPath = directoryForFileType( fileType )

   if dirPath:
      return sortDirByMtime( dirPath, reverseOrder=True )

   return None


def returnOldestFiles( fileType ):
   """
   Return the file names in oldest ctime order for the given file type.
   Invoked with the input parameter: --returnoldest
   where the argument supplied with the parameter is the fileType
   """

   # If fileType was specified validate it so if it is not recognized we can
   # record that a bad value was supplied and could not be processed
   if validateFileType( fileType ) is False:
      print "%s Error in input, unrecognized filetype supplied: %s", \
            scriptName, fileType
      return None

   dirPath = directoryForFileType( fileType )

   if dirPath:
      return sortDirByMtime( dirPath )

   return None


def returnListForFileType( fileType ):
   """
   Return the list of file names for the given file type.
   Invoked with the input parameter: --returnlist
   where the argument supplied with the parameter is the fileType
   """

   # If fileType was specified validate it so if it is not recognized we can
   # record that a bad value was supplied and could not be processed
   if validateFileType( fileType ) is False:
      print "%s Error in input, unrecognized filetype supplied: %s", \
            scriptName, fileType
      return None

   dirPath = directoryForFileType( fileType )

   if dirPath:
      return os.listdir( dirPath )

   return None


def writeAgentFile( path, agentFileName ):
   """
   Write the specified agent file for the desired size.
   Can be called multiple times for the same file and writes will be appended.
   """
   agentFilePath = os.path.join( path, agentFileName )
   wrbuff = " " * defaultFileSize
   with open( agentFilePath, 'a' ) as afile:
      afile.write( wrbuff )
      os.fsync( afile )

   return


def waitForAgentFileArchived( filePath ):
   """
   Wait until the archived agent file is written
   """
   while not os.path.exists( filePath ):
      time.sleep( 1 )

   return


def waitForAgentFilesArchived( count ):
   """
   Wait until the archived agent files are written
   """
   for ctr in xrange( 1, count + 1 ):
      agentFileName = "testingAgents" + str( ctr )
      path = os.path.join( defaultArchiveAgentsDir, agentFileName )
      waitForAgentFileArchived( path )

   return


def writeAgentFiles( count ):
   """
   Write the specified number of agent files to the src agent directory.
   """
   for ctr in xrange( 1, count + 1 ):
      agentFileName = "testingAgents" + str( ctr )
      writeAgentFile( defaultSrcAgentsDir, agentFileName )

   return


def deleteAgentFiles( count ):
   """
   Delete the agent files from both directory locations.
   """
   for ctr in xrange( 1, count + 1 ):
      agentFileName = "testingAgents" + str( ctr )
      agentSrcFilePath = os.path.join( defaultSrcAgentsDir, agentFileName )
      try:
         os.remove( agentSrcFilePath )
      except OSError:
         pass

      agentArchiveFilePath = os.path.join( defaultArchiveAgentsDir, agentFileName )
      try:
         os.remove( agentArchiveFilePath )
      except OSError:
         pass

   return


def statAgentFile( path, agentFileName, fileSize ):
   """
   Check the specified file to ensure it exists and is the correct size.
   """
   agentFilePath = os.path.join( path, agentFileName )
   fileSize = fileSize
   try:
      fileStatSize = os.stat( agentFilePath ).st_size
      if fileStatSize != fileSize:
         print "Error: File %s is %d bytes, expected %d bytes" % \
               ( agentFilePath, fileStatSize, fileSize )
   except OSError:
      print "Error: Unable to stat file %s" % agentFilePath


def statAgentFiles( count, fileSize ):
   """
   Verify the agent files exist in both directories and are the correct size.
   """
   for ctr in xrange( 1, count + 1 ):
      agentFileName = "testingAgents" + str( ctr )
      statAgentFile( defaultSrcAgentsDir, agentFileName, fileSize )
      statAgentFile( defaultArchiveAgentsDir, agentFileName, fileSize )
   return


def createAndTestAgentFiles( numFiles, fileSize ):
   """
   Create Test Agent Files and add a 2nd and 3rd write append to them as well.
   After each write and append, verify 90 seconds later that the files have
   been archived to the EOS SSD file archive.

   Once both the write and append have completed, remove the agents files
   both from the src and archive directories to clean things up.
   """
   testName = inspect.getframeinfo(inspect.currentframe())[2]
   if numFiles is None:
      numFiles = defaultCreateAgentFiles
   if fileSize is None:
      fileSize = defaultFileSize
   print "Number of agent files to create:", numFiles, "file size:", fileSize
   try:
      writeAgentFiles( numFiles )
      print "RUN: Test Agent Files written"
      waitForAgentFilesArchived( numFiles )
      statAgentFiles( numFiles, fileSize )
      print "RUN: Test Agent Files verified"

   except TypeError, err:
      print "EXCEPTION: in %s" % testName
      print traceback.print_exc()
      raise ReRaiseException( str( err ) ), None, sys.exc_info()[2]

   finally:
      deleteAgentFiles( numFiles )
      cmdList = [ 'sync' ]
      subprocess.call( cmdList, shell=False )
      print "RUN: Agent Files removed"

   return


def cleanupTestAgentFiles( numFiles ):
   """
   Ensure we delete and agent files that may have been created by
   createAndTestAgentFiles()
   """
   testName = inspect.getframeinfo(inspect.currentframe())[2]
   if numFiles is None:
      numFiles = defaultCreateAgentFiles
   try:
      deleteAgentFiles( numFiles )

   except TypeError, err:
      print "EXCEPTION: in %s" % testName
      print traceback.print_exc()
      raise ReRaiseException( str( err ) ), None, sys.exc_info()[2]

   return


def getMostRecentAgentsFiles( path, agentList ):
   """
   Get the most recent agent filenames for any agents specified in the agentList
   in the specified agent directory path. To get the current agent file, there may
   be more than one if the agent was restarted, we must sort the files in most
   recent to least recent order.
   """
   dirFiles = []

   for dirPath, _, fileNames in os.walk( path ):
      for fileName in fileNames:
         dirFiles.append( os.path.join( dirPath, fileName ) )

   fileStats = []

   for fnm in dirFiles:
      if os.path.isfile( fnm ):
         fileStats.append( ( os.stat( fnm ).st_mtime, os.path.basename ( fnm ) ) )

   sortedFiles = []

   for _, fileName in sorted( fileStats, reverse=True ):
      sortedFiles.append( fileName )

   matchList = [ fileName + '*' for fileName in agentList ]
   matchedFiles = []
   for matchName in matchList:
      for fname in sortedFiles:
         if fnmatch.fnmatch( fname, matchName ):
            matchedFiles.append( fname )
            break

   return matchedFiles


def getPidNumFromAgentFileName( agentFiles ):
   """
   Given agent file names that of the form agentName-pidNum we want to extract
   and return the pid number for each agentFile in the list supplied to us.
   """
   pidNumList = []

   for agentFile in agentFiles:
      pidNumList.append( agentFile.split( "-" )[1] )

   return pidNumList


def createAgentCoreFiles( path, agentNames ):
   """
   For a path and one or more agentnames (agentName may be a comma separated list
   of agentnames, so we split the names into a list) get the pid numbers of the
   agent agentNames we're passed. To get the current process, we must get the
   most recent invocation of the agent. Using the pid number for the most recent
   agent log file we can send a sginal to cause the agent to create a core file(s)
   and restart the agent which will generate a new agent file. This creates core
   files that need to be archived.
   """
   if path is None:
      path = defaultSrcAgentsDir

   fileList = []
   fileList = agentNames.split( "," )
   agentFiles = getMostRecentAgentsFiles( path, fileList )
   pidNumbers = getPidNumFromAgentFileName( agentFiles )
   sendQuitSignal( pidNumbers[ 0 ] )

   return


def copytree( src, dst ):
   """
   Do a recursive file / directory / symlink copy but do not retain any metadata
   stats as we use this function to create test copies of files and directories
   that we want to have updated files times.
   """
   names = os.listdir( src )
   if not os.path.exists( dst ):
      os.makedirs( dst )
   errors = []

   for name in names:
      srcname = os.path.join( src, name )
      dstname = os.path.join( dst, name )
      try:
         if os.path.islink( srcname ):
            linkto = os.readlink( srcname )
            os.symlink( linkto, dstname )
         elif os.path.isdir( srcname ):
            copytree( srcname, dstname )
         else:
            shutil.copy( srcname, dstname )
      # catch the Error from the recursive copytree, so we can continue
      except OSError as err:
         errstr = "OSError: src %s dst %s err %s" % \
                  ( srcname, dstname, os.strerror( err ) )
         errors.append( errstr )
      except ValueError as err:
         errstr = "ValueError: src %s dst %s args %s" % \
                  ( src, dst, err.args[0] )
         errors.append( errstr )
      except IOError as (err, strerror):
         estr = "IOError src %s dst %s error({0}): {1}".format(err, strerror) \
               % ( src, dst )
         errors.append( estr )

   if errors:
      raise Exception( errors )

   return


def copyDirAndContents( dirPath, oldDirName, newDirName ):
   """
   Create a new new directory and copy the contents from the old directory to
   the new directory. Both directories have the same path.
   """
   if not os.path.exists( dirPath ):
      os.makedirs( dirPath )

   try:
      # Don't replicate file stats, this is for test and we want different times
      src = os.path.join( dirPath, oldDirName )
      dest = os.path.join( dirPath, newDirName )
      os.makedirs( dest )
      os.chown( dest, ARCHIVE_USER_ID, ARCHIVE_GROUP_ID )

      os.setegid( ARCHIVE_GROUP_ID )
      os.seteuid( ARCHIVE_USER_ID )
      copytree( src, dest )
   except OSError as err:
      # Any error encountered creating directory or writing testfile
      print 'Directory not copied. Error: %s' % err
   finally:
      os.seteuid( ROOT_USER_ID )
      os.setegid( ROOT_GROUP_ID )

   return


def copyAndGenerateArchivedDir( path, numFiles ):
   """
   Generate a new archived dir file using existing archived archive dir/file.
   Put the new archived dir in the same path as the existing dir but give it an
   approproiate new name.
   """
   for _ in xrange( 1, int( numFiles + 1 ) ):
      newDirPath = os.path.dirname( path )
      datestr = datetime.datetime.now().strftime( "%Y-%m-%d-%H:%M:%S" )
      oldDirName = os.path.basename( path )
      newDirName = "var_archive." + datestr + ".dir"
      copyDirAndContents( newDirPath, oldDirName, newDirName )
      time.sleep( 1 )

   return


def genArchivedArchiveFromExistingDir( path, numFiles ):
   """
   Generate one or more new archived archive directories using an existing
   archived archive file as a template to copy from.
   We default to using the most recent archived archive to use as our existing
   archived archive dir/file. If no exsting archived archive exists we fail.

   To create one or more new archived archives input the following parameters:
   --archived --copy --number
   where --number is the number of copies to create
   --filesize can be specified to set the desired file size, default is 1GB
   --path can be used to specify an alternate path to place the archived archives
   """
   ssdMountDir = defaultArchiveDevMountDir
   if path:
      ssdMountDir = path

   if numFiles is None:
      numFiles = 1

   dirs = []
   for fileName in os.listdir( ssdMountDir ):
      if fileName.startswith( archivedArchiveFileName ):
         filePath = os.path.join( ssdMountDir, fileName )
         dirs.append( (os.stat( filePath ).st_mtime, fileName ) )

   sortedDirs = []
   for _, fileName in sorted( dirs, reverse=True ):
      sortedDirs.append( fileName )

   if sortedDirs:
      fileName = sortedDirs[ 0 ]
      archivedDir = os.path.join( ssdMountDir, fileName )
      copyAndGenerateArchivedDir( archivedDir, numFiles )
      return None

   print "No existing archived archive so no copies made"

   return None


def createArchivedArchivesTestDir( path, fileSize, numFiles ):
   """
   Generate a new archived archive dir with a specified space usage.
   Put the new archived archive dir in the SSD mount point directroy but with an
   approproiate new name.

   To create one or more new archived archives input the following parameters:
   --archived --create --number
   where --number is the number of copies to create
   --filesize can be specified to set the desired file size, default is 1GB
   --path can be used to specify an alternate path to place the archived archives
   """
   ssdMountDir = defaultArchiveDevMountDir
   if path:
      ssdMountDir = path

   if numFiles is None:
      numFiles = 1

   if fileSize is None:
      fileSize = oneGB

   # Create the new archived archive directory
   try:
      datestr = datetime.datetime.now().strftime( "%Y-%m-%d-%H:%M:%S" )
      newDirName = "var_archive." + datestr + ".dir"
      newDirPath = os.path.join( ssdMountDir, newDirName )
      os.makedirs( newDirPath )
      os.chown( newDirPath, ARCHIVE_USER_ID, ARCHIVE_GROUP_ID )

      os.setegid( ARCHIVE_GROUP_ID )
      os.seteuid( ARCHIVE_USER_ID )

      # Add file to new archived archive directory content
      createFile( newDirPath, "testfile1", fileSize )
      time.sleep( 1 )

      # make more archived archives of the same if requested to do so
      numFiles -= 1
      if numFiles > 0:
         os.seteuid( ROOT_USER_ID )
         os.setegid( ROOT_GROUP_ID )
         copyAndGenerateArchivedDir( newDirPath, numFiles )

   finally:
      os.seteuid( ROOT_USER_ID )
      os.setegid( ROOT_GROUP_ID )

   return


def createTestFiles( path, fileSize, numFiles, srcDest=False ):
   """
   Create the specified number of test files of the specified size in the 
   designated path.
   """
   if not os.path.exists( path ):
      os.makedirs( path )

   for fileNum in xrange( 1, int( numFiles + 1 ) ):
      fileName = 'testfile' + str( fileNum )
      createFile( path=path, fileName=fileName, fileSize=fileSize, srcDest=srcDest )
      print "path: %s filename: %s filesize: %d srcDest: %r" \
            % ( path, fileName, fileSize, srcDest )

   return


def genArchiveTestFiles( path, fileSize, numFiles ):
   """
   Generate one or more archive test files and place the files in the designated
   path with the designated fileSize. We only generate archive side files for
   this test, designate src test files if those must be created as well.
   Note: the default 1GB size for the archive side files is too large for src
         test files since tmpfs is very limited.
   """
   if path is None:
      path = defaultArchiveTestFileDir

   if fileSize is None:
      fileSize = oneGB

   if numFiles is None:
      numFiles = 1

   createTestFiles( path, fileSize, numFiles )

   return


def genSrcTestFiles( path, fileSize, numFiles ):
   """
   Generate one or more src test files and place the files in the designated
   path with the designated fileSize.
   """
   if path is None:
      path = defaultSrcTestFileDir

   if fileSize is None:
      fileSize = oneMB

   if numFiles is None:
      numFiles = 1

   createTestFiles( path, fileSize, numFiles, srcDest=True )

   return


def createLogFiles( path, logName, fileSize, numFiles, srcDest=False ):
   """
   Generate a log file for the specified logName of the given size and number of
   copies in the specified path.
   """
   if not os.path.exists( path ):
      os.makedirs( path )

   for _ in xrange( 1, int( numFiles + 1 ) ):
      dateTimeName = datetime.datetime.now().strftime( ".%Y-%m-%d--%H-%M-%S.tgz" )
      fileName = logName + dateTimeName
      createFile( path=path, fileName=fileName, fileSize=fileSize, srcDest=srcDest )
      time.sleep( 1 )

   return


def genSrcLogFiles( path, logNames, fileSize, numFiles ):
   """
   Generate one or more src logfiles using logName and place the files in the
   designated path with the designated fileSize. The logNames parameter may be
   a comma separated list of logNames if we want to create logs for multiple
   logNames for our test.
   Can be invoked with e.g.:
     --lognames ...,... --logfiles --create --filesize #K/M/G --number ##
   """
   if path is None:
      path = defaultSrcLogDir

   if logNames is None:
      logNames.append( 'testLog' )

   if fileSize is None:
      fileSize = oneMB

   if numFiles is None:
      numFiles = 1

   for logName in logNames:
      createLogFiles( path, logName, fileSize, numFiles, srcDest=True )

   return


def logFileCleanup( path, logName ):
   """
   Clean up a src log file for the given path and logName.
   """
   logFiles = os.listdir( path )

   for logFile in logFiles:
      if logFile.startswith( logName ):
         filePath = os.path.join( path, logFile )
         removeFileObject( filePath )

   return


def srcLogCleanup( path, logNames ):
   """
   Clean up any src log files for the given logName or logNames as the logNames
   parmeter may be a comma separated list of logfile names to remove.
   Can be invoked with:
     --lognames ...,... --logfiles --remove
   """
   if path is None:
      path = defaultSrcLogDir

   if logNames is None:
      logNames.append( 'testLog' )

   for logName in logNames:
      logFileCleanup( path, logName )

   return


def archiveLogCleanup( path, logNames ):
   """
   Clean up any archive log files for the given logName or logNames as the logNames
   parmeter may be a comma separated list of logfile names to remove.
   Can be invoked with e.g.:
     --lognames ...,... --archivefiles --remove
   """
   if path is None:
      path = defaultArchiveLogDir

   if logNames is None:
      logNames.append( 'testLog' )

   for logName in logNames:
      logFileCleanup( path, logName )

   return


def genArchiveLogFiles( path, logNames, fileSize, numFiles ):
   """
   Generate one or more archive logfiles using logName and place the files in the
   designated path with the designated fileSize. The logNames parameter may be
   a comma separated list of logNames if we want to create logs for multiple
   logNames for our test.
   Can be invoked with e.g.:
     --lognames ...,... --archivefiles --create --filesize #K/M/G --number ##
   """
   if path is None:
      path = defaultArchiveLogDir

   if logNames is None:
      logNames.append( 'testLog' )

   if fileSize is None:
      fileSize = oneMB

   if numFiles is None:
      numFiles = 1

   for logName in logNames:
      createLogFiles( path, logName, fileSize, numFiles )

   return


def createAgentFiles( path, agentName, fileSize, netns, numFiles, pidNum=None,
                      srcDest=False ):
   """
   Create agent process log files for the specified agentName and for the specified
   number of copies. To avoid conflicts with existing agent logs we typically invent
   our own agent name for these test files.
   """
   global startingAgentFileProcNum

   if fileSize is None:
      fileSize = defaultFileSize

   if numFiles is None:
      numFiles = numFiles

   if netns is None:
      incrValue = 1
   else:
      incrValue = 2

   pidNumList = []
   for copynum in xrange( 0, int(numFiles) ):
      if pidNum is None:
         pid = str( startingAgentFileProcNum )
      else:
         pid = str( pidNum[ copynum ] )
      fileName = agentName + '-' + pid
      createFile( path, fileName, fileSize, srcDest )
      pidNumList.append( pid )
      if pidNum is None:
         startingAgentFileProcNum += incrValue

   return pidNumList


def genTestCoreFile( path, pidNum, agentName, fileSize, netPid, netSize,
                     srcDest=False, symLink=None ):
   """
   Generate a core file in the specified path using the pidNum and agentName
   If a netPid is supplied then a netns core file with the specified pidNum is
   also created. Some of our agents generate both core and a netns core so this
   code will do the same. If a netns core file is created it will be sized based
   on the netSize specified. Both fileSize and netSize have default values, 16MB
   and 64KB respectively.

   These are dummy core files, not cores generated by signaling an agent.

   The symLink if present supplies a path to set a symbolic link that points to the
   core file beaing created. This is useful if we want to create the effect of
   a core file that has been archived and has a symlink from the src directory to
   the archived core file. Also create a symlink for netns file if one is present.
   """
   if fileSize is None:
      fileSize = 16 * oneMB

   netFileName = None

   if netPid:
      netFileName = "core." + str( netPid ) +  "." + str( int( time.time() ) ) \
                 + ".netns.gz"
      if netSize is None:
         netSize = 64 * oneKB
      createFile( path, netFileName, netSize, srcDest )

   fileName = "core." + str( pidNum ) + "." + str( int( time.time() ) ) + \
              "." + agentName + ".gz"

   createFile( path, fileName, fileSize, srcDest )

   # Create symlink to archived core file, optionally symlink to archived netns file
   # if specified to do so
   if symLink:
      createSymLink( symLinkPath=symLink, filePath=path, fileName=fileName )
      if netFileName:
         createSymLink( symLinkPath=symLink, filePath=path, fileName=netFileName )

   return


def genSrcAgentFiles( agentName, fileSize, core, netns, copies, pidList=None ):
   """
   Create copies number of agent files of the specified size for the specified agent.
   If pidList is supplied use the pid(s) supplied to create the src agent files.

   Src files generate only src directory files. Archive copies must be created by
   archival. For testing you can create agents files by specifying --create plus:
   --agentfiles # : generates src agent files
   --agentfiles # --corefiles : generates src agent and core files
   Also:
   --agent will name the agent to create files for
   --filesize specifies the size of the files for agent and optionally core files
   --netns specifies you want a netns core file that uses your agent file pid
   --pid can be used to specify the starting pidNum to use for files

   Creates src agent files with optional core agent and netns files if specified.
   These src files can then be archived if the archivetree.py script is running.
   """
   path = defaultSrcAgentsDir

   if pidList is None:
      pidList = createAgentFiles( path, agentName, fileSize, netns, copies,
                                  pidNum=None, srcDest=True )
   else:
      for pid in pidList:
         createAgentFiles( path, agentName, fileSize, netns, copies,
                           pidNum=pid, srcDest=True )

   if core:
      path = defaultSrcCoreDir

      for pid in pidList:
         if netns:
            pidNum = str( int( pid ) + 1 )
            netPid = pid
            netSize = None
         else:
            pidNum = str( pid )
            netPid = None
            netSize = defaultNetNsCoreSize

         genTestCoreFile( path, pidNum, agentName, fileSize, netPid, netSize,
                          srcDest=True )

   return



def genSrcAgentFilesByPct( pct, agentName, fileSize, core, netns ):
   """
   Create some number of agent files of the specified size for the specified agent.
   If pidList is supplied use the pid(s) supplied to create the src agent files.

   The pct value must be specified and is the percent increase in tmpfs usage
   requested, so a minimum of a 1% worth of agent files will be generated.

   Using percent allows for changes in device size with out having to further
   scale the request.

   Src files generate only src directory files. Archive copies must be created by
   archival. For testing you can create agents files by specifying --create plus:
   --agentfiles # : generates src agent files
   --agentfiles # --corefiles : generates src agent and core files
   Also:
   --agent will name the agent to create files for
   --filesize specifies the size of the files for agent and optionally core files
   --netns specifies you want a netns core file that uses your agent file pid
   --pid can be used to specify the starting pidNum to use for files

   Creates src agent files with optional core agent and netns files if specified.
   These src files can then be archived if the archivetree.py script is running.
   """
   path = defaultSrcAgentsDir

   # Get the size of the device and calc target growth based on pct value
   _, srcSizeKB, _, _, _, _ = fileSystemInfo( path )
   growKB = calcSizeFromPct( long( srcSizeKB ), pct )

   if fileSize is None:
      fileSizeMB = 16
      fileSize = 16 * oneMB
   else:
      fileSizeMB = ( fileSize + ( oneMB - 1 ) ) / oneMB

   numMB = fileSizeMB if core is None else 2 * fileSizeMB

   growMB = ( growKB + ( oneKB - 1 ) ) / oneKB

   numFiles = growMB / numMB

   t0( "growMB %d numMB %d numFiles %d" % ( growMB, numMB, numFiles ) )

   genSrcAgentFiles( agentName, fileSize, core, netns, numFiles )

   return


def genArchiveAgentFiles( pct, agentName, fileSize, core, netns, noSrc ):
   """
   Create some number of agent files of the specified size for the specified agent.
   Creates archived agent files with optional core agent & netns files if specified.

   The pct value must be specified and is the percent increase in SSD usage
   requested, so a minimum of a 1% worth of agent files will be generated. This
   is useful to generate files to allow space management to cull them as needed
   when over the quota. Using percent allows for changes in device size with out
   having to further scale the request.

   Archived files have src counterparts so we automatically generate both files
   unless for testing we are told not to produce these files, since we have very
   little src space to hold files and typically a lot more archive space.

   For testing to get archived agents files specify:
   --agentfiles # : where # is a pct between 1 and 100.
   --create this option must be specified to create files.
   --agentfiles --archivefiles -nosrc : generates just archived agents files
   --agentfiles --archivefiles : generates src and archived agents files
   --agentfiles --archivefiles --corefiles : above plus core & archived core files
   Also:
   --agent will name the agent to create files for
   --filesize specifies the size of the files for agent and optionally core files
   --netns specifies you want a netns core file that uses your agent file pid
   --pid can be used to specify the starting pidNum to use for files
   --nosrc is specified to generate just archive side files, though in that case
           we automatically generate /var/core symlinks to core files.
   """
   path = defaultArchiveAgentsDir

   # Get the size of the device and calc target growth based on pct value
   _, devSizeKB, _, _, _, _ = fileSystemInfo()
   growKB = calcSizeFromPct( long( devSizeKB ), pct )

   if fileSize is None:
      fileSizeMB = 16
      fileSize = 16 * oneMB
   else:
      fileSizeMB = ( fileSize + ( oneMB - 1 ) ) / oneMB

   numMB = fileSizeMB if core is None else 2 * fileSizeMB

   growMB = ( growKB + ( oneKB - 1 ) ) / oneKB

   numFiles = growMB / numMB

   t0( "growMB %d numMB %d numFiles %d" % ( growMB, numMB, numFiles ) )

   pidList = createAgentFiles( path, agentName, fileSize, netns, numFiles )
   if noSrc is False:
      genSrcAgentFiles( agentName=agentName, fileSize=fileSize, core=None,
                        netns=None, copies=numFiles, pidList=pidList )

   if core:
      path = defaultArchiveCoreDir
      symLink = defaultSrcCoreDir

      for pid in pidList:
         if netns:
            pidNum = str( int( pid ) + 1 )
            netPid = pid
            netSize = None
         else:
            pidNum = pid
            netPid = None
            netSize = defaultNetNsCoreSize

         genTestCoreFile( path, pidNum, agentName, fileSize, netPid, \
                          netSize, srcDest=False, symLink=symLink )

   return


def ssdDirectoryCleanup( path=None ):
   """
   Clean out the SSD mount point directory of everything but the active
   archive directory, *quota.* quota files and the lost+found.
   """
   excludeFiles = r"^(?!(archive|lost\+found|.?quota.*|.arista_archive))"

   if path is None:
      path = defaultArchiveDevMountDir

   if path == '/mnt/flash':
      for fileName in os.listdir( path ):
         if fileName.startswith( 'var_archive' ):
            filePath = os.path.join( path, fileName )
            removeFileObject( filePath )
   else:
      for fileName in os.listdir( path ):
         matchFile = re.search( excludeFiles, fileName )
         if matchFile:
            filePath = os.path.join( path, fileName )
            removeFileObject( filePath )


def ssdDirectoryCleanupSkipArchived( path=None ):
   """
   Clean out the SSD mount point directory of everything but the active
   archive directory, *quota.* quota files, lost+found and "var_archive.*"
   archived archive files.
   """
   excludeFiles = (
      r"^(?!(archive|lost\+found|.?quota.*|var_archive.*|.arista_archive))"
   )

   if path is None:
      path = defaultArchiveDevMountDir

   if path == '/mnt/flash':
      return

   for fileName in os.listdir( path ):
      matchFile = re.search( excludeFiles, fileName )
      if matchFile:
         filePath = os.path.join( path, fileName )
         removeFileObject( filePath )

   return


def archiveTestFileCleanup( path=None ):
   """
   Clean up any residual archive test files.
   The default archive test file directory may be overriden by specifying a path.
   """
   if path is None:
      path = defaultArchiveTestFileDir

   if not os.path.isdir( path ):
      return

   archiveTestFiles = os.listdir( path )

   for fileName in archiveTestFiles:
      filePath = os.path.join( path, fileName )
      removeFileObject( filePath )

   removeFileObject( path )

   return


def srcTestFileCleanup( path=None ):
   """
   Clean up any residual src test files.
   The default src test file directory may be overriden by specifying a path.
   """
   if path is None:
      path = defaultSrcTestFileDir

   if not os.path.isdir( path ):
      return

   srcTestFiles = os.listdir( path )

   for fileName in srcTestFiles:
      filePath = os.path.join( path, fileName )
      removeFileObject( filePath )

   removeFileObject( path )

   return


def archiveAgentCoreFilesCleanup():
   """
   Clean up any residual archive agent files.
   """
   archiveCoreDir = defaultArchiveCoreDir

   archiveCoreFiles = os.listdir( archiveCoreDir )

   for fileName in archiveCoreFiles:
      filePath = os.path.join( archiveCoreDir, fileName )
      removeFileObject( filePath )

   return


def srcAgentCoreFilesCleanup():
   """
   Clean up any residual src agent files.
   """
   srcCoreDir = defaultSrcCoreDir

   srcCoreFiles = os.listdir( srcCoreDir )

   for fileName in srcCoreFiles:
      filePath = os.path.join( srcCoreDir, fileName )
      removeFileObject( filePath )

   return


def removeCoreFile( path, agentName, pidNum, symLink=False ):
   """
   Caller supplies path to target directory (the /var/core or /archive/var/core or
   equivalent), the agentName and pidNum of the core file we want to remove.
   The pidNum may match a netns file for agents that are started by netns and thus
   have two processes associated with them (the netns proc and the agent proc). The
   agentName-#### agents log file with point at the netns file if the agent has two
   processes associated with it. See AID 719 for more details.
   """
   netFilesName = "core." + str( pidNum ) +  ".*" + ".netns.gz"
   noNetFilesName = "core." + str( pidNum ) + ".*." + agentName + ".gz"
   filesName = "core." + ".*..*." + agentName + ".gz"

   dirList = os.listdir( path )
   statsList = []
   for fileName in dirList:
      filePath = os.path.join( path, fileName )
      if os.path.isfile( filePath ) :
         statsList.append( ( os.stat( filePath ).st_mtime, fileName, filePath ) )

   sortedStatsList = []
   filesList = []
   for mdate, fileName, filePath in sorted( statsList, reverse=True ):
      sortedStatsList.append( ( mdate, fileName, filePath ) )
      filesList.append( os.path.basename( filePath ) )

   for _, fileName, filePath in sortedStatsList:
      matchNetFileName = re.search( netFilesName, fileName )
      if matchNetFileName:
         removeFileObject( filePath )
         if symLink:
            # Symlink is always in the t,pfs src path
            srcPath = os.path.join( defaultSrcCoreDir, fileName )
            removeFileObject( srcPath )
         for _, aFileName, aFilePath in sortedStatsList:
            matchFileName = re.search( filesName, aFileName )
            if matchFileName:
               removeFileObject( aFilePath )
               if symLink:
                  # Symlink is always in the t,pfs src path
                  srcPath = os.path.join( defaultSrcCoreDir, aFileName )
                  removeFileObject( srcPath )
               break
      else:
         matchNoNetFileName = re.search( noNetFilesName, fileName )
         if matchNoNetFileName:
            removeFileObject( filePath )
            if symLink:
               # Symlink is always in the t,pfs src path
               srcPath = os.path.join( defaultSrcCoreDir, fileName )
               removeFileObject( srcPath )
            break

   return


def agentsFilesNotOpen( path=None ):
   """
   Generate a list of the src agents directory files that are not open.
   This can be used to determine which agents files we can remove.
   """
   agentsDir = defaultSrcAgentsDir
   if path:
      agentsDir = path

   dirFiles = os.listdir( agentsDir )

   cmdList = [ 'lsof', '+D', agentsDir ]
   process = Popen( cmdList, shell=False, stdout=PIPE )

   out, _ = process.communicate()
   outStr = ''.join( out )
   outLines = outStr.split( '\n' )

   openFiles = set([])
   iterLines = iter( outLines )
   next( iterLines )
   for line in iterLines:
      words = line.split()
      if len( words ) == 9:
         fileName = os.path.basename( words[ 8 ] )
         openFiles.add( fileName )

   notOpenFiles = set( dirFiles ) - openFiles

   return notOpenFiles


def removeAgentFile( path, agentFileName ):
   """
   Caller supplies path to target directory (the /var/log or /archive/var/log/agents
   or equivalent), the agentName and pidNum which forms the agent-pid# of the
   agent log file that we want to remove.
   """
   filePath = os.path.join( path, agentFileName )
   removeFileObject( filePath )

   return


def archiveAgentFileCleanup( agentName=None, core=None ):
   """
   Clean up any residual archive agent files.
   Only cleans up archived agent filess and optiotnally their related core files.
   To cleanup src and archived files for a given agent(s) use the src (logfiles)
   option. To initiate a archive agent file cleanup use the following options:
   --agentfiles 0 --archivefiles --remove : removes archived agent files
   --agentfiles 0 --archivesfiles --remove --corefile : above plus related corefiles
   Also: --agent will have the name of the agent to remove files for
         --pid can be used to specify the starting pidNum to use for files
         You don't need to specify netns, the system checks for them automatically
   Note: ommitting --agent generates the default that all agent files are removed
         and if --corefiles is specified all archived agent files and their related
         core files are removed.
   """
   archiveAgentDir = defaultArchiveAgentsDir
   archiveCoreDir = defaultArchiveCoreDir

   agentFilesList = os.listdir( archiveAgentDir )

   for fileName in agentFilesList:
      agent, pid = fileName.rsplit( '-', 1 )
      if agent == agentName:
         removeAgentFile( archiveAgentDir, fileName )

         if core:
            name = agentName
            if name is None:
               name = "*"

            removeCoreFile( path=archiveCoreDir, agentName=name, pidNum=pid,
                            symLink=True )

   return


def srcAgentFileCleanup( agentName=None, core=None ):
   """
   Clean up any residual src agent files and related core files.
   Also, since src files are archived we clean up archived agent and core files too.
   To initiate a src agent file cleanup use the following options:
   --agentfiles --logfiles --remove : removes src agent files
   --agentfiles --logfiles --remove --corefile : the above plus related corfiles
   Also: --agent will have the name of the agent to remove files for
         --pid can be used to specify the starting pidNum to use for files
         You don't need to specify netns, the system checks for them automatically
   Note: ommitting --agent generates the default that all agent files are removed
         and if --corefiles is specified all src agent files and their related
         core files are removed.
   """
   srcAgentDir = defaultSrcAgentsDir
   archiveAgentDir = defaultArchiveAgentsDir
   srcCoreDir = defaultSrcCoreDir
   archiveCoreDir = defaultArchiveCoreDir

   agentFilesList = agentsFilesNotOpen( srcAgentDir )

   statsList = []
   for fileName in agentFilesList:
      filePath = os.path.join( srcAgentDir, fileName )
      if os.path.isfile( filePath ) :
         statsList.append( ( os.stat( filePath ).st_mtime, filePath ) )

   # For our testing, always process the most recent files first
   filesList = []
   for _, filePath in sorted( statsList, reverse=True ):
      filesList.append( os.path.basename( filePath ) )

   for agentFile in filesList:
      if agentName is None or agentFile.startswith( agentName ):
         pid = agentFile.split( "-" )[1]
         removeAgentFile( srcAgentDir, agentFile )
         removeAgentFile( archiveAgentDir, agentFile )

         if core:
            name = agentName
            if name is None:
               name = "*"

            removeCoreFile( path=srcCoreDir, agentName=name, pidNum=pid )
            removeCoreFile( path=archiveCoreDir, agentName=name, pidNum=pid )

   return


def cleanUpFull():
   """
   Do a full clean up of both src and archive files that were generated for
   testing. This will give us a clean system to start with or to end with
   for testing.

   To invoke an SSD archive cleanup, just supply as input the following option:
   --fullclean

   Note: takes the default parameters, for more specific cleanups use the
         appropriate cleanup with any parameters that may be needed.
   """
   srcTestFileCleanup()
   archiveTestFileCleanup()
   srcAgentFileCleanup()
   archiveAgentFileCleanup()
   srcAgentCoreFilesCleanup()
   archiveAgentCoreFilesCleanup()
   removeAnySsdTempFiles()
   ssdDirectoryCleanup()

   return


def cleanUpForPtest():
   """
   Clean up the SSD archive at ptest start. We really just want to remove
   any SSD mount directory files that we don't need and make sure that there
   aren't any residual temp files from other ptests that often accumulate.
   --ptestclean
   """
   ssdDirectoryCleanup()
   removeAnySsdTempFiles()

   return


def getFileSize( fileSize ):
   """
   Get the file size as number, supplied as string and may suffix size.
   Valid suffixes include: K, M and G for 1024, 1048576 and 1073741824
   So, for example, --filesize 8K is equivalent to --filesize 8192 and
   --filesize 2M is equivalent to --filesize 2097152 and finally
   --filesize 4G is equivalent to --filesize 4294967296
   Either lowercase or uppercase valid: k,K or m,M or g,G
   """
   numBytes = 0

   if isinstance( fileSize, ( int, long ) ):
      return fileSize

   if isinstance(fileSize, basestring):
      if fileSize.isdigit():
         numBytes = int( fileSize )
      elif fileSize.endswith( 'K' ) or fileSize.endswith( 'k' ):
         numBytes = oneKB * int( fileSize[ :-1 ] )
      elif fileSize.endswith( 'M' ) or fileSize.endswith( 'm' ):
         numBytes = oneMB * int( fileSize[ :-1 ]  )
      elif fileSize.endswith( 'G' ) or fileSize.endswith( 'g' ):
         numBytes = oneGB * int( fileSize[ :-1 ]  )
      else:
         raise Exception( "Invalid fileSize parameter, value: %s" % fileSize )
   else:
      raise Exception( "Invalid fileSize parameter, value: %s" % fileSize )

   return numBytes


def compareVarLogMtimes( change=False, duration=100 ):
   """
   Compare the times on the tmpfs and archived copy of the /var/log/messages
   file. If archival was enabled then the times for the archive version
   should change about once every 60+ seconds. This because the tmpfs version
   should be updated about once every 60+ seconds and so the archived version
   should also be updated shortly after that. If file archival is disabled
   then only the tmpfs version should be updated, the archival version should
   remain unchanged.

   Note: Duration should be at least 100 seconds to give archive a chance to
         run before the second check occurs.
   """

   assert duration >= 100

   def checkMtimes():
      tmpfsMtime1, archiveMtime1 = getVarLogMtimes()

      time.sleep( duration )

      tmpfsMtime2, archiveMtime2 = getVarLogMtimes()

      # Ensure the tmpfs messages file was updated as expected
      t1( "Expect mtime2 %d >= mtime1 %d" % ( tmpfsMtime2, tmpfsMtime1 ) )
      assert tmpfsMtime2 >= tmpfsMtime1

      if not change:
         # Ensure the archived messages file was unchanged
         t1( "Expect mtime2 %d == mtime1 %d" % ( archiveMtime2, archiveMtime1 ) )
         return archiveMtime2 == archiveMtime1
      else:
         # Ensure the archived messages file was updated
         t1( "mtime2 %d mtime1 >= %d" % ( archiveMtime2, archiveMtime1 ) )
         return archiveMtime2 >= archiveMtime1

   if change:
      desc = 'mtime of archived copy of /var/log/messages to change'
   else:
      desc = 'mtime of archived copy of /var/log/messages to become stable'

   Tac.waitFor( checkMtimes, description=desc, sleep=True )


def checkArchiveExists():
   """
   Check to ensure the Archive storage has been properly defined.
   This check needed for after we redefine Archive after removing it.
   The archive recreate after Cli removal happens asynchronously after
   space for a new archive is reserved by Cli. So we need a check.
   If the Archive is not present we assert as this is not expected.
   """
   assert os.path.isdir( defaultArchiveDir ) is True, "No Archive Dir"
   assert os.path.isdir( defaultArchiveLogDir ) is True, "No Archive Log"
   assert os.path.isdir( defaultArchiveCoreDir ) is True, "No Archive Core"
   assert os.path.isdir( defaultArchiveAgentsDir ) is True, "No Archive agents"
   assert os.path.islink( defaultTmpfsArchiveDir ) is True, "No Archive link"

   return


def fileCtime( filepath ):
   """
   Return the file ctime, returned as epoch in seconds.
   """
   return os.path.getctime( filepath )


def checkArchiveRemoved( previousArchiveEpochStr ):
   """
   Check to ensure the Archive storage has been completely removed.
   This would be because the user explicitly removed it with the Cli.

   The archive if it exists must be newer than the original archive directory
   which if it existed before has its epoch passed in. The newer version will
   have a greater epoch seconds if the archive was recreated. Archive dir would
   be recreated after a clear or if explicitly deleted and then created.
   """
   if os.path.exists( defaultArchiveDir ) and previousArchiveEpochStr is not None:
      currentArchiveEpoch = fileCtime( defaultArchiveDir )
      previousArchiveEpoch = int( previousArchiveEpochStr )
      t1( "Previous ctime %d" % previousArchiveEpoch )
      t1( "Current ctime %d" % currentArchiveEpoch )
      assert currentArchiveEpoch > previousArchiveEpoch, "Archive not newer"

   return


def getVarLogMtimes():
   """
   Returns the modification time for two files the tmpfs /var/log/messages
   file and the archived version of the same file. If the archived version
   is not present a value of zero should be returned for it. The tmpfs
   version of the file is expected to always be present, if it isn't then
   Linux has problems.
   """
   tmpfsMsgs = os.path.join( defaultSrcLogDir, "messages" )
   archiveMsgs = os.path.join( defaultArchiveLogDir, "messages" )

   tmpfsMtime = os.stat( tmpfsMsgs ).st_mtime
   if os.path.isfile( archiveMsgs ):
      archiveMtime = os.stat( archiveMsgs ).st_mtime
   else:
      archiveMtime = 0

   return tmpfsMtime, archiveMtime


def setupArchiveTestLoad( numAgents=None, agentPct=None, numLogs=None,
                          numLogFiles=None ):
   """
   Generate an archive test load, the number of agents the percentage of
   resource consumed by agent files and the number of logs and log files per
   log may be specified to size the load desired, though reasobale defaults
   are specified so sizing the load is optional.
   """
   agentName = "testAgent"
   logName = "testLog"
   fileSize = 16 * oneMB

   if numAgents is None:
      numAgents = 3

   if agentPct is None:
      agentPct = 1

   if numLogs is None:
      numLogs = 3

   if numLogFiles is None:
      numLogFiles = 48

   for num in range( 1, numAgents+1 ):
      agent = agentName + str( num )
      netns = True if num % 2 == 1 else False
      genArchiveAgentFiles( pct=agentPct, agentName=agent, fileSize=fileSize,
                            core=True, netns=netns, noSrc=True )

   logs = []

   for num in range( 1, numLogs+1 ):
      logs.append( logName + str( num ) )

   genArchiveLogFiles( path=defaultArchiveLogDir, logNames=logs,
                       fileSize=fileSize, numFiles=numLogFiles )

   return


def createFullSrcTestLoad():
   """
   Create a full src side test load that occupies about 75% of tmpfs dirs to
   give file archival a load to archive.
   """
   # Let's try and allocate about 75% of the src log (/var/log) as our load
   pct = 75
   _, sizeKB, usedKB, _, _, _ = fileSystemInfo( defaultSrcLogDir )
   growKB = calcSizeFromPct( long( sizeKB ), pct )
   neededKB = growKB - long( usedKB )

   # fileSizeKB in kilobytes this is 16 MB, fileSize is in bytes
   fileSizeKB = 16 * oneKB
   fileSize = fileSizeKB * oneKB

   numFiles = ( neededKB + ( fileSizeKB - 1 ) ) / fileSizeKB

   # Setting path to None results in default being used, what we want
   genSrcTestFiles( path=None, fileSize=fileSize, numFiles=numFiles )

   _, sizeKB, usedKB, _, _, _ = fileSystemInfo( defaultSrcCoreDir )
   growKB = calcSizeFromPct( long( sizeKB ), pct )
   neededKB = growKB - long( usedKB )

   numFiles = ( neededKB + ( fileSizeKB - 1 ) ) / fileSizeKB

   # Generate core files
   agentName = "testingAgent"
   path = defaultSrcCoreDir
   netSize = None
   netPid = None
   pidNumStart = 100000 * runNum

   for num in range( 1, numFiles ):
      pidNum = pidNumStart + num
      genTestCoreFile( path, pidNum, agentName, fileSize, netPid,
                       netSize, srcDest=True )

   return


def cleanupFullSrcTestLoad():
   """
   Cleanup the full src side test load created by createFullSrcTestLoad
   and make sure to cleanup the files from the archive store as well.
   """
   # The src test file cleanup can handle the src side
   srcTestFileCleanup()
   # The archive test file cleanup takes care of the archive side
   archiveTestFileCleanup()

   # Find the core files that we added and remove them to clean up test
   for fileName in os.listdir( defaultSrcCoreDir ):
      names = fileName.split( '.' )
      if len( names ) == 5 and names[ -2 ] == "testingAgent":
         path = os.path.join( defaultSrcCoreDir, fileName )
         removeFileObject( path )
         # Don't forget to remove archived core file if it exists
         path = os.path.join( defaultArchiveCoreDir, fileName )
         if os.path.isfile( path ):
            removeFileObject( path )

   return


def cleanupArchiveTestLoad( numAgents=None, numLogs=None ):
   """
   The cleanup function for setupArchiveTestLoad which should be run
   after testing to cleanup any residual test files.
   """
   agentName = "testAgent"
   logName = "testLog"

   if numAgents is None:
      numAgents = 3

   if numLogs is None:
      numLogs = 3

   for num in range( 1, numAgents+1 ):
      agent = agentName + str( num )
      archiveAgentFileCleanup( agentName=agent, core=True )

   logs = []

   for num in range( 1, numLogs+1 ):
      logs.append( logName + str( num ) )

   archiveLogCleanup( path=defaultArchiveLogDir, logNames=logs )

   return


def setupSrcTestLoad( numAgents=None, agentPct=None, numLogs=None,
                      numLogFiles=None ):
   """
   The src side test load generating function that is equivalent to
   setupArchiveTestLoad but on the src side instead of the archive side.
   See setupArchiveTestLoad for more details.
   """
   agentName = "testSrcAgent"
   logName = "testsrcLog"
   fileSize = 2 * oneMB

   if numAgents is None:
      numAgents = 2

   if agentPct is None:
      agentPct = 25

   if numLogs is None:
      numLogs = 3

   if numLogFiles is None:
      numLogFiles = 24

   for num in range( 1, numAgents+1 ):
      agent = agentName + str( num )
      netns = True if num % 2 == 1 else False
      genSrcAgentFilesByPct( pct=agentPct, agentName=agent, fileSize=fileSize,
                             core=True, netns=netns )

   logs = []

   for num in range( 1, numLogs+1 ):
      logs.append( logName + str( num ) )

   genSrcLogFiles( path=defaultSrcLogDir, logNames=logs,
                   fileSize=oneMB, numFiles=numLogFiles )

   return


def cleanupSrcTestLoad( numAgents=None, numLogs=None ):
   """
   The cleanup function for setupSrcTestLoad which should be run
   after testing to cleanup any residual test files.
   """
   agentName = "testSrcAgent"
   logName = "testSrcLog"

   if numAgents is None:
      numAgents = 2

   if numLogs is None:
      numLogs = 3

   for num in range( 1, numAgents+1 ):
      agent = agentName + str( num )
      srcAgentFileCleanup( agentName=agent, core=True )

   logs = []

   for num in range( 1, numLogs+1 ):
      logs.append( logName + str( num ) )

   srcLogCleanup( path=defaultSrcLogDir, logNames=logs )

   return


def main( agent, agentFiles, archived, archiveFiles, checkDisabled, \
          checkEnabled, checkExists, checkRemoved, copy, core, coreFiles, create, \
          eosagenttest, eosagentcleanup, fileSize, fileSystemStats, fullClean, \
          logFiles, logNames, netns, noSrc, numFiles, path, pid, ptestClean, \
          remove, returnList, returnNewest, returnOldest, testFiles, testLoad, \
          testSrcFileLoad, flashTest ) :
   """
   Our main routine determines which test functions the user has requested to
   determine which test files they want to created and the parameters they want
   to use when they create those files, or if they want to remove files they
   created for cleanup, or if they just want some information back about the
   system thay may be useful to their test.
   """
   global startingAgentFileProcNum
   global ARCHIVE_GROUP_ID
   global ARCHIVE_USER_ID
   global defaultArchiveDevMountDir

   if flashTest:
      defaultArchiveDevMountDir = "/mnt/flash"
   else:
      defaultArchiveDevMountDir = "/mnt/drive"

   if pid:
      startingAgentFileProcNum = int( pid )

   # If the caller requested the fileSystem Information we return that here
   # Typically the default path is used but the caller may override if needed
   if fileSystemStats:
      print fileSystemInfo( path )
      return

   # If specified asks that we return the contents of a directory specified by
   # fileType either in newest to oldest ctime, oldest to newest ctime or just
   # any order. Our code takes care of pointing the request correct directory
   # path so the caller does not need to know the details where files reside.
   if returnNewest:
      print returnNewestFiles( fileType=returnNewest )
      return
   elif returnOldest:
      print returnOldestFiles( fileType=returnOldest )
      return
   elif returnList:
      print returnListForFileType( fileType=returnList )
      return

   # Check either disabled or enabled can be requested in addition to other
   # other options is supported. Asserts if expected state is not found.
   if checkDisabled:
      compareVarLogMtimes( change=False, duration=checkDisabled )
   elif checkEnabled:
      compareVarLogMtimes( change=True, duration=checkEnabled )

   # Check exists ensures that the Archive was correctly defined and if so
   # other options may also need to be checked and processing continues.
   #
   # Check removed to see if Archive Storage has been removed, is so we're
   # done because there isn't much to do if the archive no longer exists.
   #
   # Both methods assert if expected state is not found
   if checkExists:
      checkArchiveExists()
   elif checkRemoved:
      checkArchiveRemoved( int( checkRemoved ) )
      return

   #
   # Check if requested to do a full clean up which cleans up all possible archive
   # files except the active archive, active log files, quota files and lost+found
   # or if the simpler cleanup for ptest which gets rid of any extra files in the
   # SSD mount directory that may have accumulated and that shouldn't be there at
   # the start or end of a ptest.
   #
   if fullClean:
      cleanUpFull()
   elif ptestClean:
      cleanUpForPtest()

   if fileSize:
      fileSize = getFileSize( fileSize )

   ARCHIVE_USER_ID = getpwnam( ARCHIVE_ID_USER_NAME ).pw_uid
   ARCHIVE_GROUP_ID = grp.getgrnam( ARCHIVE_ID_GROUP_NAME ).gr_gid

   # If caller specified an EOS Agent File Test they want a synchronous
   # test that verifies two rounds of agent file writes and that the src
   # agent test files are copied to the EOS file archive. Once done the
   # test will clean it self up.
   if eosagenttest:
      createAndTestAgentFiles( numFiles=numFiles, fileSize=fileSize )
      return
   elif eosagentcleanup:
      cleanupTestAgentFiles( numFiles=numFiles )
      return

   # Create or Remove a src side test file load that fills up /var/log to
   # 75% of it's capacity to give us a migration load to archive.
   # A torture test for SSD file archival when coupled with agent restarts.
   if testSrcFileLoad:
      if create:
         createFullSrcTestLoad()
      elif remove:
         cleanupFullSrcTestLoad()
      else:
         print "%s: --testsrcfileload requires --create or --remove parameter" % \
               scriptName

   # TestLoad specifies a test load is generated to exercise the archive and
   # space management facilities.
   if testLoad:
      if noSrc:
         if create:
            setupArchiveTestLoad()
         elif remove:
            cleanupArchiveTestLoad()
         else:
            print "%s: --testload requires --create or --remove parameter" % \
                  scriptName
      else:
         if create:
            setupSrcTestLoad()
         elif remove:
            cleanupSrcTestLoad()
         else:
            print "%s: --testload requires --create or --remove parameter" % \
                  scriptName
      return

   # Testfiles are files we add in the tesdir in either the src or archive path
   # or both if so desired depending on what we want to test. They are useful
   # simply for increasing the amount of space we use and to add files as a test
   # load.
   if testFiles:
      if logFiles:
         if create:
            genSrcTestFiles( path=path, fileSize=fileSize, numFiles=numFiles )
         if remove:
            srcTestFileCleanup( path=path )
      if archiveFiles:
         if create:
            genArchiveTestFiles( path=path, fileSize=fileSize, numFiles=numFiles )
         if remove:
            archiveTestFileCleanup( path=path )

   # Generate additional archived archive files either from an existing archive
   # file (the most recent one if present) which is realistic in content (copy)
   # or just create our own fake archived archive which means we can size it
   # exactly to fit our needs
   if archived:
      if copy:
         genArchivedArchiveFromExistingDir( path=path, numFiles=numFiles )
      if create:
         createArchivedArchivesTestDir( path=path, fileSize=fileSize,
                                        numFiles=numFiles )
      if remove:
         ssdDirectoryCleanup( path=path )

   # For agentFiles an agent name is passed and used to create either src or
   # archive agent files depending on the src setting, agent files can also
   # automatically create assoicated core files, netns option has agent log pid
   # point at netns and agent's pid is a higher number after the netns pid.
   # If core is specified with agentFiles option, the agentFile has assoicated
   # core files (or files if netns file is present) and must also be cleaned up.
   #
   # The agentFiles parameter takes a number between 0 and 100 that is a pct.
   # For creating files the agentFiles pct must be between 1 and 100.
   # For removing files the agentFiles pct must be 0.
   if agentFiles is not None and 0 <= agentFiles <= 100:
      if logFiles:
         if create and 0 < agentFiles <= 100:
            genSrcAgentFilesByPct( pct=agentFiles, agentName=agent,
                                   fileSize=fileSize, core=coreFiles, netns=netns )
         if remove and agentFiles == 0:
            srcAgentFileCleanup( agentName=agent, core=coreFiles )
      elif archiveFiles:
         if create and 0 < agentFiles <= 100:
            genArchiveAgentFiles( pct=agentFiles, agentName=agent,
                                  fileSize=fileSize, core=coreFiles,
                                  netns=netns, noSrc=noSrc )
         if remove and agentFiles == 0:
            archiveAgentFileCleanup( agentName=agent, core=coreFiles )

   # Create logfiles for the given logNames which may be a comma seperated list
   # of one more logNames to use to create the log files. Valid parameters include
   # fileSize to use and numFiles to create for each logName.
   # 
   if logNames:
      logNamesList = []
      if ',' in logNames:
         logNamesList = logNames.split( ',' )
      else:
         logNamesList.append( str( logNames ) )

      if logFiles:
         if create:
            genSrcLogFiles( path=path, logNames=logNamesList, fileSize=fileSize,
                            numFiles=numFiles )
         if remove:
            srcLogCleanup( path=path, logNames=logNamesList )
      elif archiveFiles:
         if create:
            genArchiveLogFiles( path=path, logNames=logNamesList, fileSize=fileSize,
                                numFiles=numFiles )
         if remove:
            archiveLogCleanup( path=path, logNames=logNamesList )

   # This option generates real core files from a real agent for our testing
   # It will create (per agent) a new src agent files and one or two src core files
   # either a core file named after the agent and pid of the agent that died or a
   # core file named netns with the same pid as the agent that died and a second
   # core file with the agent name that died and a pid number that is one or more
   # greater than the netns core file pid since the agent is started just after
   # the netns.
   if core:
      if create:
         createAgentCoreFiles( path=path, agentNames=agent )
      if remove:
         srcAgentCoreFilesCleanup()

   return


if __name__ == '__main__':
   parser = argparse.ArgumentParser( prog=sys.argv[0] )

   group1 = parser.add_mutually_exclusive_group( required=False )
   group2 = parser.add_mutually_exclusive_group( required=False )
   group3 = parser.add_mutually_exclusive_group( required=False )

   parser.add_argument( '--agent',
                        help='agent name to use',
                        default=None )

   parser.add_argument( '--agentfiles',
                        help='Create/Remove agents directory files',
                        type=int, default=None, choices=xrange( 1, 128 ) )

   parser.add_argument( '--archivefiles',
                        help='Create/Remove files in archive directories',
                        action="store_true" )

   parser.add_argument( '--archived',
                        help='Create/Remove archived archive files',
                        action="store_true" )

   group1.add_argument( '--checkdisabled',
                        help='Check that EOS File Archival is disabled',
                        type=int, default=None )

   group1.add_argument( '--checkenabled',
                        help='Check that EOS File Archival is enabled',
                        type=int, default=None )

   group2.add_argument( '--checkexists',
                        help='Check that EOS File Archival is present',
                        action="store_true" )

   group2.add_argument( '--checkremoved',
                        help='Check that EOS File Archival was removed',
                        type=int, default=None )

   parser.add_argument( '--copy',
                        help='Copy the most recent archived archive number times',
                        action="store_true" )

   parser.add_argument( '--core',
                        help='Create/Remove real core files, use agent(s) specified',
                        action="store_true" )

   parser.add_argument( '--corefiles',
                        help='Create/Remove core files using agent(s) specified',
                        action="store_true" )

   group3.add_argument( '-c', '--create',
                        help='Create for actions specified by other parms',
                        action="store_true" )

   parser.add_argument( '--eosagenttest',
                        help='EOS Agent files create, append, verify and delete',
                        action="store_true" )

   parser.add_argument( '--eosagentcleanup',
                        help='EOS Agent files delete',
                        action="store_true" )

   parser.add_argument( '-f', '--filesize',
                        help='size of any files that need to be created',
                        default=defaultFileSize )

   parser.add_argument( '--filesystemstats',
                        help='return in formation about the file system',
                        action="store_true" )

   parser.add_argument( '--fullclean',
                        help='Clean everything but active archive and quota files',
                        action="store_true" )

   parser.add_argument( '--logfiles',
                       help='Create/Remove the specified log files',
                       action="store_true" )

   parser.add_argument( '--lognames',
                        help='Create/Remove logfile(s) using this logname',
                        default=None )

   parser.add_argument( '--netns',
                        help='generate namespace files along with agent files',
                        action="store_true" )

   parser.add_argument( '--nosrc',
                        help='do not generate src files when archivefiles requested',
                        action="store_true" )

   parser.add_argument( '-n', '--number',
                        help='number of files or directories to create',
                        type=int, default=defaultNumFiles )

   parser.add_argument( '--path',
                        help='use specified path(s), may be a comma separated list',
                        default=None )

   parser.add_argument( '--pid',
                        help='starting pid number to use for creating files',
                        default=None )

   parser.add_argument( '--ptestclean',
                        help='Clean up files for ptest to run',
                        action="store_true" )

   group3.add_argument( '-r', '--remove',
                        help='Remove/cleanup for actions specified by other parms',
                        action="store_true" )

   parser.add_argument( '--returnlist',
                        help='Return the newest file of the specified file type',
                        default=None )

   parser.add_argument( '--returnnewest',
                        help='Return the newest file of the specified file type',
                        default=None )

   parser.add_argument( '--returnoldest',
                        help='Return the oldest file of the specified file type',
                        default=None )

   parser.add_argument( '--run',
                        help='Return the oldest file of the specified file type',
                        type=int, default=1 )

   parser.add_argument( '-t', '--testfiles',
                        help='Create/Remove testfiles',
                        action="store_true" )

   parser.add_argument( '--testload',
                        help='Create/Remove testload',
                        action="store_true" )

   parser.add_argument( '--testsrcfileload',
                        help='Create/Remove test src file load',
                        action="store_true" )

   parser.add_argument( '--flashTest',
                        help='The archive is on the flash',
                        action="store_true" )

   args = parser.parse_args()

   if args.run > 1:
      runNum = args.run

   main( args.agent, args.agentfiles, args.archived, args.archivefiles,
         args.checkdisabled, args.checkenabled, args.checkexists,
         args.checkremoved, args.copy, args.core, args.corefiles,
         args.create, args.eosagenttest, args.eosagentcleanup, args.filesize,
         args.filesystemstats, args.fullclean, args.logfiles,
         args.lognames, args.netns, args.nosrc, args.number, args.path,
         args.pid, args.ptestclean, args.remove, args.returnlist,
         args.returnnewest, args.returnoldest, args.testfiles, args.testload,
         args.testsrcfileload, args.flashTest )
