#!/bin/bash

#
# Script to be called before logrotate starts processing files under
# /var/log/agents and /var/core. 
#

agent_logs_dir="/var/log/agents"
core_files_dir="/var/core"

# When we have bunch of empty log files for an agent(s) (repeated agent crash due to, 
# say, lack of space), logrotate can spend lot of time processing these empty files.
# In such scenario, we can delete all empty logs of an agent except the 
# one that is opened by agent. This cannot be done with a section in logrotate.conf 
delete_empty_agent_logs()
{
    # for each agent (which has empty logs)
    for f in `find $agent_logs_dir -size 0 -regex "$agent_logs_dir/[A-Z].*\-[0-9]+$" | rev | cut -d '-' -f2- | rev | sort | uniq`
    do
       an=$(basename $f)
       # delete all empty files but log of latest file (maybe opened by agent)
       fcount=`find $agent_logs_dir -size 0 -regex "$agent_logs_dir/${an}\-[0-9]+$" | wc -l`
       if [ $fcount -gt 1 ]
       then
           # Keep most recent file
           find $agent_logs_dir -size 0 -regex "$agent_logs_dir/${an}\-[0-9]+.*" |\
                         xargs ls -1t | tail -n +2 | xargs rm
       fi
    done
}

delete_empty_cores()
{
    # Delete empty core files that are older than a min
    find $core_files_dir -size 0 -mmin +1 -exec rm {} \;
}

# When logrotate fails to rotate due to lack of space, it leaves empty or partially
# copied logs around. When logrotate starts next time, even though space is available
# it fails to rotate as it sees a file with rotated (copy) name already exists.
delete_rotate_failed_logs()
{
    find $agent_logs_dir -regex "$agent_logs_dir/[A-Z].*\-[0-9]+\.[0-9]+$" -exec rm {} \;
}

# Limit the total number of TerminAttr logs to keep from filling up disk with old logs.
# Treat logs which have been rotated and compressed as a single logfile and remove all
# rotated logs along with the original when they age out.
delete_old_TerminAttr_logs()
{
   num_recent_logs=100
   if [ ! -z "$NUM_RECENT_TERMINATTR_LOGS" ]; then
      if [ "$NUM_RECENT_TERMINATTR_LOGS" -eq "0" ]; then
         # TerminAttr logfile deletion disabled
         return
      fi
      num_recent_logs="$NUM_RECENT_TERMINATTR_LOGS"
   fi

   # skip any log that has a file extension. These are most likely rotated logs which
   # will be removed when the base log they are generated goes away.
   if ! logs="$(ls -tr1 "$agent_logs_dir"/TerminAttr-* 2> /dev/null | grep -v "\.")" ; then
         return
   fi

   recent_logs="$(printf '%s\n' "$logs" | tail -n "$num_recent_logs")"

   old_logs="${logs%$recent_logs}"
   for log in $old_logs; do
      # remove any rotated logfiles as well
      rm -f "${log}" "${log}".*
   done
}

delete_rotate_failed_logs
delete_empty_agent_logs
delete_empty_cores
delete_old_TerminAttr_logs

exit 0
