#!/usr/bin/env python
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""
The original dhclient-script configures the interfaces directly.
In EOS, this is taken care of by Ira. This script populates the
IntfStatus TAC model to which Ira reacts to and configures the interface
appropriately.
dhclient invokes dhclient-script from time to time. dhclient defines
a set of environment variables while invoking the script and sets the
'reason' as to why the script has been invoked.
"""

import sys
import os
import Arnet
import Arnet.NsLib
import DhcpClientLogMsgs
import Logging
import PyClient
import Tac

from IpUtils import Mask
from SysdbHelperUtils import SysdbPathHelper
from DhcpClientConsts import DHCP_DEFAULT_ROUTE_PREFERENCE

UPDATE_REASONS = [ 'BOUND', 'RENEW', 'REBIND', 'TIMEOUT', 'REBOOT' ]

def getAddrWithMask( env ):
   return Tac.Value( 'Arnet::IpAddrWithMask',
                     env[ 'new_ip_address' ],
                     Mask( env[ 'new_subnet_mask' ] ).maskLen )

def getRouteKey( prefix, preference ):
   return Tac.Value( "Routing::RouteKey",
                     prefix=Arnet.IpGenPrefix( prefix ),
                     preference=preference )

def getVia( hop, intfId ):
   return Tac.Value( "Routing::Via",
                     hop=Arnet.IpGenAddr( hop ),
                     intfId=intfId )

def logEvents( reason, intf, env ):
   ip = env.get( 'new_ip_address' )
   leaseTime = env.get( 'new_dhcp_lease_time' )

   if reason == 'FAIL':
      # pylint: disable-msg=E1101
      Logging.log( DhcpClientLogMsgs.DHCP_ADDRESS_ASSIGNMENT_FAIL, intf )
   elif reason == 'EXPIRE':
      # pylint: disable-msg=E1101
      Logging.log( DhcpClientLogMsgs.DHCP_ADDRESS_EXPIRE,
                   env.get( 'old_ip_address' ),
                   Mask( env.get( 'old_subnet_mask' ) ).maskLen,
                   intf )
   elif reason in UPDATE_REASONS:
      # pylint: disable-msg=E1101
      Logging.log( DhcpClientLogMsgs.DHCP_ADDRESS_ASSIGNMENT_SUCCESS,
                   intf,
                   ip,
                   Mask( env.get( 'new_subnet_mask' ) ).maskLen,
                   leaseTime )
   elif reason == 'RELEASE':
      # pylint: disable-msg=E1101
      Logging.log( DhcpClientLogMsgs.DHCP_ADDRESS_RELEASE,
                   env.get( 'old_ip_address' ),
                   Mask( env.get( 'old_subnet_mask' ) ).maskLen,
                   intf )

# Add or delete per vrf the default route obtained from the option 'routers'
# during dhcp negiotation by the dhcp client. Current implementation only
# installs one IP address route with an admin distance of 254.
# If there are multiple IP address routes, only the highest preference
# IP address route is used to install the default route (RFC-2132),
# other IP address routes are discarded.
def addOrDeleteDefaultRoute( dhclientIntfStatus, rtInputdhcpRouter, env, add ):
   if not env.has_key( 'new_routers' ):
      return

   nexthops = env[ 'new_routers' ]
   prefix = '0.0.0.0/0'
   key = getRouteKey( prefix, DHCP_DEFAULT_ROUTE_PREFERENCE )
   if not nexthops:
      return

   nexthop = nexthops.split( ' ' )[ 0 ]

   via = getVia( nexthop, '' )
   if add:
      r1 = rtInputdhcpRouter.newRoute( key, 'forward' )
      r1.via[ via ] = 0
      dhclientIntfStatus.dhcpRouters = nexthop
   else:
      try:
         r1 = rtInputdhcpRouter.get( key )
      except PyClient.RpcError:
         r1 = rtInputdhcpRouter.newRoute( key, 'forward' )
      if r1 and via in r1.via:
         dhclientIntfStatus.dhcpRouters = Tac.Value( "Arnet::IpAddr" )
         del r1.via[ via ]
         if not r1.via:
            del rtInputdhcpRouter.route[
                  getRouteKey( prefix, DHCP_DEFAULT_ROUTE_PREFERENCE ) ]

def updateStatus( dhclientIntfStatus, rtInputdhcpRouter, env, routeEnable ):
   leaseTime = float( env[ 'new_dhcp_lease_time' ] )
   expiryTime = float( env[ 'new_expiry' ] )

   dhclientIntfStatus.addrWithMask = getAddrWithMask( env )
   dhclientIntfStatus.pending = False
   dhclientIntfStatus.leaseTime = leaseTime
   # RFC2131 states that T1 = renewalTime = now + 0.5 * leaseTime
   # T2 = rebindingTime = now + 0.875 * leaseTime
   # Changed to not need the current time
   dhclientIntfStatus.rebindingTime = round( expiryTime - leaseTime * 0.125 )
   dhclientIntfStatus.renewalTime = round( expiryTime - leaseTime * 0.5 )
   dhclientIntfStatus.expiryTime = expiryTime
   addOrDeleteDefaultRoute( dhclientIntfStatus, rtInputdhcpRouter,
                           env, routeEnable )

def cleanupStatus( dhclientIntfStatus, rtInputdhcpRouter, env ):
   dhclientIntfStatus.addrWithMask = Tac.Value( "Arnet::IpAddrWithMask" )
   dhclientIntfStatus.leaseTime = 0
   dhclientIntfStatus.rebindingTime = 0
   dhclientIntfStatus.renewalTime = 0
   dhclientIntfStatus.expiryTime = 0
   addOrDeleteDefaultRoute( dhclientIntfStatus, rtInputdhcpRouter, env, False )

def run( env, dhclientObj ): # pylint: disable-msg=W0621
   dhclientStatus = dhclientObj[ 'dhclientStatus' ]
   rtInputdhcpRouter = dhclientObj[ 'dhcpRouter' ]
   ipConfig = dhclientObj[ 'ipConfig' ]
   reason = env[ 'reason' ] if 'reason' in env else None
   # STOP means dhclient is going to exit. Exit gracefully
   # without doing anything.
   # NOTE: We can't invoke the default dhclient-script when the
   # reason is STOP since it will unconfigure the interfaces
   if reason == 'STOP':
      sys.exit( 0 )
   devName = env[ 'interface' ]
   intfName = env[ devName ]

   dhclientIntfStatus = None

   routeEnable = False
   if intfName in ipConfig.ipIntfConfig:
      if ipConfig.ipIntfConfig[ intfName ].defaultRouteSource == 'dhcp':
         routeEnable = True

   if reason in UPDATE_REASONS:
      dhclientIntfStatus = dhclientStatus.newIntfStatus( intfName )
      updateStatus( dhclientIntfStatus, rtInputdhcpRouter, env, routeEnable )
   elif reason == 'EXPIRE':
      dhclientIntfStatus = dhclientStatus.newIntfStatus( intfName )
      cleanupStatus( dhclientIntfStatus, rtInputdhcpRouter, env )

   logEvents( reason, intfName, env )

   # Push local entity changes
   Tac.runActivities( 0 )

def mountDhclientObjects( env ):
   dhclientObj = {}
   vrfName = env[ 'VRFNAME' ]

   pathHelper = SysdbPathHelper( env[ 'SYSNAME' ] )
   dhclientObj[ 'dhclientStatus' ] = pathHelper.getEntity( 'ip/dhclient/status' )
   vrfDhcpRouterPath = 'routing/vrf/input/' + vrfName + '/dhcpRouter'
   dhclientObj[ 'dhcpRouter' ] = pathHelper.getEntity( vrfDhcpRouterPath )
   dhclientObj[ 'ipConfig' ] = pathHelper.getEntity( 'ip/config' )
   return dhclientObj

setns = Tac.newInstance( 'Arnet::Netns', 'dummy' )
# Move to the default network namespace so that
# we can PyClient into the provider agent(s) running in
# the default network namespace only
setns.ns = Arnet.NsLib.DEFAULT_NS
environ = os.environ
run( environ, mountDhclientObjects( environ ) )
