#!/usr/bin/env python
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import sys
import Tac
from datetime import datetime

# Don't invoke the dependency generator here since SmashMount was an old module
# which doesn't exist in the current code base.
import importlib
SmashMount = importlib.import_module( 'SmashMount' )

# Force reload the latest AsuPatchBase from swi
# pylint: disable-msg=import-error
# pylint: disable-msg=wrong-import-position
sys.modules.pop( "AsuPatchBase", None )
import AsuPatchBase

# pylint: disable-msg=no-member, E1101

class AsuPatchArp( AsuPatchBase.AsuPatchBase ):

   def check( self ):
      return 0

   def reboot( self ):
      try:
         asuPatchArp( self.em, asuArpFileName, self )
      except IOError as e:
         self.log( 'asuPatchArp() failed with err=%s' % e )
         return 1
      except ( StopIteration, RuntimeError ):
         return 1
      except KeyboardInterrupt:
         self.log( 'User requested abort during smash iteration' )
         return 1
      else:
         return 0

   def cleanup( self ):
      try:
         os.remove( asuArpFileName )
      except OSError:
         pass

# pylint: enable-msg=no-member, E1101

def execute( stageVal, *args, **kwargs ):
   obj = AsuPatchArp( 'AsuPatchArp' )
   return obj.execute( stageVal, *args, **kwargs )

# To test the rest of this code on a user server during development run:
# '/src/AsuTest/stest/AsuPatchArpTest.py' or 'a4 make -p AsuTest stests'.
# There's also a patternrule for this file.

asuArpFileName = '/mnt/flash/arpAsuDb'
arpDbFileHeader = '''#
# This file is generated by AsuPatchArp
# It was last written at: %s
#
Version 2.0
'''

def isMgmtIntf( intfName ):
   intfId = Tac.Value( "Arnet::IntfId", intfName )
   return Tac.Type( "Arnet::MgmtIntfId" ).isMgmtIntfId( intfId )

def asuPatchArp( entityManager, fileName, obj=None ):
   smg = SmashMount.mountGroup( entityManager )
   arpSmash = smg.doMount( 'arp/status',
                           'Arp::Table::Status',
                           SmashMount.mountInfo( 'reader' ) )

   def writeSmashTable( table ):
      for smashKey, smashEntry in table.iteritems():
         if isMgmtIntf( smashKey.intfId ):
            continue
         line = '%s %s %s\n' % ( smashKey.intfId, smashKey.addr, smashEntry.ethAddr )
         arpDbFile.write( line )

   with open( fileName, 'w' ) as arpDbFile:
      hdr = arpDbFileHeader % ( datetime.now() )
      arpDbFile.write( hdr )
      if obj:
         obj.log( 'Iterating over %d arp entries and %d neighbor entries' %
                  ( len( arpSmash.arpEntry ), len( arpSmash.neighborEntry ) ) )
      try:
         writeSmashTable( arpSmash.arpEntry )
         writeSmashTable( arpSmash.neighborEntry )
      except StopIteration as e:
         obj.log( 'Smash iteration (%d arp entries, %d neighbor entries) '
                  'failed with err=%s.' % ( len( arpSmash.arpEntry ),
                  len( arpSmash.neighborEntry ), e ) )
         raise
      except RuntimeError as e:
         obj.log( 'Smash iteration (%d arp entries, %d neighbor entries) '
                  'encountered err=%s.' % ( len( arpSmash.arpEntry ),
                  len( arpSmash.neighborEntry ), e ) )
         raise
